import * as React from 'react';

import { ICar } from '../../../../../types';
import { CarInfoHandler } from '../../../../models/car';
import { Button } from '../../../../ui/Button';
import { Window } from '../../../../ui/FullModal';
import * as coreStyles from '../../../../ui/index.css';
import TextArea from '../../../../ui/TextArea';
import { Request2 } from '../../../../utils/request';
import { parseNumber } from '../../../CarNumber';
import GroupAttachModal from '../../../CarsFilters/GroupAttachModal';
import GroupModal from '../../../CarsFilters/GroupModal';
import Spin from '../../../Spin';
import TagModal from '../../../TagModal';
import { CARS_PACKAGE_REQUESTS, REQUESTS } from '../request';
import * as styles from './index.css';

interface ICarNumberArbitraryListProps {
    onClose: () => void;
}

interface ICarNumberArbitraryListState {
    error: Error | null;
    isLoading: boolean;
    carNumberInput: string;
    cars: ICar[];
    showGroupModal: boolean;
    showConflictModal: boolean;
    carNumbers: string[];
    callback: () => void;
    showTagModal: boolean;
    attachData: any;
    isMainModalVisible: boolean;
    filteredCars: ICar[];

    [x: number]: any;
}

const MODAL = {
    conflict: 'showConflictModal',
    tag: 'showTagModal',
    group: 'showGroupModal',
    groupAttach: 'attachData',
    main: 'isMainModalVisible',
};

export class CarNumberArbitraryList extends
    React.Component<ICarNumberArbitraryListProps, ICarNumberArbitraryListState> {
    state: ICarNumberArbitraryListState = {
        error: null,
        isLoading: false,
        carNumberInput: '',
        cars: [],
        filteredCars: [],
        showGroupModal: false,
        showConflictModal: false,
        carNumbers: [],
        callback: () => undefined,
        showTagModal: false,
        attachData: null,
        isMainModalVisible: true,
    };

    request = new Request2({ requestConfigs: CARS_PACKAGE_REQUESTS });

    getData() {
        this.setState({
            isLoading: true,
        }, async () => {
            try {
                const response = await this.request.exec(REQUESTS.GET_CARS);

                this.setState({
                    isLoading: false,
                    cars: response?.cars ?? [],
                });
            } catch (error) {
                this.setState({
                    isLoading: false,
                    error,
                });
            }
        });
    }

    formatCarNumber(carNumber): string {
        return parseNumber(carNumber.toLowerCase())?.number ?? '';
    }

    matchCars(callback) {
        const { cars, carNumberInput } = this.state;
        const MATCH_WORDS_REGEX = /[a-zа-я\d]+/ig;
        let carNumbers = carNumberInput?.match(MATCH_WORDS_REGEX)
            ?.map((carNumber) => this.formatCarNumber(carNumber)) ?? [];
        const filteredCars: ICar[] = [];

        cars.forEach((car) => {
            const currentCarNumber = this.formatCarNumber(CarInfoHandler.getNumber.call(car));

            if (carNumbers?.includes(currentCarNumber)) {
                filteredCars.push(car);
            }

            carNumbers = carNumbers.filter((carNumber) => carNumber !== currentCarNumber);
        });

        if (carNumbers.length) {
            this.setState({
                carNumbers,
                filteredCars,
                showConflictModal: true,
                callback,
            });
        } else {
            this.setState({
                filteredCars,
            }, () => {
                callback();
            });
        }
    }

    triggerModal(key, value) {
        const DONT_CLOSE_MODAL = [MODAL.conflict, MODAL.tag, MODAL.groupAttach];
        this.setState({
            [key]: value,
            isMainModalVisible: key === MODAL.conflict,
            showConflictModal: key === MODAL.conflict ? value : false,
        });

        if (!DONT_CLOSE_MODAL.includes(key) && !value) {
            this.props.onClose();
        }
    }

    attachmentHandler(data, isProposition) {
        this.setState((prev) => ({
            attachData: {
                ...data,
                isProposition,
                cars: prev.filteredCars.map((car) => CarInfoHandler.getId.call(car)),
            },
        }));
        this.triggerModal(MODAL.tag, false);
    }

    onChangeTextArea(carNumberInput) {
        this.setState({
            carNumberInput,
        });
    }

    componentDidMount() {
        this.getData();
    }

    componentWillUnmount() {
        this.request.abort();
    }

    render() {
        const {
            error,
            isLoading,
            carNumberInput,
            carNumbers,
            showConflictModal,
            showGroupModal,
            cars,
            callback,
            showTagModal,
            attachData,
            isMainModalVisible,
            filteredCars,
        } = this.state;
        const { onClose } = this.props;

        return <>
            {isMainModalVisible && <Window title={'Управление тегами по списку ГРЗ'}
                                           onClose={onClose.bind(this)}
                                           error={error}>
                <div className={styles.loading}>
                    Загрузка списка машин {isLoading ? <Spin size={'s'}/> : error ? '❌' : '✅'}
                </div>
                <TextArea placeholder={'Введите ГРЗ'}
                          value={carNumberInput}
                          onChange={this.onChangeTextArea.bind(this)}/>
                <div className={coreStyles.button_container}>
                    <Button disabled={!cars.length || !carNumberInput?.length}
                            onClick={this.matchCars.bind(this, this.triggerModal.bind(this, MODAL.group, true))}>
                        Пакетная обработка
                    </Button>
                    <Button disabled={!cars.length || !carNumberInput?.length}
                            onClick={this.matchCars.bind(this, this.triggerModal.bind(this, MODAL.tag, true))}>
                        Навесить тег
                    </Button>
                </div>
            </Window>}
            {showGroupModal && <GroupModal isWorking={false}
                                           onClose={this.triggerModal.bind(this, MODAL.group, false)}
                                           cars={filteredCars}/>}
            {attachData && <GroupAttachModal onClose={this.triggerModal.bind(this, MODAL.groupAttach, null)}
                                             openOnFinish
                                             data={[attachData]}/>}
            {showTagModal && <TagModal onClose={this.triggerModal.bind(this, MODAL.tag, false)}
                                       attachmentHandler={this.attachmentHandler.bind(this)}
                                       queueLength={filteredCars.length}/>}
            {showConflictModal && <Window title={'Не найдены совпадения по ГРЗ'}
                                          onClose={this.triggerModal.bind(this, MODAL.conflict, false)}>
                <div>Для этих номеров не были найдены совпадения:</div>
                <ul>
                    {carNumbers.map((carNumbers, index) => <li key={index}>{carNumbers}</li>)}
                </ul>
                <div className={coreStyles.button_container}>
                    <Button onClick={this.triggerModal.bind(this, MODAL.conflict, false)}>Вернуться</Button>
                    <Button onClick={callback}
                            disabled={!filteredCars.length}>Продолжить</Button>
                </div>
            </Window>}
        </>;
    }
}
