import * as React from 'react';

import { Dict, ICar } from '../../../../types';
import { EMPTY_DATA } from '../../../constants';
import { Button, ButtonTypes } from '../../../ui/Button';
import { Window } from '../../../ui/FullModal';
import { isValidJSONString } from '../../../utils/isValidJSONString';
import { readTable } from '../../../utils/readTable';
import { Request2 } from '../../../utils/request';
import { buttonLocationDetails, buttonNameDetails } from '../../../utils/sendLogs/eventTypes/buttonDetails';
import { download, makeLatCarNumber } from '../../../utils/utils';
import GroupAttachModal from '../../CarsFilters/GroupAttachModal';
import { SimpleError } from '../../SimpleError';
import TagModal from '../../TagModal';
import CarNumberArbitraryList from './CarNumberArbitraryList';
import style from './index.css';
import { CARS_PACKAGE_REQUESTS, REQUESTS } from './request';

interface ICarsPackageProps {
}

interface ICarsPackageState {
    isCarListLoading: boolean;
    error: Error | null;
    parsingError: Error | null;
    cars: ICar[];
    warning: string;
    tagModalIsOpen: boolean;
    attachData: any;
    closeOnFinish: boolean;
    carNumberArbitraryList: boolean;

    [x: number]: boolean;
}

const HEADER_LENGTH = 1;
const ID_KEYS = ['id', 'car_id'];
const TAG_KEYS = ['tag', 'tag_name'];
const COMMENT_KEY = ['comment'];

export default class CarsPackage extends React.Component<ICarsPackageProps, ICarsPackageState> {
    state: ICarsPackageState = {
        isCarListLoading: false,
        error: null,
        parsingError: null,
        cars: [],
        warning: '',
        tagModalIsOpen: false,
        attachData: null,
        closeOnFinish: false,
        carNumberArbitraryList: false,
    };
    request = new Request2({ requestConfigs: CARS_PACKAGE_REQUESTS });

    onExportClick() {
        this.setState({ isCarListLoading: true, error: null }, () => {
            this.request.exec(REQUESTS.GET_SAAS_CARS_DETAILS)
                .then(response => {
                    let csv = 'number;number_lat;status;ctc;imei;vin;id;lessor;model;tags;fuel_level\n';
                    let carsDetails = response;

                    if (typeof carsDetails === 'string') {
                        if (isValidJSONString(carsDetails)) {
                            carsDetails = JSON.parse(carsDetails);
                        } else {
                            this.setState({
                                isCarListLoading: false,
                                parsingError: new Error('Не удалось загрузить данные. Попробуйте еще раз.'),
                            });

                            return;
                        }
                    }

                    carsDetails?.cars?.forEach((car: any) => {
                        const tags = car?.tags?.map((tagInfo: any) => tagInfo.tag).join(' ');
                        csv += `${car.number || EMPTY_DATA};${car.number ? makeLatCarNumber(car.number) : EMPTY_DATA};${car.status || EMPTY_DATA};${car.registration_id || EMPTY_DATA};${car.imei || EMPTY_DATA};${car.vin || EMPTY_DATA};${car.id || EMPTY_DATA};${car.lessor || EMPTY_DATA};${car.model_id || EMPTY_DATA};${tags || EMPTY_DATA};${car.telematics && car.telematics.fuel_level || EMPTY_DATA}\n`;
                    });
                    download(csv, 'cars.csv', 'text/csv');
                    this.setState({ isCarListLoading: false });
                })
                .catch((error) => {
                    this.setState({ error, isCarListLoading: false });
                });
        });
    }

    onGroupAttachClick(result) {
        const idKey = result?.[0]?.findIndex((cell) => ID_KEYS.includes(cell.toLowerCase())) ?? -1;

        if (idKey !== -1 && result?.length > HEADER_LENGTH) {
            const cars = result.reduce((acc: string[], row: string[]) => {
                if (!ID_KEYS.includes(row[idKey]) && row[idKey]) {
                    acc.push(row[idKey]);
                }

                return acc;
            }, []);

            this.setState({
                cars,
                closeOnFinish: false,
            });

            this.triggerModal('tagModalIsOpen');
        } else {
            this.setState({
                warning: 'Кажется, в файле нет данных (возможно нет колонки id (car_id))',
            });
        }
    }

    onTagsAttachClick(result) {
        const idKey = result?.[0]?.findIndex((cell) => ID_KEYS.includes(cell.toLowerCase())) ?? -1;
        const tagKey = result?.[0]?.findIndex((cell) => TAG_KEYS.includes(cell.toLowerCase())) ?? -1;
        const commentKey = result?.[0]?.findIndex((cell) => COMMENT_KEY.includes(cell.toLowerCase())) ?? -1;

        if (idKey !== -1 && tagKey != -1 && result?.length > HEADER_LENGTH) {
            const attachData = result.reduce((acc: Dict<any>[], row: string[]) => {
                if (!ID_KEYS.includes(row[idKey]) && row[idKey]) {
                    acc.push({
                        tag: row[tagKey],
                        cars: [row[idKey]],
                        comment: commentKey !== -1 ? row[commentKey] ?? '' : '',
                    });
                }

                return acc;
            }, []);

            this.setState({
                attachData,
                closeOnFinish: true,
            });
        } else {
            this.setState({
                warning: 'Кажется в файле не хватает данных',
            });
        }
    }

    triggerModal(key) {
        this.setState((prev) => ({
            [key]: !prev?.[key],
        }));
    }

    customTagModal(data) {
        this.setState({
            attachData: data,
        });
    }

    attachmentHandler(data, isProposition) {
        this.customTagModal([{
            ...data,
            isProposition,
            cars: this.state.cars,
        }]);
        this.triggerModal('tagModalIsOpen');
    }

    render() {
        const {
            error,
            carNumberArbitraryList,
            isCarListLoading,
            warning,
            tagModalIsOpen,
            cars,
            attachData,
            closeOnFinish,
            parsingError,
        } = this.state;

        return <div>
            {warning.length
                ? <Window title={'Предупреждение'}
                          onClose={() => {
                              this.setState({
                                  warning: '',
                              });
                          }}>
                    <div className={style.warning_modal}>
                        {warning}
                        <Button basic
                                colorType={ButtonTypes.negative}
                                onClick={() => {
                                    this.setState({
                                        warning: '',
                                    });
                                }}>Закрыть</Button>
                    </div>
                </Window>
                : null
            }
            {tagModalIsOpen &&
            <TagModal onClose={this.triggerModal.bind(this, 'tagModalIsOpen')}
                      attachmentHandler={this.attachmentHandler.bind(this)}
                      queueLength={cars.length}/>
            }
            {attachData &&
            <GroupAttachModal onClose={this.customTagModal.bind(this, null)}
                              data={attachData}
                              openOnFinish={closeOnFinish}/>
            }
            {carNumberArbitraryList && (
                <CarNumberArbitraryList onClose={this.triggerModal.bind(this, 'carNumberArbitraryList')}/>
            )}
            {error
                ? <SimpleError error={error}/>
                : <>
                    <Button className={style.button}
                            colorType={ButtonTypes.positive}
                            isLoading={isCarListLoading}
                            onClick={this.onExportClick.bind(this)}>Экспорт CSV</Button>
                    {parsingError && <SimpleError error={parsingError}/>}
                    <Button className={style.button}
                            onClick={readTable.bind(this, this.onGroupAttachClick.bind(this))}
                            ytLog={{
                                button_name: buttonNameDetails.GROUP_ATTACH_TAG,
                                location: buttonLocationDetails.CARS_PACKAGE,
                            }}>
                        Групповой тег
                    </Button>
                    <Button className={style.button}
                            onClick={readTable.bind(this, this.onTagsAttachClick.bind(this))}
                            ytLog={{
                                button_name: buttonNameDetails.MULTI_TAG_ATTACH,
                                location: buttonLocationDetails.CARS_PACKAGE,
                            }}>Навешивание из таблицы id + tag + comment(необяз.)</Button>
                    <Button className={style.button}
                            onClick={this.triggerModal.bind(this, 'carNumberArbitraryList')}
                            ytLog={{
                                button_name: buttonNameDetails.CAR_NUMBER_ARBITRARY_LIST,
                                location: buttonLocationDetails.CARS_PACKAGE,
                            }}>Управление тегами по списку ГРЗ</Button>
                </>
            }
        </div>;
    }
}
