import produce from 'immer';
import React, { RefObject } from 'react';
import VirtualList from 'react-tiny-virtual-list';

import { Dict } from '../../../../types';
import { EMPTY_DATA, ONE_SECOND } from '../../../constants';
import { Button } from '../../../ui/Button';
import { Cross } from '../../../ui/Cross';
import FormatDate from '../../../ui/FormatDate';
import { Confirm, Window } from '../../../ui/FullModal';
import { Input } from '../../../ui/Input';
import { Link } from '../../../ui/Link';
import { Request2 } from '../../../utils/request';
import { FormConstructor } from '../../FormConstructor';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import * as style from './index.css';
import { REQUESTS, TO_REQUESTS } from './request';
import { maintenance } from './shema';

const LIST_ITEM_HEIGHT = 40;

interface ICustomMaintenanceState {
    isLoading: boolean;
    error: Error | null;
    data: Dict<any> [];
    filteredData: Dict<any> [];

    showEditDialog: boolean;
    showConfirmDelete: boolean;
    errorDialog: Error | null;
    selectedItem: Dict<any> ;
    isWorking: boolean;
    inputValue: string;

    windowHeight: number;
}

export class CustomMaintenance extends React.Component<any, ICustomMaintenanceState> {
    request = new Request2({ requestConfigs: TO_REQUESTS });
    state: ICustomMaintenanceState = {
        isLoading: false,
        error: null,
        data: [],
        filteredData: [],

        showEditDialog: false,
        showConfirmDelete: false,
        errorDialog: null,
        selectedItem: {} as any,
        isWorking: false,
        inputValue: '',

        windowHeight: window.innerHeight,
    };
    private listRef: RefObject<HTMLDivElement> | null = null;

    componentDidMount(): void {
        this.getData();
    }

    getData() {
        this.setState(produce(this.state, draft => {
            draft.isLoading = true;
            draft.error = null;
        }), () => {
            this.request.exec(REQUESTS.GET_MAINTENANCE)
                .then((request) => {
                    this.setState(produce(this.state, draft => {
                        draft.isLoading = false;
                        draft.error = null;
                        draft.data = request.maintenance || [];
                    }));
                })
                .catch(error => {
                    this.setState(produce(this.state, draft => {
                        draft.isLoading = false;
                        draft.error = error;
                    }));
                });
        });
    }

    crossHandle() {
        this.setState(produce(this.state, draft => {
            draft.isWorking = true;
        }), () => {
            this.request.exec(REQUESTS.DELETE_MAINTENANCE, {
                queryParams: {
                    vin: this.state.selectedItem.vin,
                },
            })
                .then(() => {
                    this.setState(produce(this.state, draft => {
                        draft.isWorking = false;
                        draft.errorDialog = null;
                        draft.showConfirmDelete = false;
                    }));
                    this.getData();
                })
                .catch(error => {
                    this.setState(produce(this.state, draft => {
                        draft.isWorking = false;
                        draft.errorDialog = error;
                    }));
                });
        });
    }

    openEditDialog(selectedItem) {
        this.setState(produce(this.state, draft => {
            draft.showEditDialog = true;
            draft.isWorking = false;
            draft.selectedItem = selectedItem;
        }));
    }

    showDialog(dialog: string, state: boolean, item: any) {
        this.setState(produce(this.state, draft => {
            draft[dialog] = state;
            draft.isWorking = false;
            draft.selectedItem = item;
        }));
    }

    setData(data) {
        this.setState(produce(this.state, draft => {
            draft.isWorking = true;
        }), () => {
            this.request.exec(REQUESTS.SET_MAINTENANCE, { body: data })
                .then(() => {
                    this.setState(produce(this.state, draft => {
                        draft.isWorking = false;
                        draft.errorDialog = null;
                        draft.showEditDialog = false;
                    }));
                    this.getData();
                })
                .catch(error => {
                    this.setState(produce(this.state, draft => {
                        draft.isWorking = false;
                        draft.errorDialog = error;
                    }));
                });
        });
    }

    onInputChange(inputValue) {
        this.setState({ inputValue, isLoading: true }, () => {
            this.filterData();
        });
    }

    filterData() {
        const value = this.state.inputValue.toLowerCase();
        const filteredData: Dict<any> [] = [];

        this.state.data?.forEach(item => {
            const { vin, source, mileage } = item;

            if(vin?.toLowerCase()?.includes(value)
                || source?.toLowerCase()?.includes(value)
                || mileage?.toString()?.toLowerCase()?.includes(value)) {
                filteredData.push(item);
            }
        });

        this.setState({ filteredData, isLoading: false });
    }

    getListHeight() {
        const topPosition = this.listRef?.current?.offsetTop || 0;
        const MARGIN = 275;

        return this.state.windowHeight - topPosition - MARGIN;
    }

    render() {
        const { showEditDialog, error, selectedItem, inputValue, filteredData,
            errorDialog, isWorking, isLoading, showConfirmDelete } = this.state;

        const data = inputValue ? filteredData : this.state.data;

        return <div className={style.component}>
            <div className={style.controls}>
                <Button onClick={this.openEditDialog.bind(this, {})}
                        className={style.button}>Добавить</Button>
                <Input value={inputValue}
                       onChange={this.onInputChange.bind(this)}
                       placeholder={'Фильтр по ТО'}
                       className={style.input}/>
            </div>

            {
                showEditDialog
                && <EditDialog selectedItem={selectedItem}
                               error={errorDialog}
                               setData={this.setData.bind(this)}
                               isWorking={isWorking}
                               onClose={this.showDialog.bind(this, 'showEditDialog', false)}/>
            }
            {
                showConfirmDelete
                && <Confirm error={errorDialog}
                            onClose={this.showDialog.bind(this, 'showConfirmDelete', false)}
                            isWorking={isWorking}
                            accept={this.crossHandle.bind(this)}
                            question={<><span>Удалить выбранный элемент с</span>
                                <div><strong>vin: </strong>{selectedItem?.vin || ''}</div></>}
                            title={`Удаление информации о TO`}/>
            }
            {
                isLoading
                    ? <Spin/>
                    : error
                        ? <SimpleError error={error}/>
                        : <>
                            <div className={style.header}>
                                <span className={style.index}>#</span>
                                <span className={style.vin}>vin</span>
                                <span className={style.source}>source</span>
                                <span className={style.date}>start_date</span>
                                <span className={style.date}>ready_date</span>
                                <span className={style.mileage}>mileage</span>
                                <span className={style.edit}/>
                                <span className={style.cross}/>
                            </div>

                            <VirtualList width={'100%'}
                                         height={this.getListHeight()}
                                         itemCount={data.length}
                                         itemSize={LIST_ITEM_HEIGHT}
                                         renderItem={({ index, style }) => {
                                             const item: Dict<any> = data[index];

                                             return <TableItem _style={style}
                                                               index={index}
                                                               item={item}
                                                               key={item?.vin}
                                                               openEditDialog={this.openEditDialog.bind(this)}
                                                               crossHandle={this.showDialog
                                                                   .bind(this, 'showConfirmDelete', true)}/>;
                                         }}/>
                        </>
            }
        </div>;
    }
}

interface ITableItemProps {
    _style: Dict<any> ;
    index: number;
    item: Dict<any> ;
    key: string;
    openEditDialog: () => void;
    crossHandle: () => void;
}

const TableItem = (props: ITableItemProps) => {
    let { _style, item, index, openEditDialog, crossHandle } = props;

    return <div key={`${index}_${item.vin}`} style={_style} className={style.row}>
        <span className={style.index}>{++index}.</span>
        <span className={style.vin}>{item?.vin || EMPTY_DATA}</span>
        <span className={style.source}>{item?.source || EMPTY_DATA}</span>
        <span className={style.date}><FormatDate value={item?.start_date * ONE_SECOND} withSecond/></span>
        <span className={style.date}><FormatDate value={item?.ready_date * ONE_SECOND} withSecond/></span>
        <span className={style.mileage}>{item?.mileage || EMPTY_DATA}</span>
        <span className={style.edit}>
            <Link onClick={openEditDialog.bind(null, item)}>редактировать</Link>
        </span>
        <span className={style.cross}>
            <Cross onClick={crossHandle.bind(null, item)} className={style.cross}/>
        </span>
    </div>;
};

interface IEditDialogProps {
    selectedItem: any;
    onClose: () => void;
    setData: (data) => void;
    isWorking: boolean;
    error: any;
}

class EditDialog extends React.Component<IEditDialogProps> {
    data = {};

    onChange(data) {
        this.data = data;
    }

    send() {
        this.props.setData(this.data);
    }

    render() {
        return <Window onClose={this.props.onClose}
                       error={this.props.error}
                       title={`Информация о vin: ${this.props.selectedItem.vin || ''}`}>
            <FormConstructor schema={maintenance}
                             initialData={this.props.selectedItem}
                             onChange={this.onChange.bind(this)}/>
            <div className={style.controls}>
                <Button isLoading={this.props.isWorking}
                        onClick={this.send.bind(this)}>Записать</Button>
            </div>
        </Window>;
    }
}
