import * as React from 'react';
import XLSX from 'xlsx';

import { Dict } from '../../../../types';
import { Button } from '../../../ui/Button';
import { Link } from '../../../ui/Link';
import { Request2 } from '../../../utils/request';
import { ExcelUploader } from '../../ExcelUploader';
import { SimpleError } from '../../SimpleError';
import FuelCardsModal from './FuelCardsModal/component';
import * as style from './index.css';
import { REQUEST, REQUESTS } from './request';
import { IFuelCardCarItem } from './types';

const SUCCESS_LABEL_TIMEOUT = 5000;

interface IFuelCardsUploaderState {
    isUploading: boolean;
    uploadingSuccess: boolean;
    uploadingError: Error | null;
    fuelCardsList: IFuelCardCarItem[] | null;
    isListModalOpen: boolean;
}

export default class FuelCardsUploader extends React.Component<{}, IFuelCardsUploaderState> {
    state: IFuelCardsUploaderState = {
        isUploading: false,
        uploadingSuccess: false,
        uploadingError: null,
        fuelCardsList: null,
        isListModalOpen: false,
    };
    request = new Request2({ requestConfigs: REQUEST });

    componentWillUnmount() {
        this.request.abort();
    }

    ExcelFileInfoHandler(JSON: Dict<any>, workbook: XLSX.WorkBook) {
        const firstSheetName = workbook.SheetNames[0];
        const sheet = workbook.Sheets[firstSheetName];
        const listRaw: Dict<any>[] = XLSX.utils.sheet_to_json(sheet, { raw: false });

        const fuelCardsList: IFuelCardCarItem[] = listRaw.map((listItem: any) => {
            listItem.vin = listItem.VIN;
            delete listItem.VIN;

            return listItem;
        });

        this.setState({ fuelCardsList });
    }

    openListModal() {
        this.setState({ isListModalOpen: true });
    }

    closeListModal() {
        this.setState({ isListModalOpen: false });
    }

    uploadFuelCards() {
        const { fuelCardsList } = this.state;
        const body = {
            cars: fuelCardsList ?? [],
        };

        this.setState({ isUploading: true, uploadingError: null }, () => {
            this.request.exec(REQUESTS.UPLOAD_FUEL_CARDS, { body })
                .then(() => {
                    this.setState({ isUploading: false, uploadingSuccess: true }, () => {
                        setTimeout(() => {
                            this.setState({ uploadingSuccess: false });
                        }, SUCCESS_LABEL_TIMEOUT);
                    });
                })
                .catch(uploadingError => {
                    this.setState({ uploadingError, isUploading: false });
                });
        });
    }

    render() {
        const { fuelCardsList, isListModalOpen, isUploading, uploadingError, uploadingSuccess } = this.state;

        return <div>
            <ExcelUploader disabled={isUploading}
                           title={'Загрузить файл'}
                           JSONHandler={this.ExcelFileInfoHandler.bind(this)}/>
            {
                fuelCardsList
                    ? <>
                        <div className={style.list_info}>
                            {fuelCardsList?.length
                                ? <div>
                                    Найдено элементов: <Link onClick={this.openListModal.bind(this)}>
                                        {fuelCardsList.length}
                                    </Link>
                                </div>
                                : <h4>Элементов списка не найдено</h4>}
                        </div>
                        <div className={style.upload_button_container}>
                            <Button disabled={fuelCardsList?.length === 0}
                                    isLoading={isUploading}
                                    onClick={this.uploadFuelCards.bind(this)}>Загрузить</Button>
                        </div>
                    </>
                    : null
            }
            {
                uploadingError
                    ? <SimpleError error={uploadingError}
                                   data={{ label: 'Ошибка при загрузке списка топливных карт' }}/>
                    : null
            }
            {
                uploadingSuccess
                    ? <div className={style.success_info}>Загрузка прошла успешно</div>
                    : null
            }
            {
                isListModalOpen
                    ? <FuelCardsModal onClose={this.closeListModal.bind(this)} fuelCards={fuelCardsList ?? []}/>
                    : null
            }
        </div>;
    }
}
