import React from 'react';

import PhoneIcon from '../../../../svg-components/phone.component.svg';
import { ClassificationType } from '../../../../types';
import { TIMERS } from '../../../constants';
import { UserInfoHandler } from '../../../models/user';
import { Button, ButtonTypes } from '../../../ui/Button';
import { Cross } from '../../../ui/Cross';
import { Link } from '../../../ui/Link';
import { isObjectEqual } from '../../../utils/isObjectEqual';
import ClassificationDialog from '../../Chats/ClassificationDialog';
import { getTagId } from '../../ChatsCommon/chatUtils';
import { AttachUserRequestModal } from './AttachUserRequestModal';
import { ICallsNotifyProps } from './index';
import style from './index.css';
import { LastPerformers } from './LastPerformers';

interface ICallsNotifyState {
    menuIsOpen: boolean;
    showClassification: boolean;
    answered: boolean;
    userAttachModalOpen: boolean;
}

export class CallsNotify extends React.Component<ICallsNotifyProps, ICallsNotifyState> {
    state: ICallsNotifyState = {
        menuIsOpen: false,
        showClassification: false,
        answered: false,
        userAttachModalOpen: false,
    };
    closeTimer;

    menu: React.RefObject<HTMLDivElement>;

    call() {
        const callTag = this.props.calls?.[0]?.tag_data || null;
        const user = this.props.users?.find(user => user.id === callTag?.object_id) || {};
        const tag = this.props.tags?.find(tag => tag.name === callTag?.tag) || {};

        return { callTag, user, tag };
    }

    blurMenu() {
        this.setState(() => ({
            menuIsOpen: false,
        }));
    }

    componentWillUnmount(): void {
        this.menu?.current?.removeEventListener('blur', this.blurMenu.bind(this));
        clearTimeout(this.closeTimer);
    }

    constructor(props) {
        super(props);
        this.menu = React.createRef();
    }

    closeNotify() {
        this.setState({
            answered: false,
            menuIsOpen: false,
        });
    }

    componentDidUpdate(
        prevProps: Readonly<ICallsNotifyProps>,
        prevState: Readonly<ICallsNotifyState>,
        snapshot?: any,
    ): void {
        const user_id = this.props.calls?.[0]?.tag_data?.object_id || null;
        const prev_user_id = prevProps.calls?.[0]?.tag_data?.object_id;
        if (user_id !== prev_user_id) {
            clearTimeout(this.closeTimer);
            if (!user_id) {
                this.closeTimer = setInterval(() => {
                    this.closeNotify();
                }, TIMERS.CLOSE_CALL_NOTIFY);
            } else {
                this.closeNotify();
            }
        }
    }

    toggle() {
        clearTimeout(this.closeTimer);

        this.setState((prev) => ({
            menuIsOpen: !prev.menuIsOpen,
            answered: true,
        }), () => {
            if (this.state.menuIsOpen) {
                this.menu?.current?.focus();
            }
        });
    }

    onKeyUp(e: KeyboardEvent) {
        if (e.key === 'Escape') {
            this.blurMenu();
        }
    }

    showClassification(showClassification: boolean) {
        this.setState({
            showClassification,
        });
    }

    shouldComponentUpdate(
        nextProps: Readonly<ICallsNotifyProps>,
        nextState: Readonly<ICallsNotifyState>,
        nextContext: any,
    ): boolean {
        return !isObjectEqual(nextProps, this.props) || !isObjectEqual(nextState, this.state);
    }

    openUserAttachModal(userAttachModalOpen: boolean) {
        this.setState({ userAttachModalOpen });
    }

    render() {
        const { callTag, user, tag } = this.call();
        const userId = UserInfoHandler.getId.call(user);
        const printName = UserInfoHandler.getPrintName.call(user);
        const values = {
            tag_id: callTag?.tag_id ?? '',
            call_id: callTag?.call_id ?? '',
            user_id: callTag?.object_id ?? '',
        };

        return callTag && <div>
            <div className={style.component} onClick={this.toggle.bind(this)}>
                <div className={style.user}>
                    <div className={style.name}>{user.first_name || user.username}</div>
                    <div className={style.second_name}>{user.last_name}</div>
                </div>
                <PhoneIcon className={style.icon}/>
                <div className={`${style.ring} ${this.state.answered ? style.answered : ''}`}/>
            </div>

            {
                this.state.menuIsOpen &&
                <div className={style.menu}
                     ref={this.menu}
                     tabIndex={1}
                     onKeyUp={this.onKeyUp.bind(this)}>

                    <Cross onClick={this.blurMenu.bind(this)} className={style.close}/>
                    <Link href={`${location.origin}${location.pathname}`
                        + `#/clients/${userId}/info`}>
                        <h3>{printName || `${userId} (external)`}</h3>
                    </Link>
                    <div title={tag.comment} className={style.tag}>{tag.display_name}</div>
                    <Button colorType={ButtonTypes.warning}
                            className={style.classification_btn}
                            onClick={this.showClassification.bind(this, true)}>
                        Классификация
                    </Button>

                    <Button colorType={ButtonTypes.warning}
                            basic={true}
                            className={style.button}
                            onClick={this.openUserAttachModal.bind(this, true)}>
                        Привязать пользователя
                    </Button>

                    <LastPerformers user_id={this.props.calls?.[0]?.tag_data?.object_id || ''}/>
                    {this.state.showClassification
                        ? <ClassificationDialog onClassified={this.showClassification.bind(this, false)}
                                                type={ClassificationType.call_incoming}
                                                tag_id={getTagId(callTag)}
                                                user_id={callTag?.object_id}
                                                chat_id={callTag?.id || callTag?.tag_details?.topic_link}
                                                onClose={this.showClassification.bind(this, false)}/>
                        : null
                    }

                    {this.state.userAttachModalOpen
                        ? <AttachUserRequestModal values={values}
                                                  openBindClientModal={this.openUserAttachModal.bind(this)}/>
                        : null
                    }
                </div>
            }
        </div>;
    }
}
