import React from 'react';
import ReactTooltip from 'react-tooltip';

import { GlobalSearchProvider } from '../../../../decorators/GlobalSearchProvider';
import { GlobalSearchProviderExport } from '../../../../decorators/GlobalSearchProvider/types';
import CarInfo from '../../../models/car';
import { SessionHistoryInfoHandler } from '../../../models/session';
import { BasicUserInfo, UserInfoHandler } from '../../../models/user';
import { Arrow, ArrowPosition } from '../../../ui/Arrow';
import { Cross } from '../../../ui/Cross';
import FormatDate from '../../../ui/FormatDate';
import { Link } from '../../../ui/Link';
import { UserStatus } from '../../../ui/Status';
import { LabelStatus, TLabel } from '../../../ui/Table';
import { isObjectEqual } from '../../../utils/isObjectEqual';
import CarNumber from '../../CarNumber';
import { VisitedLog } from '../../Content/db';
import { SessionBadges } from '../../SessionsList/SessionItem/badges';
import Spin from '../../Spin';
import * as style from './index.css';

interface IGlobalSearchProps extends GlobalSearchProviderExport {
    BlockRules?: any;
}

const carCurrentTooltipId = 'car-tooltip';

@GlobalSearchProvider()
class GlobalSearch extends React.Component<IGlobalSearchProps> {
    state = { findText: '', showResult: false };
    ignoreBlur = false;
    topData = {};

    clear() {
        this.setState({
            findText: '',
            showResult: false,
        });
    }

    changeFindText(e) {
        this.setState({
            findText: e.target.value,
        }, () => {
            this.props.search
            && this.props.search({ has_all_of: this.state?.findText, withSessions: true });
        });
    }

    toggle() {
        this.setState({
            showResult: !this.state.showResult,
        });
    }

    arrowHandle(e: KeyboardEvent) {
        e.preventDefault();
        e.stopPropagation();
        this.toggle();
    }

    componentDidUpdate(prevProps: Readonly<IGlobalSearchProps>, prevState: Readonly<{}>, snapshot?: any) {
        if (!isObjectEqual(this.props, prevProps)) {
            const new_cars = this.props?.searchData?.objects?.cars || [];
            const new_users = this.props?.searchData?.objects?.users || [];
            const mixedSession = this.props?.searchData?.mixedSession;

            this.setState({
                showResult: !!(new_cars.length || new_users.length || mixedSession),
            });
        }
    }

    keyUp(e: KeyboardEvent) {
        if (e.key === 'Escape') {
            this.toggle();
        }
    }

    onFocus() {
        this.setState({
            showResult: true,
        });
    }

    setIgnoreBlur() {
        this.ignoreBlur = true;
    }

    clearIgnoreBlur() {
        this.ignoreBlur = false;
    }

    onBlur() {
        if (!this.ignoreBlur) {
            this.setState({
                showResult: false,
            });
        }
    }

    componentDidMount(): void {
        new VisitedLog().getTopIds().then(data => {
            this.topData = data;
        });
    }

    onRowClick(url: string) {
        this.clear();
        location.hash = url;
    }

    onLinkClick(link: string) {
        this.onBlur();
        location.hash = link;
    }

    render() {
        const cars = this.props.searchData && this.props.searchData.objects && this.props.searchData.objects.cars || [];
        const users = this.props.searchData && this.props.searchData.objects
            && this.props.searchData.objects.users || [];
        const mixedSession = this.props.searchData?.mixedSession;

        return <div className={style.component}
                    tabIndex={-1}
                    onMouseDown={this.setIgnoreBlur.bind(this)}
                    onMouseUp={this.clearIgnoreBlur.bind(this)}
                    onMouseOut={this.clearIgnoreBlur.bind(this)}
                    onBlur={this.onBlur.bind(this)}>
            <div className={`${style.all_controls} ${style.is_loading}`}>
                {this.props.searchIsLoading && <Spin size={'s'}/> || null}
            </div>
            <div className={`${style.all_controls} ${style.controls} ${!this.state.findText && style.empty_data || ''}`}>
                <Arrow position={this.state.showResult ? ArrowPosition.TOP : ArrowPosition.BOTTOM}
                       onClick={this.arrowHandle.bind(this)}/>
                <Cross onClick={this.clear.bind(this)}/>
            </div>
            <input placeholder={'поиск'}
                   value={this.state.findText}
                   onChange={this.changeFindText.bind(this)}
                   onFocus={this.onFocus.bind(this)}
                   onClick={this.onFocus.bind(this)}
                   onKeyUp={this.keyUp.bind(this)}/>
            <div className={`${style.result} ${this.state.showResult && style.show}`}>
                <ReactTooltip id={carCurrentTooltipId}
                              html/>
                <div className={style.summary}>
                    Найдено: <strong>машин:</strong> {cars.length}, <strong>пользователей:</strong> {users.length} {
                        mixedSession ? <><strong>сессий: </strong>1</> : null}
                </div>
                {
                    mixedSession &&
                    <div onClick={this.onRowClick.bind(this, `#/session/${SessionHistoryInfoHandler.getSessionId.call(mixedSession)}/billing`)}
                         className={style.result_item}>
                        <span className={style.car_item}>
                            <span className={style.label}>
                                <SessionBadges index={0} sessionsData={mixedSession} tagsObject={{}}/>
                            </span>
                            <CarNumber carInfo={SessionHistoryInfoHandler.getCar.call(mixedSession) as typeof CarInfo}
                                       externalTooltipId={carCurrentTooltipId}/>
                        </span>
                        <span className={style.car_item}>
                            <strong>Начало: </strong> <FormatDate withSecond
                                                                  value={SessionHistoryInfoHandler.getStart
                                                                      .call(mixedSession)}/>

                        </span>
                        <span className={style.car_item}>
                            <strong>Конец: </strong> <FormatDate withSecond
                                                                 value={SessionHistoryInfoHandler.getFinish
                                                                     .call(mixedSession)}/>
                        </span>

                    </div>
                }
                {
                    cars.map((car, index) => {
                        return <div key={index}
                                    onClick={this.onRowClick.bind(this, `#/cars/${car.id}/info`)}
                                    className={style.result_item}>
                            <span className={style.car_item}>
                                <span className={this.topData[car.id] ? style.label : style.hidden_label}>
                                    <TLabel status={LabelStatus.POSITIVE} text={'топ'}/>
                                </span>

                                <CarNumber carInfo={car as typeof CarInfo} externalTooltipId={carCurrentTooltipId}/>
                            </span>
                            <span className={style.car_item}>
                                <Link onClick={this.onLinkClick.bind(this, `#/cars/${car.id}/info`)}>
                                    {car.model_id}
                                </Link>
                            </span>
                            <span className={style.car_item}>{car.vin}</span>
                            <Link onClick={this.onLinkClick.bind(this, `#/tracks?car_id=${car.id}&status=`)}>
                                трек
                            </Link>
                        </div>;
                    })
                }
                {
                    users.map((user: BasicUserInfo, index) => {
                        const id = UserInfoHandler.getId.call(user);
                        const userUrl = `#/clients/${id}`;

                        return <div onClick={this.onRowClick.bind(this, `${userUrl}/info`)}
                                    key={index}
                                    className={style.result_item}>
                            <span className={`${style.user_item_span} ${style.user_print}`}>
                                <span className={this.topData[id] ? style.label : style.hidden_label}>
                                    <TLabel status={LabelStatus.POSITIVE} text={'топ'}/>
                                </span>

                                <Link onClick={this.onLinkClick.bind(this, `${userUrl}/info`)}>
                                    {UserInfoHandler.getPrintNameWithoutUserName.call(user)}</Link>
                            </span>
                            <span className={`${style.user_item_span} ${style.user_session}`}>
                                <Link onClick={this.onLinkClick.bind(this, `${userUrl}/sessions`)}>сессии</Link>
                            </span>
                            <span className={style.user_item_span}>
                                {UserInfoHandler.getPhone.call(user)}
                            </span>
                            <span className={style.user_item_span}>
                                <UserStatus text={UserInfoHandler.getStatus.call(user)}/>
                            </span>
                            {this.props?.BlockRules?.UserLogin
                                ? <>
                                    <span className={`${style.user_item_span} ${style.user_username}`}>
                                        {UserInfoHandler.getUsername.call(user)}
                                    </span>
                                    <span className={`${style.user_item_span} ${style.user_email}`}>
                                        {UserInfoHandler.getEmail.call(user)}
                                    </span>
                                </>
                                : null
                            }
                            <span className={`${style.user_item_span} ${style.user_session}`}>
                                <Link onClick={this.onLinkClick.bind(this, `#/tracks?user_id=${UserInfoHandler.getId.call(user)}&status=`)}>
                                    трек
                                </Link>
                            </span>
                        </div>;
                    })
                }
            </div>
        </div>;
    }
}

export { GlobalSearch };
