import * as React from 'react';

import { ENVIRONMENT, LSSettingItems } from '../../../types';
import LS from '../../utils/localStorage/localStorage';
import { CenteredSpin } from '../Spin';
import * as style from './index.css';

const RETRY_LIMIT = 1;

interface IImage {
    title?: string;
    src: string;
    fitTo?: 'width' | 'height';
    controls?: boolean;
    thumbnail?: boolean;
    readyScr?: boolean;
    className?: string;
    customDataIndex?: any;
    handleImageLoaded?: (img: HTMLImageElement) => void;
    isLoading?: boolean;
    error?: Error | null;
}

interface IState {
    isLoading: boolean;
    error: string;
    tryLimitCount: number;
    tryCurrentCount: number;
}

class Img extends React.Component <IImage, IState> {
    ls = new LS();
    rootElement: HTMLElement;
    img: HTMLImageElement;
    controls: any = {
        currentAngle: 0,
    };

    public static defaultProps: Partial<IImage> = {
        title: 'image',
        fitTo: 'width',
        controls: false,
        thumbnail: false,
    };

    resize() {
        const ratio = this.rootElement.offsetWidth / this.img.offsetWidth;
        if (ratio < 1) {
            this.img.width = this.img.width * ratio;
        }

        this.props?.handleImageLoaded?.(this.img);
    }

    UNSAFE_componentWillReceiveProps(newProps: any) {
        if (newProps.src !== this.props.src) {
            this.setState({ isLoading: true, error: '' });
        }
    }

    state = {
        isLoading: true,
        error: '',
        tryLimitCount: RETRY_LIMIT,
        tryCurrentCount: 0,
    };

    handleImageLoaded() {
        this.resize();
        this.setState({ isLoading: false });
    }

    retry() {
        const TIMEOUT = 500;

        if (this.state.tryCurrentCount <= this.state.tryLimitCount) {
            this.setState({ tryCurrentCount: this.state.tryCurrentCount + 1 }, () => {
                this.setState({ error: `Перезапрос №${this.state.tryCurrentCount}...` });
                setTimeout(() => {
                    this.img.src = this.img.src + (
                        this.img.src.indexOf('?random') === -1
                            ? '?random='
                            : '&random=')
                        + new Date().getTime();
                }, TIMEOUT);
            });
        }
    }

    handleImageErrored(err: any) {
        this.setState({ isLoading: false, error: 'Ошибка' });
        //this.retry();
    }

    onClickHandler(e: Event) {
        if (this.props.controls) {
            this.controls.currentAngle += 90;
            this.img.parentElement
            && this.img.parentElement.style
                .setProperty('transform', `rotate(${this.controls.currentAngle}deg)`);
        }
    }

    public render(): JSX.Element {
        const { readyScr, thumbnail, title, className, isLoading, customDataIndex } = this.props;
        const { error } = this.state;

        let src = this.props.src;
        if (!readyScr) {
            src = convertSrcTestingQa(this.props.src, this.ls);
        }

        if (thumbnail) {
            src = src.replace('content', 'thumbnail');
        }

        return (
            <div ref={(div: HTMLDivElement) => {
                this.rootElement = div;
            }}
                 title={title}
                 className={`${style.image} ${style.image_wrapper} ${(!this.props.src ? style.image_wrapper__empty_img : '')} `
                     + `${className || ''}`}>
                {isLoading
                    ? <CenteredSpin/>
                    : this.props.src
                        ? <img ref={(img: HTMLImageElement) => {
                            this.img = img;
                        }}
                               src={src}
                               data-index={customDataIndex ?? null}
                               onClick={this.onClickHandler.bind(this)}
                               onLoad={this.handleImageLoaded.bind(this)}
                               onError={this.handleImageErrored.bind(this)}
                               className={style['loading-' + this.state.isLoading]}/>
                        : null
                }
                {this.props.src && isLoading && this.props.children}
                {(this.props.src || this.props.src === '') && (error || this.props.error)
                    ? <div className={style.error}>
                        {error || (this.props.error ? 'Ошибка' : null)}
                    </div>
                    : null
                }
            </div>
        );
    }
}

export default Img;

export const convertSrcTestingQa = function (src, ls) { //#DRIVEFRONT-676
    const isTesting = ls.envIsTesting();
    const isQA = ls.get(LSSettingItems.env) === ENVIRONMENT.QA;

    const py_testing_qa_prefix = `/p${isQA ? '-qa' : isTesting ? '-testing' : 'y'}/`;

    return src?.replace(/(carsharing.n.yandex-team.ru|carsharing-testing.n.yandex-team.ru)/, location.host)
        .replace('/api/', py_testing_qa_prefix) || '';
};
