import * as React from 'react';
import { ReactElement } from 'react';

import { Dict } from '../../../types';
import { CancelButton, SaveButton } from '../../ui/Button';
import { Confirm, Window } from '../../ui/FullModal';
import * as coreStyle from '../../ui/index.css';
import { Link } from '../../ui/Link';
import { Request2 } from '../../utils/request';
import { FormConstructor } from '../FormConstructor';
import { ISchemaItem } from '../FormConstructor/types';
import { initMapApi } from '../MainMap/utils';
import Spin from '../Spin';
import { INCIDENT_LINK } from '../Support/IncidentsFeed/types';
import { REQUEST, REQUESTS } from './request';

interface IIncidentFormProps {
    onClose: () => void;
    carInfo?: Dict<any>;
    initialData?: Dict<any>;
}

interface IIncidentFormState {
    isLoading: boolean;
    isSending: boolean;
    loadingError: Error | null;
    sendingError: Error | null;
    schema: Dict<ISchemaItem>;
    formData: Dict<any>;
    isFormValid: boolean;
    initialData: Dict<any>;
    question: ReactElement | null;
}

export default class IncidentForm extends React.Component<IIncidentFormProps, IIncidentFormState> {
    state: IIncidentFormState = {
        isLoading: false,
        isSending: false,
        loadingError: null,
        sendingError: null,
        schema: {},
        formData: {},
        isFormValid: true,
        initialData: {},
        question: null,
    };
    request = new Request2({ requestConfigs: REQUEST });

    componentDidMount(): void {

        this.getInitialData();

        this.setState({ isLoading: true, loadingError: null }, () => {
            this.request.exec(REQUESTS.GET_SCHEMA, { queryParams: { scheme: true } })
                .then((response => {
                    const request_data = response?.request_data ?? {};
                    const schema = request_data.structure ?? {};

                    const incidentTypeSchemaVariants = schema?.incident_type?.control_field?.incident_type?.variants;
                    if (incidentTypeSchemaVariants) {
                        schema?.incident_type?.control_field?.incident_type?.variants?.sort?.((a, b) => {
                            const textA = a?.text ?? a.value ?? '';
                            const textB = b?.text ?? b.value ?? '';

                            return textA.localeCompare(textB);
                        });
                    }

                    this.setState({ isLoading: false, schema });
                }))
                .catch(loadingError => {
                    this.setState({ loadingError, isLoading: false });
                });
        });
    }

    getInitialData() {
        const { carInfo, initialData: initialDataProps } = this.props;
        // @ts-ignore
        const ymaps = window.ymaps;
        if (!ymaps) {
            const mapScript = initMapApi();
            mapScript.addEventListener('load', () => {

                //one more window.ymaps because first ymaps is still empty
                // @ts-ignore
                this.processInitialData(carInfo, window.ymaps, initialDataProps);
            });
        } else {
            this.processInitialData(carInfo, ymaps, initialDataProps);
        }
    }

    processInitialData(carInfo, ymaps, initialDataProps) {
        if (carInfo) {
            const car_id = carInfo?.id;

            const coords = this.getCarInfoLocation(carInfo);

            ymaps.geocode(coords)
                .then((res: any) => {
                    const firstGeoObject = res.geoObjects.get(0);
                    const incident_car_address = `${firstGeoObject.getAddressLine()} (${coords?.reverse()?.toString()})`;
                    const incident_address = incident_car_address;

                    const initialData = { car_id, incident_car_address, incident_address };

                    this.setState({ initialData });
                });
        } else if (initialDataProps) {

            const { car_id } = initialDataProps;

            //session car info doesn't have location info
            this.request.exec(REQUESTS.GET_CAR_INFO, { queryParams: { car_id } })
                .then(carInfo => {
                    const coords = this.getCarInfoLocation(carInfo);

                    ymaps.ready(() => {
                        ymaps.geocode(coords)
                            .then((res: any) => {
                                const firstGeoObject = res.geoObjects.get(0);
                                const incident_car_address = `${firstGeoObject.getAddressLine()} (${coords?.reverse()?.toString()})`;
                                const incident_address = incident_car_address;

                                const initialData = Object.assign({
                                    incident_car_address,
                                    incident_address,
                                }, initialDataProps);
                                this.setState({ initialData });
                            });
                    });
                });
        }
    }

    getCarInfoLocation(carInfo: Dict<any>) {
        const location = carInfo?.location ?? {};
        const { lat, lon } = location;

        return [lon, lat];
    }

    onFormChange(formData: Dict<any>, isFormValid: boolean) {
        this.setState({ formData, isFormValid });
    }

    sendForm() {
        const { formData } = this.state;

        this.setState({ isSending: true }, () => {
            this.request.exec(REQUESTS.SEND_FORM, { body: formData })
                .then((response) => {
                    const stLink = response?.incident_data?.links
                        ?.find(el => el.type === INCIDENT_LINK.STARTREK_TICKET)?.ticket_url;
                    if (!stLink) {
                        this.props.onClose();
                        this.setState({ isSending: false });
                    } else {
                        this.setState({
                            isSending: false,
                            question: <div>
                                <Link href={stLink} target={'_blank'}>{stLink}</Link>
                            </div>,
                        });
                    }
                })
                .catch(sendingError => {
                    this.setState({ sendingError, isSending: false });
                });
        });
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    render() {
        const { onClose } = this.props;
        const { loadingError, schema, isLoading, isSending, sendingError, initialData, isFormValid } = this.state;

        return <Window error={loadingError || sendingError}
                       onClose={onClose.bind(this)}
                       title={'Инцидент'}
                       closeWithConfirm>
            {isLoading
                ? <Spin/>
                : <>
                    {
                        this.state.question &&
                        <Confirm onClose={this.props.onClose}
                                 acceptTitle={'Понятно'}
                                 error={null}
                                 title={'Создан тикет'}
                                 accept={this.props.onClose}
                                 question={this.state.question}/>
                    }
                    <FormConstructor rawResult
                                     schema={schema}
                                     initialData={initialData}
                                     onChange={this.onFormChange.bind(this)}/>
                    <div className={coreStyle.button_container}>
                        <CancelButton onClick={onClose.bind(this)}/>
                        <SaveButton disabled={isSending || !isFormValid} onClick={this.sendForm.bind(this)}/>
                    </div>
                </>}
        </Window>;
    }
}
