import * as React from 'react';

import { IInterfaceAdminProviderProps, InterfaceAdminProvider } from '../../../../decorators/InterfaceAdminProvider';
import { Dict } from '../../../../types';
import { Button, ButtonTypes } from '../../../ui/Button';
import { Cross } from '../../../ui/Cross';
import { Link } from '../../../ui/Link';
import { NoInformation } from '../../../ui/NoInformation';
import * as styleTable from '../../../ui/Table/index.css';
import { ISchemaItem } from '../../FormConstructor/types';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { TBodyItemProps } from '../renders';
import * as style from './index.css';

export interface ITableProps {
    heads: string[];
    body: string[];
    settingsKey: string;
    schema: Dict<ISchemaItem>;
    sortedBy: string;
    className?: string;
    renders?: Dict<any>;
    validator?: (data: any, changedData: Record<string,any>) => Error | null;
}

interface ICommonInterfaceConfig extends IInterfaceAdminProviderProps, ITableProps {
    maxItemsInList?: number;
}

const MARGIN = 230;

@InterfaceAdminProvider()
export class CommonInterfaceConfig extends React.Component<ICommonInterfaceConfig, any> {

    componentDidMount() {
        this.props.getDataByKey?.(this.props.settingsKey, this.props.schema);
    }

    render() {
        const {
            isLoading, error, addValue, editData, editRawData,
            deleteValue, data, heads, body, sortedBy, className, renders, maxItemsInList,
        } = this.props;
        const widthStyle = {
            maxHeight: `calc(100vh - ${MARGIN}px)`,
            height: 'auto',
            overflow: 'auto',
        };

        return <div className={style.wrapper}>
            {isLoading
                ? <Spin/>
                : error
                    ? <SimpleError error={error}/>
                    : <div className={className}>
                        <div>
                            {(maxItemsInList ? data !== undefined && data?.length < maxItemsInList : true)
                                ? <Button className={style.add_button}
                                          onClick={addValue?.bind(this)}>
                                    Добавить настройку
                                </Button>
                                : <span className={style.add_button}>
                                    Достинуто максимальное количество элементов в списке
                                </span>
                            }
                            <Button className={style.add_button}
                                    colorType={ButtonTypes.warning}
                                    onClick={editRawData?.bind(this)}>Сырые данные</Button>
                        </div>
                        <div style={widthStyle}>
                            <table className={`${styleTable.table} ${style.main_table}`}>
                                <Thead heads={heads}/>
                                <TBody data={data}
                                       body={body}
                                       sortedBy={sortedBy}
                                       editData={editData}
                                       renders={renders}
                                       deleteValue={deleteValue}/>
                            </table>
                        </div>
                    </div>
            }
        </div>;
    }
}

export const Thead = React.memo((props: { heads: string[] }) => {
    const { heads } = props;
    let size = heads.length;

    return <thead>
        <tr className={style.header}>
            <td key={0}>#</td>
            {
                heads.map((head, index) => {
                    return <td key={index + 1}>{head}</td>;
                })
            }
            <td key={size++}/>
            <td key={size++}/>
        </tr>
    </thead>;
});

interface ITBody {
    body: string[];
    data?: any[];
    sortedBy: string;
    editData?: (item: Dict<any>) => void;
    deleteValue?: (item: Dict<any>) => void;
    renders?: Dict<any>;
}

export const TBodyItem = React.memo((props: TBodyItemProps) => {
    const { item, index, bodyItem, renders } = props;
    let component: any;
    if (renders?.[bodyItem]) {
        const Component = renders?.[bodyItem];
        component = <Component {...props}/> || null;
    } else {
        component = item;
    }

    return <td key={index + 1}>{component}</td>;
});

export const TBody = React.memo((props: ITBody) => {
    const { body, data, editData, deleteValue, sortedBy, renders } = props;
    let size = body.length;
    const sortedData =
        Array.isArray(data) ? data.sort((item1, item2) => {
            if ((typeof item1[sortedBy] === 'number') && (typeof item2[sortedBy] === 'number')) {
                return item1[sortedBy] - item2[sortedBy];
            }

            if ((typeof item1[sortedBy] === 'string') && (typeof item2[sortedBy] === 'string')) {
                return item1[sortedBy]?.localeCompare(item2[sortedBy]);
            }

            return 0;
        }) : [];

    return <tbody>
        {
            sortedData.length
                ? sortedData.map((item, index) => {
                    return <tr key={sortedBy + index}>
                        <td className={style.index} key={0}>{index + 1}</td>
                        {
                            body.map((bodyItem, index) => {
                                return <TBodyItem item={item[bodyItem]}
                                                  index={index}
                                                  key={index}
                                                  bodyItem={bodyItem}
                                                  raw={item}
                                                  renders={renders}/>;
                            })
                        }
                        <td key={size++}>
                            <Link onClick={editData?.bind(null, item)}>
                            edit
                            </Link>
                        </td>
                        <td key={size++}>
                            <Cross className={style.delete}
                                   onClick={deleteValue?.bind(null, item)}/>
                        </td>
                    </tr>;
                })
                : <NoInformation/>}
    </tbody>;
});
