import React from 'react';

import { IInterfaceAdminProviderProps, InterfaceAdminProvider } from '../../../../../decorators/InterfaceAdminProvider';
import CopyIcon from '../../../../../svg-components/copy.component.svg';
import TrashIcon from '../../../../../svg-components/trash.component.svg';
import { GVARS_INTERFACE_ADMIN_PREFIX } from '../../../../constants';
import { Button } from '../../../../ui/Button';
import { Window } from '../../../../ui/FullModal';
import { Link } from '../../../../ui/Link';
import { NAVIGATION } from '../../../../utils/navigation';
import { Copy } from '../../../Copy';
import Spin from '../../../Spin';
import { named_filters } from '../../adminConfigKeys';
import { schema } from '../schema';
import * as style from './index.css';

interface IFiltersModalProps extends IInterfaceAdminProviderProps {
    onChooseFilter: (selectedFilter: string) => void;
    setChosenFilter: (selectedFilter: Record<string, any>) => void;
    className?: string;
}

interface IFiltersModalState {
    isFiltersModalOpen: boolean;
}

interface INamedFiltersItem {
    id: string;
    name: string;
    filter: string;
}

const settingsKey = `${GVARS_INTERFACE_ADMIN_PREFIX}.${named_filters}`;

@InterfaceAdminProvider()
export class FiltersModal extends React.Component<IFiltersModalProps, IFiltersModalState> {
    state: IFiltersModalState = {
        isFiltersModalOpen: false,
    };

    openFiltersModal(isFiltersModalOpen: boolean) {
        const { getDataByKey } = this.props;
        isFiltersModalOpen && getDataByKey?.(settingsKey, schema);
        this.setState({ isFiltersModalOpen });
    }

    chooseFilter(selectedFilter: Record<string, any>) {
        const { onChooseFilter, setChosenFilter } = this.props;
        onChooseFilter(selectedFilter.filter);
        setChosenFilter(selectedFilter);
        this.openFiltersModal(false);
    }

    render() {
        const { data, isLoading, error, className, deleteValue } = this.props;
        const { isFiltersModalOpen } = this.state;

        return (
            <>
                <Button className={className}
                        onClick={this.openFiltersModal.bind(this, true)}>
                    Фильтры
                </Button>
                {isFiltersModalOpen &&
                    <Window title={'Выбрать фильтры'}
                            className={style.modal}
                            onClose={this.openFiltersModal.bind(this, false)}
                            error={error}>
                        {isLoading
                            ? <Spin/>
                            : data?.length
                                ? <>
                                    <div className={style.header}>
                                        <span className={style.cell}>id</span>
                                        <span className={style.cell}>Название</span>
                                        <span className={style.cell}>Фильтр</span>
                                    </div>
                                    <ul>
                                        {
                                            data.map((i: INamedFiltersItem) => {
                                                return <li key={i.id}>
                                                    <span>{i.id}</span>
                                                    <span>{i.name}</span>
                                                    <Link className={style.link}
                                                          title={'Выбрать фильтр'}
                                                          onClick={this.chooseFilter.bind(this, i)}>
                                                        {i.filter}
                                                    </Link>
                                                    <Copy text={i.filter}
                                                          className={style.copy_filter_icon}>
                                                        <CopyIcon className={style.copy_filter_icon}/>
                                                    </Copy>
                                                    <TrashIcon className={style.trash_button}
                                                               onClick={deleteValue?.bind(null, i)}/>
                                                </li>;
                                            })
                                        }
                                    </ul>
                                </>
                                : !error && <span>У вас нет наименованных фильтров</span>
                        }
                        <Link className={style.link}
                              target={'_blank'}
                              href={`#/${NAVIGATION.ROUTES.INTERFACE_ADMIN_CONFIG.uri}/named-filters`}>
                            Перейти к редактированию фильтров
                        </Link>
                    </Window>}
            </>
        );
    }
}
