import * as React from 'react';

import { EMPTY_DATA } from '../../../constants';
import { UserInfoHandler } from '../../../models/user';
import { Button, ButtonTypes } from '../../../ui/Button';
import FormatDate from '../../../ui/FormatDate';
import { Link } from '../../../ui/Link';
import { isObjectEqual } from '../../../utils/isObjectEqual';
import * as styles from '../index.css';
import { LiteConfirmationsItemDiff } from '../LiteConfirmationsItemDiff';
import { ActionType, IDiffActionsRoles, ILCProposition } from '../types';

interface ILiteConfirmationsItemProps {
    proposition: ILCProposition;
    diffActionsRoles?: IDiffActionsRoles;
}

const KIND = {
    E: 'редактирование',
    N: 'новый',
    D: 'удаление',
    A: 'изменения в массиве',
};

const INITIAL_LIMIT = 0;

export const LiteConfirmationsItem = React.memo((props: ILiteConfirmationsItemProps) => {
    const { proposition, diffActionsRoles } = props;
    let diff = proposition?.diff;
    const [maxCount, setMaxCount] = React.useState(INITIAL_LIMIT);
    const [areDisabled, disableButtons] = React.useState(false);

    const diffLength = diff?.length;

    if (maxCount < diff?.length) {
        diff = proposition?.diff?.slice(0, maxCount);
    }

    return <div className={styles.itemWrap}>
        <p>
            <strong>Комментарий к предложению: </strong>
            {proposition?.description ?? EMPTY_DATA}
        </p>
        <p>
            <strong>Дата создания: </strong>
            <FormatDate value={proposition.historyInstant}/>; {proposition.historyInstantLag} назад
        </p>
        <p>
            <strong>Автор: </strong>
            <Link href={`${location.origin}/#/clients/${UserInfoHandler.getId.call(proposition.author)}/info`}>
                {UserInfoHandler.getUsername.call(proposition.author)}
            </Link>
        </p>
        <p>
            <div className={styles.strong}>Добавлены роли:</div>
            {
                diffActionsRoles?.A?.roles?.map(el => {
                    return <div key={el} className={styles.diff_item}>{el}</div>;
                }) || EMPTY_DATA
            }
            <div className={styles.strong}>Удалены роли:</div>
            {
                diffActionsRoles?.R?.roles?.map(el => {
                    return <div key={el} className={styles.diff_item}>{el}</div>;
                }) || EMPTY_DATA
            }
            <div className={styles.strong}>Добавлены экшены:</div>
            {
                diffActionsRoles?.A?.actions?.map(el => {
                    return <div key={el} className={styles.diff_item}>{el}</div>;
                }) || EMPTY_DATA
            }
            <div className={styles.strong}>Удалены экшены:</div>
            {
                diffActionsRoles?.R?.actions?.map(el => {
                    return <div key={el} className={styles.diff_item}>{el}</div>;
                }) || EMPTY_DATA
            }
        </p>
        <div className={styles.tableWrap}>
            <table>
                <caption>Изменения:</caption>
                <tbody>
                    {diff?.map((diff: any, index) => {
                        return <React.Fragment key={index}>
                            <tr key={index}>
                                <td data-label={`${index + 1}.`}>‎</td>
                                <td data-label={'Тип'}>{KIND[diff.kind]}</td>
                                <td data-label={'Путь'}>{diff?.path?.join('.') ?? EMPTY_DATA}</td>
                                <td data-label={'Было'}>
                                    <LiteConfirmationsItemDiff diff={diff?.lhs}/>
                                </td>
                                <td data-label={'Стало'}>
                                    <LiteConfirmationsItemDiff diff={diff?.rhs}/>
                                </td>
                            </tr>
                            {diff.kind === 'A'
                                ? <tr>
                                    <td data-label={`Изменения в массиве из #${index + 1}`}>‎</td>
                                    <td data-label={'Тип'}>{KIND[diff?.item?.kind]}</td>
                                    <td data-label={'Было'}>
                                        <LiteConfirmationsItemDiff diff={diff.item?.lhs}/>
                                    </td>
                                    <td data-label={'Стало'}>
                                        <LiteConfirmationsItemDiff diff={diff.item?.rhs}/>
                                    </td>
                                </tr>
                                : null
                            }
                        </React.Fragment>;
                    })}
                    {maxCount < diffLength
                        ? <tr>
                            <td className={styles.maxCount}>
                                <Link onClick={setMaxCount.bind(null, diffLength)}>
                                Показать все (осталось {diffLength - maxCount})
                                </Link>
                            </td>
                        </tr>
                        : null
                    }
                </tbody>
            </table>
        </div>
        <div>
            <Button disabled={areDisabled}
                    className={styles.button}
                    colorType={ButtonTypes.positive}
                    onClick={proposition.propositionAction?.bind(null, ActionType.CONFIRM, disableButtons)}>
                Принять предложение
            </Button>
            <Button disabled={areDisabled}
                    className={styles.button}
                    colorType={ButtonTypes.negative}
                    onClick={proposition.propositionAction?.bind(null, ActionType.REJECT, disableButtons)}>
                Отклонить предложение
            </Button>
        </div>
    </div>;
}, (prevProps, nextProps) => {
    return isObjectEqual(prevProps.proposition, nextProps.proposition);
});

LiteConfirmationsItem.displayName = 'LiteConfirmationsItem';
