import * as React from 'react';

import { UserInfoHandler } from '../../models/user';
import { LocalizationReducerState } from '../../reducers/localizationReducer';
import { Button, ButtonTypes } from '../../ui/Button';
import Checkbox from '../../ui/Checkbox';
import { Cross } from '../../ui/Cross';
import { Confirm, Window } from '../../ui/FullModal';
import { Input } from '../../ui/Input';
import { Link } from '../../ui/Link';
import * as styleTable from '../../ui/Table/index.css';
import { Request2 } from '../../utils/request';
import { Translate } from '../../utils/translate';
import { FormConstructor } from '../FormConstructor';
import { ProposeCommentModal } from '../Settings/ProposeCommentModal';
import { SimpleError } from '../SimpleError';
import Spin from '../Spin';
import * as style from './index.css';
import { CORE_DRIVE_LENS_REQUESTS as requestConfigs, REQUESTS } from './request';

export enum LANGUAGES {
    ENG = 'eng',
    RUS = 'rus',
    GER = 'ger',
    CZE = 'cze',
}

enum ProposeAction {
    REJECT = 'REJECT',
    CONFIRM = 'CONFIRM'
}

export interface ILocalItem {
    l: LANGUAGES;
    v: string;
}

const ADD_LOCALIZATION_ITEM_SCHEMA = {
    resource_id: {
        type: 'string',
        display_name: 'resource_id',
    },
    localizations: {
        type: 'array_types',
        display_name: 'Переводы',
        array_type: {
            l: {
                type: 'variants',
                multi_select: false,
                display_name: 'Язык',
                variants: [LANGUAGES.RUS, LANGUAGES.ENG, LANGUAGES.CZE, LANGUAGES.GER],
            },
            v: {
                type: 'text',
                display_name: 'Текст',
            },
        },
    },
};

export interface Localization {
    localizations: ILocalItem[];
    resource_id: string;
    revision?: number;
}

export interface Proposition {
    localizations: Localization[];
    proposition_author: string;
    proposition_description: string;
    proposition_id: string;
    resource_id: string;
}

interface ILocalSettingsState {
    dataIsLoading: boolean;
    confirmIsOpen: boolean;
    isWorking: boolean;
    question: string;
    accept: () => void;
    data: Localization[];
    item: null | string;
    removeError: null | Error;
    filter: string;
    showPropositions: boolean;
    propositionIsLoading: boolean;
    propositions?: Proposition[];
    users: Array<Record<string, any>>;
    loadingError: Error | null;
}

interface ILocalSettingsProps {
    location: any;
    Lang: LocalizationReducerState;
}

const NEW_ITEM_URL = '/settings/local?item=';
const NEW_ITEM_BACK_URL = '/settings/local';

const ITEM_URL_PARAM = 'item';

export default class LocalSettings extends React.Component<ILocalSettingsProps, ILocalSettingsState> {
    state: ILocalSettingsState = {
        dataIsLoading: false,
        isWorking: false,
        confirmIsOpen: false,
        question: '',
        accept: () => {},
        data: [],
        item: null,
        removeError: null,
        filter: '',
        showPropositions: false,
        propositionIsLoading: false,
        users: [],
        loadingError: null,
    };
    request = new Request2({
        requestConfigs,
    });
    t = this.props.Lang && new Translate(this.props.Lang) || {} as Translate;

    componentDidMount() {
        const params = new URLSearchParams(this.props.location.search);
        const item = params.get(ITEM_URL_PARAM);
        this.setState({ item });

        this.getData();
    }

    componentDidUpdate(prevProps: Readonly<any>): void {
        if (prevProps.location.search !== this.props.location.search) {
            const prevParams = new URLSearchParams(prevProps.location.search);
            const params = new URLSearchParams(this.props.location.search);
            const prevItem = prevParams.get(ITEM_URL_PARAM);
            const item = params.get(ITEM_URL_PARAM);

            if (prevItem !== item) {
                this.setState({ item });
            }
        }

    }

    getData() {
        this.setState({
            dataIsLoading: true,
        }, () => {
            this.request.exec(REQUESTS.GET_LOCALIZATIONS)
                .then(response => {
                    this.setState({
                        dataIsLoading: false,
                        data: response && response.resources || [],
                    });
                });
        });
    }

    onClose() {
        this.setState({
            confirmIsOpen: false,
            isWorking: false,
            removeError: null,
        });
    }

    remove(id: string) {
        this.setState({
            confirmIsOpen: true,
            question: `Удалить перевод <b>${id}</b>?`,
            accept: () => {
                this.setState({
                    isWorking: true,
                }, () => {
                    this.request.exec(REQUESTS.REMOVE_LOCALIZATIONS, { body: { id: [decodeURIComponent(id)] } })
                        .then(() => {
                            this.getData();
                            this.onClose();
                        })
                        .catch((removeError) => {
                            this.setState({
                                isWorking: false,
                                removeError,
                            });
                        });
                });
            },
        });
    }

    addLocalization() {
        location.hash = NEW_ITEM_URL;
    }

    close() {
        location.hash = NEW_ITEM_BACK_URL;
    }

    upsert(data: any) {
        const formattedData = this.formatData(data);

        this.setState({ isWorking: true }, () => {
            this.request.exec(REQUESTS.UPSERT_LOCALIZATIONS, { body: formattedData })
                .then(() => {
                    this.getData();
                    this.setState({
                        isWorking: false,
                    });
                    this.close();
                })
                .catch(() => {
                    this.setState({
                        isWorking: false,
                    });
                });
        });
    }

    formatData(data) {
        //DRIVEFRONT-1631
        data.localizations = data.localizations.map(el => {
            if (el.l && !el.v) {
                el.v = '';
            }

            if (!el.l && !el.v) {
                return {};
            }

            return el;
        });

        return data;
    }

    onFilterChange(filter: string) {
        this.setState({ filter });
    }

    onPropositionsChange(showPropositions: boolean) {
        if (!this.state.propositions && showPropositions) {
            this.getPropositions();
        }

        this.setState({ showPropositions, loadingError: null });
    }

    getPropositions() {
        this.setState({ propositionIsLoading: true }, () => {
            this.request.exec(REQUESTS.GET_PROPOSITIONS)
                .then((response) => {
                    this.setState({
                        propositionIsLoading: false,
                        propositions: response?.propositions || [],
                        users: response?.propositions_users || [],
                    });
                })
                .catch((loadingError) => {
                    this.setState({ loadingError, propositionIsLoading: false });
                });
        });
    }

    render() {
        const {
            dataIsLoading, filter, data, item, isWorking, confirmIsOpen, question,
            accept, removeError, showPropositions, propositionIsLoading, propositions, users, loadingError,
        } = this.state;
        const LOCAL_TEXT_PREVIEW_LENGTH = 600;
        const dataToDraw: any = showPropositions
            ? propositions || []
            : data.filter((item) => {
                return !filter || (filter && item.resource_id.toLowerCase().includes(filter.toLowerCase()));
            });

        return <>
            {
                dataIsLoading || propositionIsLoading
                && <Spin/>
                || <div className={style.localization_settings}>
                    <Input className={style.filter_input}
                           placeholder={'Фильтр'}
                           value={filter}
                           onChange={this.onFilterChange.bind(this)}
                           disabled={showPropositions}/>
                    <div className={style.button_block}>
                        <Button className={style.add_resource}
                                onClick={this.addLocalization.bind(this)}
                                disabled={showPropositions}>Новый перевод</Button>
                        <Checkbox checked={showPropositions}
                                  className={style.propositions_checkbox}
                                  onChange={this.onPropositionsChange.bind(this)}/>
                        <span>Показать предложения</span>
                    </div>
                    <div className={style.table_container}>
                        {loadingError
                            ? <SimpleError error={loadingError}/>
                            : <table className={`${styleTable.table} ${style.localization_table}`}>
                                <thead>
                                    <tr>
                                        <th>#</th>
                                        <th/>
                                        <th>ID</th>
                                        <th>{showPropositions ? 'Детали' : 'Текст локализации'}</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    {
                                        dataToDraw.map((item, index: number) => {
                                            const authorInfo = users.find((el) => el.id === item.proposition_author);
                                            const author = UserInfoHandler.getPrintName.call(authorInfo);

                                            return <tr key={index}>
                                                <td>{++index}</td>
                                                <td>
                                                    <Link className={style.remove}
                                                          onClick={this.remove.bind(this, item.resource_id)}>
                                                        {!showPropositions && <Cross/>}
                                                    </Link>
                                                </td>
                                                <td>
                                                    <Link href={`#${NEW_ITEM_URL}${decodeURIComponent(item.resource_id)}`}>
                                                        {item.resource_id}
                                                    </Link>
                                                </td>
                                                <td>
                                                    {!showPropositions
                                                        ? item.localizations
                                                && item.localizations.map((_l, key: number) => {
                                                    return <div className={style.loc_item} key={key}>
                                                        <span className={`lang lang-${_l.l}`}>
                                                            {_l.l.toLocaleUpperCase()}
                                                        </span>: <span className={style.loc_item_value}>
                                                            {_l.v.length > LOCAL_TEXT_PREVIEW_LENGTH
                                                                ? <span>{_l.v.substr(0, LOCAL_TEXT_PREVIEW_LENGTH)}
                                                                    <span> </span>
                                                                    <Link href={`#${NEW_ITEM_URL}${item.resource_id}`}>
                                                        [Ещё...]
                                                                    </Link></span>
                                                                : _l.v
                                                            }
                                                        </span>
                                                    </div>;
                                                })
                                                        : <div className={style.loc_item}>
                                                            <div>
                                                                <span>
                                                                Автор
                                                                </span>: <span className={style.loc_item_value}>
                                                                    <Link href={`#/clients/${item.proposition_author}/info`}
                                                                          target={'_blank'}>
                                                                        {author}
                                                                    </Link>
                                                                </span>
                                                            </div>
                                                            <div>
                                                                <span>
                                                                Комментарий
                                                                </span>: <span className={style.loc_item_value}>
                                                                    {item.proposition_description}
                                                                </span>
                                                            </div>
                                                        </div>
                                                    }
                                                </td>
                                            </ tr>;
                                        })
                                    }
                                </tbody>
                            </table>}
                    </div>
                    {item !== null
                        ? <LocalizationWindowItem {...this.props}
                                                  isWorking={isWorking}
                                                  schema={ADD_LOCALIZATION_ITEM_SCHEMA}
                                                  upsert={this.upsert.bind(this)}
                                                  onClose={this.close.bind(this)}
                                                  formatData={this.formatData.bind(this)}
                                                  showPropositions={showPropositions}
                                                  getPropositions={this.getPropositions.bind(this)}
                                                  getData={this.getData.bind(this)}/>
                        : null}
                    {
                        confirmIsOpen
                        && <Confirm onClose={this.onClose.bind(this)}
                                    question={question}
                                    isWorking={isWorking}
                                    error={removeError}
                                    accept={accept.bind(this)}/>
                    }
                </div>
            }
        </>;
    }
}

interface ILocalizationWindowItemProps {
    location: any;
    schema: { [key: string]: any };
    onClose: () => void;
    upsert: (values: Localization) => void;
    isWorking: boolean;
    formatData: (data: Localization) => Localization;
    showPropositions: boolean;
    getPropositions: () => void;
    getData: () => void;
}

interface ILocalizationWindowItemState {
    allResources: Localization[];
    currentResource: Localization | Proposition | null;
    isLoading: boolean;
    values: Localization;
    warning: string;
    showProposeComment: boolean;
    proposeError: Error | null;
    isWorkingPropose: boolean;
    proposeAction: ProposeAction | null;
}

class LocalizationWindowItem extends React.PureComponent<ILocalizationWindowItemProps, ILocalizationWindowItemState> {
    state: ILocalizationWindowItemState = {
        allResources: [],
        currentResource: null,
        isLoading: false,
        values: {} as Localization,
        warning: '',
        showProposeComment: false,
        proposeError: null,
        isWorkingPropose: false,
        proposeAction: null,
    };
    request = new Request2({
        requestConfigs,
    });

    getResource() {
        const { location, showPropositions } = this.props;
        const params = new URLSearchParams(location.search);
        const item: string | null = params.get(ITEM_URL_PARAM);

        this.setState({ isLoading: true }, () => {
            this.request.exec(REQUESTS[showPropositions ? 'GET_PROPOSITIONS' : 'GET_LOCALIZATIONS'])
                .then((response) => {
                    const allResources = showPropositions ? response.propositions : response.resources;
                    if (item) {
                        const currentResource = allResources.filter((resource) => resource.resource_id === item)[0];
                        this.setState({ isLoading: false, allResources, currentResource });
                    } else {
                        this.setState({ isLoading: false, allResources });
                    }

                })
                .catch(() => {
                    this.setState({ isLoading: false });
                });
        });
    }

    UNSAFE_componentWillReceiveProps(nextProps: Readonly<ILocalizationWindowItemProps>): void {
        if ((this.props.location
            && this.props.location.pathname
            && nextProps.location
            && nextProps.location.pathname)
            && (this.props.location.pathname !== nextProps.location.pathname)) {
            this.getResource();
        }
    }

    componentDidMount(): void {
        this.getResource();
    }

    onChangeForm(values: Localization) {
        const params = new URLSearchParams(this.props.location.search);
        const item: string | null = params.get(ITEM_URL_PARAM);
        let warning = '';

        const { resource_id: resourceId } = values;
        if (resourceId) {
            const allResources = this.state.allResources;

            const isExist = allResources.some(resource => resource.resource_id === resourceId);

            if (isExist && (!item || item !== resourceId)) {
                warning = `Ресурс ${resourceId} уже существует, сохранение обновит его данные`;
            }

        }

        this.setState({ values, warning });
    }

    onSaveValues() {
        const { values, currentResource } = this.state;
        const { localizations, resource_id } = values;

        const fullLocalizations = localizations?.map(el => {
            if (!el.v) {
                el.v = resource_id;
            }

            return el;
        });

        this.props.upsert({
            ...values,
            revision: (currentResource as Localization)?.revision,
            localizations: fullLocalizations,
        });
    }

    propose() {
        this.setState({
            showProposeComment: true,
            isWorkingPropose: false,
        });
    }

    showComment(state) {
        this.setState({
            showProposeComment: state,
        });
    }

    setComment(comment) {
        const { values, currentResource } = this.state;
        const { formatData, getPropositions, onClose } = this.props;
        const formattedData = formatData(values);
        const proposeObject = {
            resource_data: {
                localizations: formattedData.localizations,
            },
            resource_id: formattedData.resource_id,
            revision: (currentResource as Localization)?.revision,
        };

        this.setState({ isWorkingPropose: true, proposeError: null }, () => {
            this.request.exec(REQUESTS.LOCALIZATIONS_PROPOSE, {
                queryParams: {
                    comment: encodeURIComponent(comment),
                },
                body: proposeObject,
            })
                .then(() => {
                    this.setState({
                        isWorkingPropose: false,
                        proposeError: null,
                        showProposeComment: false,
                    }, () => {
                        onClose();
                        getPropositions();
                    });
                })
                .catch(proposeError => {
                    this.setState({ proposeError, isWorkingPropose: false });
                });
        });
    }

    showConfirm(proposeAction: ProposeAction, proposeError) {
        this.setState({
            proposeAction,
            proposeError,
        });
    }

    proposeAccept() {
        this.setState({
            proposeError: null,
            isWorkingPropose: true,
        }, () => {
            const { getPropositions, onClose, getData } = this.props;
            const currentResource = this.state.currentResource as Proposition;
            const action = `LOCALIZATIONS_${this.state.proposeAction}`;

            this.request.exec(REQUESTS?.[action], {
                queryParams: {
                    comment: currentResource.proposition_description,
                },
                body: {
                    proposition_ids: [currentResource?.proposition_id],
                },
            })
                .then(() => {
                    this.setState({
                        proposeError: null,
                        isWorkingPropose: false,
                        proposeAction: null,
                    }, () => {
                        onClose();
                        getPropositions();
                        getData();
                    });
                })
                .catch((proposeError) => {
                    let error = proposeError;
                    if (error?.data?.error_details?.special_info?.session_info
                        ?.hasOwnProperty('self confirm is denied')) {
                        error = new Error('Нельзя подтверждать свои же предложения :(');
                    }

                    this.setState({
                        proposeError: error,
                        isWorkingPropose: false,
                    });
                });
        });
    }

    render() {
        const {
            currentResource, isLoading, warning, showProposeComment, proposeError,
            isWorkingPropose, proposeAction,
        } = this.state;
        const { onClose, schema, isWorking, showPropositions } = this.props;

        return <Window title={'Localization'}
                       className={style.localization_window}
                       onClose={onClose.bind(this)}>
            {showProposeComment
                && <ProposeCommentModal isWorking={isWorkingPropose}
                                        error={proposeError}
                                        isRequired={true}
                                        setComment={this.setComment.bind(this)}
                                        onClose={this.showComment.bind(this, false)}/>
            }
            {proposeAction && <Confirm error={proposeError}
                                       isWorking={isWorkingPropose}
                                       title={proposeAction}
                                       question={`Выполнить действие?`}
                                       accept={this.proposeAccept.bind(this)}
                                       onClose={this.showConfirm.bind(this, null)}/>
            }
            <div className={style.add_localization}>
                {isLoading
                    ? <Spin/>
                    : <FormConstructor initialData={
                        currentResource && currentResource.resource_id
                            ? {
                                resource_id: currentResource.resource_id,
                                localizations: currentResource.localizations || [],
                            }
                            : {
                                localizations: [{ l: 'rus' }, { l: 'eng' }],
                                resource_id: '',
                            }
                    }
                                       schema={schema}
                                       onChange={this.onChangeForm.bind(this)}/>
                }
                {warning
                    ? <h3 className={style.warning}>{warning}</h3>
                    : null}
                {showPropositions
                    ? <div className={style.controls}>
                        <Button colorType={ButtonTypes.negative}
                                onClick={this.showConfirm.bind(this, ProposeAction.REJECT, null)}
                                isLoading={isWorking}>
                            Отклонить
                        </Button>
                        <Button colorType={ButtonTypes.positive}
                                onClick={this.showConfirm.bind(this, ProposeAction.CONFIRM, null)}
                                isLoading={isWorking}>
                            Принять
                        </Button>
                    </div>
                    : <div className={style.controls}>
                        <Button colorType={ButtonTypes.negative}
                                onClick={onClose.bind(this)}
                                basic>
                            Отмена
                        </Button>
                        <Button colorType={ButtonTypes.warning}
                                onClick={this.propose.bind(this)}
                                isLoading={isWorking}>
                            Предложить
                        </Button>
                        <Button isLoading={isWorking}
                                onClick={this.onSaveValues.bind(this)}>
                            Отправить
                        </Button>
                    </div>}
            </div>
        </Window>;
    }
}
