import * as React from 'react';

import { Dict } from '../../../types';
import { LocalizationReducerState } from '../../reducers/localizationReducer';
import { Button, ButtonTypes } from '../../ui/Button';
import Checkbox from '../../ui/Checkbox';
import { Cross } from '../../ui/Cross';
import { Window } from '../../ui/FullModal';
import * as coreStyle from '../../ui/index.css';
import { Input } from '../../ui/Input';
import { Link } from '../../ui/Link';
import * as styleTable from '../../ui/Table/index.css';
import { Request2 } from '../../utils/request';
import { Translate } from '../../utils/translate';
import { FormConstructor } from '../FormConstructor';
import Spin from '../Spin';
import * as style from './index.css';
import { CORE_DRIVE_LENS_REQUESTS as requestConfigs, REQUESTS } from './request';

export enum LANGUAGES {
    ENG = 'eng',
    RUS = 'rus',
}

export interface ILocalItem {
    l: LANGUAGES;
    v: string;
}

const ADD_LOCALIZATION_ITEM_SCHEMA = {
    resource_id: {
        type: 'string',
        display_name: 'resource_id',
        order: 1,
    },
    duplicate: {
        type: 'bool',
        display_name: 'Дублировать',
        default: true,
        order: 3,
    },
    localizations: {
        type: 'structure',
        order: 2,
        structure: {
            testing: {
                type: 'array_types',
                display_name: 'Переводы',
                tab_name: 'testing',
                array_type: {
                    l: {
                        type: 'variants',
                        multi_select: false,
                        display_name: 'Язык',
                        variants: [LANGUAGES.RUS, LANGUAGES.ENG],
                    },
                    v: {
                        type: 'text',
                        display_name: 'Текст',
                    },
                },
            },
            production: {
                type: 'array_types',
                display_name: 'Переводы',
                tab_name: 'production',
                array_type: {
                    l: {
                        type: 'variants',
                        multi_select: false,
                        display_name: 'Язык',
                        variants: [LANGUAGES.RUS, LANGUAGES.ENG],
                    },
                    v: {
                        type: 'text',
                        display_name: 'Текст',
                    },
                },
            },
        },

    },
};

export interface Localization {
    localizations: ILocalItem[];
    resource_id: string;
}

interface ILocalSettingsState {
    dataIsLoading: boolean;
    isWorking: boolean;
    question: string;
    accept: () => void;
    data: Dict<{ testing: ILocalItem[]; production: ILocalItem[] }>;
    item: null | string;
    removeError: null | Error;
    filter: string;
    isRemoveModal: boolean;
    removingResource: [string, ILocalItem[]] | null;
    upsertingError: null | Error;
}

interface ILocalSettingsProps {
    location: any;
    Lang: LocalizationReducerState;
}

const NEW_ITEM_URL = '/settings/local-compare?item=';
const NEW_ITEM_BACK_URL = '/settings/local-compare';

const ITEM_URL_PARAM = 'item';

export default class LocalSettingsCompare extends React.Component<ILocalSettingsProps, ILocalSettingsState> {
    state: ILocalSettingsState = {
        dataIsLoading: false,
        isWorking: false,
        question: '',
        accept: () => {},
        data: {},
        item: null,
        removeError: null,
        filter: '',
        isRemoveModal: false,
        removingResource: null,
        upsertingError: null,
    };
    request = new Request2({
        requestConfigs,
    });
    t = this.props.Lang && new Translate(this.props.Lang) || {} as Translate;

    componentDidMount() {
        const params = new URLSearchParams(this.props.location.search);
        const item = params.get(ITEM_URL_PARAM);
        this.setState({ item });

        this.getData();
    }

    componentDidUpdate(prevProps: Readonly<any>): void {
        if (prevProps.location.search !== this.props.location.search) {
            const prevParams = new URLSearchParams(prevProps.location.search);
            const params = new URLSearchParams(this.props.location.search);
            const prevItem = prevParams.get(ITEM_URL_PARAM);
            const item = params.get(ITEM_URL_PARAM);

            if (prevItem !== item) {
                this.setState({ item });
            }
        }

    }

    getData() {
        this.setState({
            dataIsLoading: true,
        }, () => {
            Promise.all([this.request.exec(REQUESTS.GET_LOCALIZATIONS, {
                queryParams: { id: '' }, headers: {
                    'saasnew': 'https://testing.carsharing.yandex.net',
                },
            }), this.request.exec(REQUESTS.GET_LOCALIZATIONS, {
                queryParams: { id: '' }, headers: {
                    'saasnew': 'https://prestable.carsharing.yandex.net',
                },
            })]).then(response => {
                let [testingArr = {}, productionArr = {}] = response;
                testingArr = testingArr.resources || [];
                productionArr = productionArr.resources || [];

                const localizations = {};

                testingArr.reduce((result: { testing: Dict<ILocalItem[]>; production: Dict<ILocalItem[]> },
                    localization: Localization) => {
                    if (result[localization.resource_id]) {
                        result[localization.resource_id].testing = localization.localizations;
                    } else {
                        result[localization.resource_id] = { testing: localization.localizations };
                    }

                    return result;
                }, localizations);

                productionArr.reduce((result: { testing: Dict<ILocalItem[]>; production: Dict<ILocalItem[]> },
                    localization: Localization) => {
                    if (result[localization.resource_id]) {
                        result[localization.resource_id].production = localization.localizations;
                    } else {
                        result[localization.resource_id] = { production: localization.localizations };
                    }

                    return result;
                }, localizations);

                this.setState({
                    dataIsLoading: false,
                    data: localizations,
                });
            });

        });
    }

    onClose() {
        this.setState({
            removingResource: null,
            removeError: null,
            isWorking: false,
        });
    }

    openRemoveModal(removingResource: [string, ILocalItem[]]) {
        this.setState({ removingResource });
    }

    closeRemoveModal() {
        this.setState({ removingResource: null });
    }

    remove(environments: Dict<boolean>) {
        this.setState({
            isWorking: true,
        }, () => {

            const id: any = this.state.removingResource && this.state.removingResource[0];

            Promise.all(Object.entries(environments).map(entry => {
                return this.request.exec(REQUESTS.REMOVE_LOCALIZATIONS,
                    {
                        body: { id: [decodeURIComponent(id)] },
                        headers: {
                            saasnew: `https://${entry[0] === 'testing' ? 'testing' : 'prestable'}.carsharing.yandex.net`,
                        },
                    });
            }))
                .then(() => {
                    this.getData();
                    this.onClose();
                })
                .catch((removeError) => {
                    this.setState({
                        isWorking: false,
                        removeError,
                    });
                });
        });
    }

    addLocalization() {
        location.hash = NEW_ITEM_URL;
    }

    close() {
        location.hash = NEW_ITEM_BACK_URL;
    }

    upsert(data: any) {
        this.setState({ isWorking: true }, () => {
            Promise.all(Object.entries(data.localizations).map((entry: [string, { [key: string]: ILocalItem[] }]) => {
                const [env, localizations] = entry;
                const body = {
                    resource_id: data.resource_id,
                    localizations: data.duplicate ? data.localizations.production : localizations,
                };

                return this.request.exec(REQUESTS.UPSERT_LOCALIZATIONS, {
                    body,
                    headers: {
                        saasnew: `https://${env === 'testing' ? 'testing' : 'prestable'}.carsharing.yandex.net`,
                    },
                });
            }))
                .then(() => {
                    this.getData();
                    this.setState({
                        isWorking: false,
                        upsertingError: null,
                    });
                    this.close();
                })
                .catch((upsertingError) => {
                    this.setState({
                        upsertingError,
                        isWorking: false,
                    });
                });
        });
    }

    onFilterChange(filter: string) {
        this.setState({ filter });
    }

    render() {
        const LOC_ITEM_MAX_LENGTH = 600;

        return <>
            {
                this.state.dataIsLoading
                && <Spin/>
                || <div className={style.localization_settings}>
                    <Input className={style.filter_input}
                           placeholder={'Фильтр'}
                           value={this.state.filter}
                           onChange={this.onFilterChange.bind(this)}/>
                    <Button className={style.add_resource}
                            onClick={this.addLocalization.bind(this)}>Новый перевод</Button>
                    <div className={style.table_container}>

                        <table className={`${styleTable.table} ${style.local_compare_table}`}>
                            <thead>
                                <tr>
                                    <th>#</th>
                                    <th/>
                                    <th>ID</th>
                                    <th>Текст Testing</th>
                                    <th>Текст Production</th>
                                </tr>
                            </thead>
                            <tbody>
                                {this.state.data && Object.entries(this.state.data)
                                    .filter(item => {
                                        return !this.state.filter
                                        || (this.state.filter
                                            && item[0].toLowerCase().includes(this.state.filter.toLowerCase()));
                                    })
                                    .sort((entry1, entry2) => {
                                        const [key1, values1] = entry1;
                                        const [key2, values2] = entry2;
                                        const isFull1 = values1.testing && values1.production;
                                        const isFull2 = values2.testing && values2.production;

                                        if (isFull1 && isFull2) {
                                            return key1.localeCompare(key2);
                                        }

                                        if (isFull1 && !isFull2) {
                                            return -1;
                                        }

                                        if (!isFull1 && isFull2) {
                                            return 1;
                                        }

                                        return 0;
                                    })
                                    .map((item, index) => {
                                        const [resource_id, values] = item;

                                        return <tr key={index}>
                                            <td>{++index}</td>
                                            <td>
                                                <Link className={style.remove}
                                                      onClick={this.openRemoveModal.bind(this, item)}><Cross/></Link>
                                            </td>
                                            <td>
                                                <Link href={`#${NEW_ITEM_URL}${decodeURIComponent(resource_id)}`}>
                                                    {resource_id}
                                                </Link>
                                            </td>
                                            <td>
                                                {values && values.testing
                                            && values.testing.map((_l, key: number) => {
                                                return <div className={style.loc_item} key={key}>
                                                    <span className={`lang lang-${_l.l}`}>
                                                        {_l.l.toLocaleUpperCase()}
                                                    </span>: <span className={style.loc_item_value}>
                                                        {_l.v.length > LOC_ITEM_MAX_LENGTH
                                                            ? <span>{_l.v.substr(0, LOC_ITEM_MAX_LENGTH)}
                                                                <span> </span>
                                                                <Link href={`#${NEW_ITEM_URL}${resource_id}`}>
                                                        [Ещё...]
                                                                </Link></span>
                                                            : _l.v
                                                        }
                                                    </span>
                                                </div>;
                                            })}
                                            </td>
                                            <td>
                                                {values && values.production
                                            && values.production.map((_l, key: number) => {
                                                return <div className={style.loc_item} key={key}>
                                                    <span className={`lang lang-${_l.l}`}>
                                                        {_l.l.toLocaleUpperCase()}
                                                    </span>: <span className={style.loc_item_value}>
                                                        {_l.v.length > LOC_ITEM_MAX_LENGTH
                                                            ? <span>{_l.v.substr(0, LOC_ITEM_MAX_LENGTH)}
                                                                <span> </span>
                                                                <Link href={`#${NEW_ITEM_URL}${resource_id}`}>
                                                        [Ещё...]
                                                                </Link></span>
                                                            : _l.v
                                                        }
                                                    </span>
                                                </div>;
                                            })}
                                            </td>
                                        </tr>;
                                    })}
                            </tbody>
                        </table>
                    </div>
                    {this.state.removingResource
                        ? <RemoveLocalizationWindow removingResource={this.state.removingResource}
                                                    remove={this.remove.bind(this)}
                                                    onClose={this.closeRemoveModal.bind(this)}/>
                        : null}
                    {this.state.item !== null
                        ? <LocalizationWindowItem {...this.props}
                                                  isWorking={this.state.isWorking}
                                                  schema={ADD_LOCALIZATION_ITEM_SCHEMA}
                                                  upsert={this.upsert.bind(this)}
                                                  upsertingError={this.state.upsertingError}
                                                  onClose={this.close.bind(this)}/>
                        : null}
                </div>
            }
        </>;
    }
}

class RemoveLocalizationWindow extends React.PureComponent<any, any> {
    state = {
        switcherValues: {},
    };

    componentDidMount(): void {
        const values = this.props.removingResource[1];

        const switcherValues = Object.keys(values).reduce((result: Dict<boolean>, env: string) => {
            result[env] = false;

            return result;
        }, {});

        this.setState({ switcherValues });
    }

    onSwitch(type: string, value: boolean) {
        const switcherValues = { ...this.state.switcherValues };
        switcherValues[type] = value;
        this.setState({ switcherValues });
    }

    render() {
        const [resourceId] = this.props.removingResource;

        return <Window title={`Удаление ресурса ${resourceId}`}
                       className={style.remove_window}
                       onClose={this.props.onClose.bind(this)}>
            <h3>Удалить для окружений:</h3>
            <table>
                <tbody>
                    {Object.keys(this.state.switcherValues).map((env: string, index) => {
                        return <tr key={index}>
                            <td>{env}</td>
                            <td>
                                <Checkbox checked={this.state.switcherValues[env]}
                                          onChange={this.onSwitch.bind(this, env)}/>
                            </td>
                        </tr>;
                    })}
                </tbody>
            </table>
            <div className={coreStyle.button_container}>
                <Button basic colorType={ButtonTypes.negative} onClick={this.props.onClose.bind(this)}>Отмена</Button>
                <Button onClick={this.props.remove.bind(this, this.state.switcherValues)}>Удалить</Button>
            </div>
        </Window>;
    }
}

interface ILocalizationWindowItemProps {
    location: any;
    schema: { [key: string]: any };
    onClose: () => void;
    upsert: () => void;
    isWorking: boolean;
    upsertingError: null | Error;
}

interface ILocalizationWindowItemState {
    allResources: Dict<{ testing: ILocalItem[]; production: ILocalItem[] }>;
    currentResource: { testing: ILocalItem[]; production: ILocalItem[] } | null;
    isLoading: boolean;
    values: {};
    warning: string;
}

class LocalizationWindowItem extends React.PureComponent<ILocalizationWindowItemProps, ILocalizationWindowItemState> {
    state: ILocalizationWindowItemState = {
        allResources: {},
        currentResource: null,
        isLoading: false,
        values: {},
        warning: '',
    };
    request = new Request2({
        requestConfigs,
    });

    getResource() {
        const params = new URLSearchParams(this.props.location.search);
        const item: string | null = params.get(ITEM_URL_PARAM);

        this.setState({ isLoading: true }, () => {

            Promise.all([this.request.exec(REQUESTS.GET_LOCALIZATIONS, {
                queryParams: { id: '' }, headers: {
                    'saasnew': 'https://testing.carsharing.yandex.net',
                },
            }), this.request.exec(REQUESTS.GET_LOCALIZATIONS, {
                queryParams: { id: '' }, headers: {
                    'saasnew': 'https://prestable.carsharing.yandex.net',
                },
            })]).then(response => {
                let [testingArr = {}, productionArr = {}] = response;
                testingArr = testingArr.resources || [];
                productionArr = productionArr.resources || [];

                const allResources = {};

                testingArr.reduce((result: { testing: Dict<ILocalItem[]>; production: Dict<ILocalItem[]> },
                    localization: Localization) => {
                    if (result[localization.resource_id]) {
                        result[localization.resource_id].testing = localization.localizations;
                    } else {
                        result[localization.resource_id] = { testing: localization.localizations };
                    }

                    return result;
                }, allResources);

                productionArr.reduce((result: { testing: Dict<ILocalItem[]>; production: Dict<ILocalItem[]> },
                    localization: Localization) => {
                    if (result[localization.resource_id]) {
                        result[localization.resource_id].production = localization.localizations;
                    } else {
                        result[localization.resource_id] = { production: localization.localizations };
                    }

                    return result;
                }, allResources);

                if (item) {
                    const currentResource = allResources[item];
                    this.setState({ isLoading: false, allResources, currentResource });
                } else {
                    this.setState({ isLoading: false, allResources });
                }
            });

        });
    }

    UNSAFE_componentWillReceiveProps(nextProps: Readonly<ILocalizationWindowItemProps>): void {
        if ((this.props.location
            && this.props.location.pathname
            && nextProps.location
            && nextProps.location.pathname)
            && (this.props.location.pathname !== nextProps.location.pathname)) {
            this.getResource();
        }
    }

    componentDidMount(): void {
        this.getResource();
    }

    onChangeForm(values: { [key: string]: any }) {
        const params = new URLSearchParams(this.props.location.search);
        const item: string | null = params.get(ITEM_URL_PARAM);
        let warning = '';

        const { resource_id: resourceId } = values;
        if (resourceId) {
            const allResources = this.state.allResources;

            const isExist = Object.keys(allResources).some(id => id === resourceId);

            if (isExist && (!item || item !== resourceId)) {
                warning = `Ресурс ${resourceId} уже существует, сохранение обновит его данные`;
            }
        }

        this.setState({ values, warning });
    }

    render() {
        const currentResource = this.state.currentResource;
        const params = new URLSearchParams(this.props.location.search);
        const item: string | null = params.get(ITEM_URL_PARAM);

        return <Window title={'Localization'}
                       className={style.localization_window}
                       error={this.props.upsertingError}
                       onClose={this.props.onClose.bind(this)}>
            <div className={style.add_localization}>
                {this.state.isLoading
                    ? <Spin/>
                    : <FormConstructor initialData={
                        currentResource && item
                            ? {
                                resource_id: item,
                                localizations: {
                                    testing: currentResource.testing,
                                    production: currentResource.production,
                                },
                            }
                            : {
                                localizations: {
                                    testing: [{ l: 'rus', v: 'русский текст' }, {
                                        l: 'eng',
                                        v: 'english text',
                                    }],
                                    production: [{ l: 'rus', v: 'русский текст' },
                                        {
                                            l: 'eng',
                                            v: 'english text',
                                        }],
                                },
                                resource_id: '',
                            }
                    }
                                       schema={this.props.schema}
                                       onChange={this.onChangeForm.bind(this)}/>
                }
                {this.state.warning
                    ? <h3 className={style.warning}>{this.state.warning}</h3>
                    : null}
                <div className={style.controls}>
                    <Button colorType={ButtonTypes.negative}
                            basic
                            onClick={this.props.onClose.bind(this)}>
                        Отмена
                    </Button>
                    <Button isLoading={this.props.isWorking}
                            onClick={this.props.upsert.bind(this, this.state.values)}>
                        Отправить
                    </Button>
                </div>
            </div>
        </Window>;
    }
}
