import React from 'react';
import ReactTooltip from 'react-tooltip';

import { EMPTY_DATA, ONE_DAY, ONE_HUNDRED_PERCENT, ONE_SECOND } from '../../constants';
import SessionsHistoryItem, { SessionHistoryInfoHandler } from '../../models/session';
import FormatDate from '../../ui/FormatDate';
import { Window } from '../../ui/FullModal';
import { rubs } from '../../utils/rubs';
import style from './index.css';

interface IUnplannedFinishDialog {
    session: SessionsHistoryItem;
    onClose: () => void;
}

export const UnplannedFinishDialog = React.memo((props: IUnplannedFinishDialog) => {
    const { session, onClose }: any = props;
    const offer = SessionHistoryInfoHandler.getOffer.call(session) || {};
    const state = SessionHistoryInfoHandler.getSessionState.call(session) || {};

    const includedTimeInSec = (offer.LongTermOffer?.Until - offer.LongTermOffer?.Since) / ONE_SECOND;
    const factTimeInSec = +new Date() - state.since * ONE_SECOND;
    const plannedDaysNum = Math.abs(includedTimeInSec) / ONE_DAY;
    const factDaysNum = Math.floor(Math.abs(factTimeInSec) / ONE_DAY);
    const completedPayments = getCompletedPayments(state.payments);

    const childProps = {
        offer,
        state,
        plannedDaysNum,
        factDaysNum,
        completedPayments,
    };

    return <Window title={'Расчёт досрочного завершения аренды'} onClose={onClose.bind(null)}>
        {RentPeriodBlock(childProps)}

        {RentDetailsBlock(childProps)}

        {SummaryTable(childProps)}
    </Window>;
});

const RentPeriodBlock = (props) => {
    const { state } = props;

    const sinceDate = <FormatDate value={state.since * ONE_SECOND}/>;
    const untilDate = <FormatDate value={state.until * ONE_SECOND}/>;
    const nowDate = <FormatDate value={new Date()}/>;

    return <div className={style.rent_block}>
        <div className={style.title}>
            Срок аренды
        </div>
        <div>
            <div>План: {sinceDate} - {untilDate}</div>
            <div>Факт: {sinceDate} - {nowDate}</div>
        </div>
    </div>;
};

const RentDetailsBlock = (props) => {
    const { offer, state, plannedDaysNum, factDaysNum, completedPayments } = props;
    const ONE_MINUTE_IN_SECONDS = 60;
    const DIGITS_IN_FRACTION = 1;

    const plannedMileage = (offer.PackOffer?.MileageLimit ?? 0) + (offer.PackOffer?.ExtraMileageLimit ?? 0);
    const spentMileage = state.remaining_distance
        ? Math.round(plannedMileage - state.remaining_distance)
        : Math.round(plannedMileage);
    const spentDays = factDaysNum <= plannedDaysNum ? factDaysNum : plannedDaysNum;

    const plannedPrice = state.pack_price;
    const factPrice = state.return_cost;

    const overrun = state.overrun_distance;
    const overtime = state.overrun_time && (state.overrun_time / ONE_MINUTE_IN_SECONDS).toFixed(DIGITS_IN_FRACTION);
    const daysPercent = Math.round((spentDays / plannedDaysNum) * ONE_HUNDRED_PERCENT);
    const kmPercent = Math.round((spentMileage / plannedMileage) * ONE_HUNDRED_PERCENT);

    return <div>
        <table className={style.table}>
            <tbody>
                <tr>
                    <td>
                        <div className={style.title}>
                            Текущее использование аренды
                        </div>
                    </td>
                </tr>
                <tr>
                    <td>Дней</td>
                    <td>{`${spentDays}дн / ${plannedDaysNum}дн = ${daysPercent}%`}</td>
                </tr>
                <tr>
                    <td>Пробег</td>
                    <td>{`${spentMileage}км / ${plannedMileage}км = ${kmPercent}%`}</td>
                </tr>
                <tr>
                    <td>Перепробег</td>
                    <td>{overrun} км</td>
                </tr>
                <tr>
                    <td>Перерасход</td>
                    <td>{overtime} мин</td>
                </tr>

                <tr>
                    <td>
                        <div className={style.title}>
                            Стоимость аренды
                        </div>
                    </td>
                </tr>
                <tr>
                    <td>Первоначальная стоимость</td>
                    <td>{rubs(plannedPrice)}</td>
                </tr>
                <tr>
                    <td>Стоимость при завершении сейчас</td>
                    <td>{rubs(factPrice)}</td>
                </tr>
                <tr>
                    <td>Оплачено</td>
                    <td>{rubs(completedPayments)}</td>
                </tr>
            </tbody>
        </table>
    </div>;
};

const SummaryTable = (props) => {
    const { state, completedPayments } = props;
    const SummaryTooltipId = 'SUMMARY_TABLE_TOOLTIP';

    const overrunPrice = state.return_overrun_cost;
    const overtimePrice = state.overtime_price;
    const returnFine = state.return_fee;
    const priceForToday = state.return_pack_price;
    const factPrice = state.return_cost + (overtimePrice || 0);
    const payExtra = factPrice - completedPayments;

    return <div className={style.summary_block}>
        <div className={style.title}>
            Пересчет
        </div>
        <div>
            <table className={style.summary_table}>
                <tbody>
                    <tr className={style.table_title}>
                        <td data-for={SummaryTooltipId}
                            data-tip={matchTooltipData(SUMMARY_VALUES.rent)}>
                        Аренда
                        </td>
                        <td data-for={SummaryTooltipId}
                            data-tip={matchTooltipData(SUMMARY_VALUES.overrun)}>
                        Перепробег ₽
                        </td>
                        <td data-for={SummaryTooltipId}
                            data-tip={matchTooltipData(SUMMARY_VALUES.overtime)}>
                        Перерасход ₽
                        </td>
                        <td data-for={SummaryTooltipId}
                            data-tip={matchTooltipData(SUMMARY_VALUES.fine)}>
                        Штраф за возврат
                        </td>
                        <td data-for={SummaryTooltipId}
                            data-tip={matchTooltipData(SUMMARY_VALUES.total)}
                            className={style.total}>
                        Итого
                        </td>
                        <td data-for={SummaryTooltipId}
                            data-tip={matchTooltipData(SUMMARY_VALUES.extraPay)}
                            className={style.final}>
                        Доплатить
                        </td>
                    </tr>
                    <tr>
                        <td>{priceForToday ? rubs(priceForToday) : EMPTY_DATA}</td>
                        <td>{overrunPrice ? rubs(overrunPrice) : EMPTY_DATA}</td>
                        <td>{overtimePrice ? rubs(overtimePrice) : EMPTY_DATA}</td>
                        <td>{returnFine ? rubs(returnFine) : EMPTY_DATA}</td>
                        <td className={style.total}>{factPrice ? rubs(factPrice) : EMPTY_DATA}</td>
                        <td className={style.final}>{payExtra ? rubs(payExtra) : EMPTY_DATA}</td>
                    </tr>
                </tbody>
            </table>
        </div>

        <ReactTooltip id={SummaryTooltipId} html={true}/>
    </div> || null;
};

const getCompletedPayments = function (payments) {
    let finalSum = 0;

    payments?.forEach(payment => {
        if (payment.completed) {
            finalSum += payment.value;
        }
    });

    return finalSum;
};

enum SUMMARY_VALUES {
    rent = 'rent',
    overrun = 'overrun',
    overtime = 'overtime',
    fine = 'fine',
    total = 'total',
    extraPay = 'extraPay',
}

const matchTooltipData = (type: SUMMARY_VALUES) => {
    switch (type) {
    case SUMMARY_VALUES.rent:
        return `Первоначальная стоимость * большее из "Дней" и "Пробег"`;
    case SUMMARY_VALUES.overrun:
        return `"Перепробег" * стоимость 1 км перепробега`;
    case SUMMARY_VALUES.overtime:
        return `"Перерасход" * стоимость 1 мин перерасхода`;
    case SUMMARY_VALUES.fine:
        return `Штраф за досрочный возврат`;
    case SUMMARY_VALUES.total:
        return `Аренда + Перепробег + Перерасход + Штраф`;
    case SUMMARY_VALUES.extraPay:
        return `Сумма, которую нужно оплатить с учетом осуществленных платежей`;
    default:
        return ``;
    }
};
