import * as React from 'react';
import XLSX from 'xlsx';

import { Button } from '../../../ui/Button';
import * as tblStyle from '../../../ui/Table/index.css';
import { Tabs } from '../../../ui/Tabs';
import { Request2 } from '../../../utils/request';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { MAJOR_REQUESTS, REQUESTS } from '../request';
import * as style from './index.css';

interface IMajorStorageProps {
}

interface IMajorStorageState {
    isLoading: boolean;
    error: Error | null;
    selectedTab: string;
    data: any;
    selectedSecondaryTab: string;
}

const DEFAULT_SECONDARY_TAB = 'rims';

export default class MajorStorage extends React.Component<IMajorStorageProps, IMajorStorageState> {
    state: IMajorStorageState = {
        isLoading: false,
        error: null,
        selectedTab: 'Service',
        data: {},
        selectedSecondaryTab: DEFAULT_SECONDARY_TAB,
    };

    request = new Request2({ requestConfigs: MAJOR_REQUESTS });
    tableContainer = React.createRef<HTMLDivElement>();

    componentDidMount(): void {
        this.getData();
    }

    format(data) {
        const _obj = Object.entries(data).reduce((_p: any, _c: any) => {
            const _item = Object.keys(_c[1]);
            _item.forEach(element => {
                _c[1][element] && _c[1][element].map(el => {
                    const id = el.type_info.id;
                    const model = el.type_info.model;
                    const brand = el.type_info.brand;
                    const season = el.type_info.season;
                    const statuses = el.statuses && el.statuses.map(s => ({
                        ...s,
                        area: _c[0],
                        id,
                        ...model && { model },
                        ...brand && { brand },
                        ...season && { season },
                    })) || [];

                    if (!_p[element].hasOwnProperty(id)) {
                        _p[element][id] = {
                            ...el, statuses,
                        };
                    } else {
                        _p[element][id].statuses.push(...statuses);
                    }
                });
            });

            return _p;
        }, { rims: {}, tyres: {} });

        return _obj;
    }

    getData() {
        this.setState({
            isLoading: true,
            error: null,
        }, () => {

            this.request.exec(REQUESTS.STORAGE, {
                queryParams: {
                    regions: 'MSK,KZN,SPB',
                    storage: this.state.selectedTab,
                },
            }).then(response => {
                this.setState({
                    isLoading: false,
                    error: null,
                    data: this.format(response),
                });
            })
                .catch(error => {
                    this.setState({
                        isLoading: false,
                        error,
                    });
                });
        });
    }

    selectTab(selectedTab) {
        this.setState({
            selectedTab,
        }, () => {
            this.selectSecondaryTab(DEFAULT_SECONDARY_TAB);
            this.getData();
        });
    }

    selectSecondaryTab(selectedSecondaryTab) {
        this.setState({
            selectedSecondaryTab,
        });

        if (this.tableContainer?.current?.scrollTop) {
            this.tableContainer.current.scrollTop = 0;
        }
    }

    addWs(workbook, type) {
        const _type = Object.entries(this.state.data[type]);

        const typeWs = XLSX.utils.json_to_sheet(_type.map((el: any) => el[1].type_info));
        const typeStatusWs = XLSX.utils.json_to_sheet(_type.reduce((_p, _c: any) => {
            return [..._p, ..._c[1].statuses];
        }, []));

        XLSX.utils.book_append_sheet(workbook, typeWs, `${type}`);
        XLSX.utils.book_append_sheet(workbook, typeStatusWs, `${type}Statuses`);
    }

    save() {
        const workbook = XLSX.utils.book_new();
        this.addWs(workbook, 'rims');
        this.addWs(workbook, 'tyres');

        XLSX.writeFile(workbook, `${this.state.selectedTab}.xlsx`);
    }

    render() {
        const tabs = [{ name: 'Ремфонд', link: 'Service' }, { name: ' Сезонный ШМ', link: 'Season' }];
        const secondaryTabs = [{ name: 'Диски', link: 'rims' }, { name: 'Шины', link: 'tyres' }];

        return <div>
            <Tabs currentTab={this.state.selectedTab}
                  selectTab={this.selectTab.bind(this)}
                  tabs={tabs}/>
            {
                this.state.isLoading
                    ? <Spin/>
                    : this.state.error
                        ? <SimpleError error={this.state.error}/>
                        : <div>
                            <Button onClick={this.save.bind(this)}
                                    className={style.download_button}>Скачать {this.state.selectedTab}.xlsx</Button>
                            <Tabs currentTab={this.state.selectedSecondaryTab}
                                  selectTab={this.selectSecondaryTab.bind(this)}
                                  tabs={secondaryTabs}/>
                            <div className={style.table_container}
                                 ref={this.tableContainer}>
                                {this.state.selectedSecondaryTab === 'rims'
                                    ? <RimsTable data={this.state.data.rims}/>
                                    : <TyresTable data={this.state.data.tyres}/>
                                }
                            </div>
                        </div>
            }
        </div>;
    }
}

const RimsTable = (props: { data: any }) => {

    const data = props.data && Object.entries(props.data) || [];

    return <table className={tblStyle.table}>
        <thead>
            <tr className={'header'}>
                <th>#</th>
                <th>id</th>
                <th>type</th>
                <th>count</th>
                <th>model</th>
                <th>diameter</th>
                <th/>
            </tr>
        </thead>
        <tbody>
            {
                data.map((el, index) => {
                    return <RimTr key={el[0]} data={el[1]} index={index}/>;
                })
            }
        </tbody>

    </table>;
};

const Statuses = (data: any) => {
    const statuses: any = [];
    const count = data && data.statuses
        .reduce((_p, _c, _i) => {
            statuses.push(<tr key={_i + '_' + [data[0]]} className={style.statuses}>
                <td colSpan={2}/>
                <td>{_c.area}</td>
                <td>{_c.count}</td>
                <td colSpan={3}>{_c.status}</td>
            </tr>);

            return _p + _c.count;
        }, 0);

    return [statuses, count];
};

const RimTr = (props: { data: any; index }) => {
    const { data, index } = props;
    const [statuses, count] = Statuses(data);

    const [isExpanded, toggle] = React.useState(false);
    const onClick = () => {
        toggle(!isExpanded);
    };

    const tr = <>
        <tr className={`${style.type_info} ${isExpanded ? style.expanded : ''}`} onClick={onClick}>
            <td>{index + 1}</td>
            <td>{data.type_info.id}</td>
            <td>{data.type_info.type}</td>
            <td><strong>{count}</strong></td>
            <td>{data.type_info.model}</td>
            <td colSpan={2}>{data.type_info.diameter}</td>
        </tr>
        {
            isExpanded ? statuses : null
        }
    </>;

    return tr;
};

const TyreTr = (props: { data: any; index }) => {
    const { data, index } = props;
    const [statuses, count] = Statuses(data);

    const [isExpanded, toggle] = React.useState(false);
    const onClick = () => {
        toggle(!isExpanded);
    };

    const tr = <>
        <tr className={`${style.type_info} ${isExpanded ? style.expanded : ''}`} onClick={onClick}>
            <td>{index + 1}</td>
            <td>{data.type_info.id}</td>
            <td>{data.type_info.brand}</td>
            <td><strong>{count}</strong></td>
            <td>{data.type_info.model}</td>
            <td>{data.type_info.size}</td>
            <td>{data.type_info.season}</td>
        </tr>
        {
            isExpanded ? statuses : null
        }
    </>;

    return tr;
};

const TyresTable = (props: { data: any }) => {

    const data = props.data && Object.entries(props.data) || [];

    return <table className={tblStyle.table}>
        <thead>
            <tr className={'header'}>
                <th>#</th>
                <th>id</th>
                <th>brand</th>
                <th>count</th>
                <th>model</th>
                <th>size</th>
                <th>season</th>
            </tr>
        </thead>
        <tbody>
            {
                data.map((el, index) => {
                    return <TyreTr key={el[0]} data={el[1]} index={index}/>;
                })
            }
        </tbody>

    </table>;
};
