import produce from 'immer';
import * as React from 'react';
import VirtualList from 'react-tiny-virtual-list';

import { ONE_SECOND } from '../../../constants';
import { useWindowResize } from '../../../hooks/useWindowResize';
import FormatDate from '../../../ui/FormatDate';
import { LabelStatus, TLabel } from '../../../ui/Table';
import { Request2 } from '../../../utils/request';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { MAJOR_REQUESTS, REQUESTS } from '../request';
import * as style from './index.css';

interface IProps {
}

interface IState {
    isLoading: boolean;
    data: any;
    error: any;
}

export class Service extends React.Component<IProps, IState> {
    state = {
        isLoading: false,
        data: {} as any,
        error: null,
    };
    request = new Request2({ requestConfigs: MAJOR_REQUESTS });

    getData() {
        this.setState(produce(this.state, draft => {
            draft.isLoading = true;
            draft.error = null;
        }), () => {
            this.request.exec(REQUESTS.SERVICE)
                .then(response => {
                    this.setState(produce(this.state, draft => {
                        draft.isLoading = false;
                        draft.data = response.queries || [];
                    }));
                })
                .catch(error => {
                    this.setState(produce(this.state, draft => {
                        draft.isLoading = false;
                        draft.error = error;
                    }));
                });
        });
    }

    componentDidMount(): void {
        this.getData();
    }

    render() {
        return <div>
            {
                this.state.isLoading
                    ? <Spin/>
                    : this.state.error
                        ? <SimpleError error={this.state.error}/>
                        : this.state.data && Array.isArray(this.state.data) && this.state.data.length
                            ? <MajorTable data={this.state.data} />
                            : <div>Нет данных</div>
            }
        </div>;
    }
}

interface IMajorTableProps {
    data: any;
}

const LIST_ITEM_SIZE = 45;
const TABLE_PADDING = 200;

const MajorTable = React.memo((props: IMajorTableProps) => {
    const { data } = props;
    const windowSize = useWindowResize();

    const listHeight = windowSize.height - TABLE_PADDING;

    return <div className={style.major_to_list}>
        <div className={style.header}>
            <span className={style.cell}>#</span>
            <span className={style.cell}>create_time</span>
            <span className={style.cell}>state</span>
            <span className={style.cell}>service_name/type</span>
            <span className={style.cell}>vin/work</span>
        </div>
        <VirtualList width={'100%'}
                     height={listHeight}
                     itemCount={data.length}
                     itemSize={LIST_ITEM_SIZE}
                     renderItem={({ index, style }) => {
                         return <MajorTableItem key={data[index].id}
                                                index={index}
                                                _style={style}
                                                item={data[index]}/>;
                     }
                     }/>
    </div>;
});

export class MajorTableItem extends React.Component<any, any> {
    getType(item) {
        switch (item.work) {
        case 'maintenance':
            return LabelStatus.POSITIVE;
        case 'diagnostics':
            return LabelStatus.WARNING;
        case 'body_repair':
            return LabelStatus.NEGATIVE;
        case 'mechanic_repair':
            return LabelStatus.POSITIVE;
        default:
            return LabelStatus.DEFAULT;
        }
    }

    render() {
        const { index, item } = this.props;

        return <div style={this.props._style} className={style.row} key={index}>
            <span className={style.cell}>{index + 1}</span>
            <span className={style.cell}>
                <FormatDate value={item.create_time * ONE_SECOND} withSecond/>
            </span>
            <span className={style.cell}>
                <div className={style.state}>{item.state}</div>
                <div className={style.state_details}>{item.state_details}</div>
            </span>
            <span className={style.cell}>
                <div>{item.service_name}</div>
                <div className={style.state_details}>{item.type}</div>
            </span>
            <span className={style.cell}>
                <div>{item.vin}</div>
                <div>
                    <TLabel status={this.getType(item)} text={item.work}/>
                </div>
            </span>
        </div>;
    }
}
