import React from 'react';

import { Dict } from '../../../../types';
import { CAR_COLORS_RGB, EMPTY_DATA, FREE_DELEGATION, ONE_SECOND } from '../../../constants';
import CarInfo, { CAR_STATUS, CarInfoHandler } from '../../../models/car';
import { SessionHistoryInfoHandler } from '../../../models/session';
import { DelegationIcon } from '../../../ui/DelegationIcon';
import FormatDate, { FormatDateInString } from '../../../ui/FormatDate';
import { CarIndicatorsBlock } from '../../../utils/carIndicators';
import { Request2 } from '../../../utils/request';
import CarNumber from '../../CarNumber';
import style from '../../MainMap/MapCanvas/index.css';
import { MAP_REQUESTS as requestConfigs, REQUESTS } from '../../MainMap/request';

interface ICarTooltipProps {
    car: Dict<any>;
    modelsData: Dict<any>;
}

interface ICarTooltipState {
    isLoading: boolean;
    car: Dict<any> | null;
    notZero: string;
    tags: string;
}

export class CarTooltip extends React.Component<ICarTooltipProps, ICarTooltipState> {
    state: ICarTooltipState = {
        isLoading: true,
        car: null,
        notZero: '...',
        tags: '...',
    };
    request = new Request2({ requestConfigs });

    componentDidMount(): void {
        this.getData();
    }

    componentDidUpdate(prevProps: Readonly<ICarTooltipProps>): void {
        if (prevProps.car.id !== this.props.car.id
            || JSON.stringify(prevProps.modelsData) !== JSON.stringify(this.props.modelsData)) {
            this.getData();
        }
    }

    getData() {
        this.setState({ isLoading: true }, () => {
            const car = this.props.car;
            const { id } = car;

            this.request.exec(REQUESTS.GET_CAR_INFO, { queryParams: { car_id: id } })
                .then(response => {
                    this.setState({ car: response, isLoading: false }, () => {
                        if (CarInfoHandler.getStatus.call(this.state.car) === CAR_STATUS.service) {
                            let tags = this.state.car?.tags?.reduce((_p: string, _c: any) => {
                                return _c.priority > 0
                                    ? _p += `${_c.priority}: ${_c.display_name || _c.tag}`
                                        + `${_c?.performer_info?.name
                                            ? ` (${_c?.performer_info.name})`
                                            : ''}; `
                                    : _p;
                            }, '') ?? '';

                            if (!tags.length) {
                                tags = EMPTY_DATA;
                            }

                            this.setState({
                                tags,
                            });
                        } else {
                            this.setState({
                                tags: EMPTY_DATA,
                            });
                        }

                        this.request.exec(REQUESTS.GET_SESSIONS, {
                            queryParams: {
                                car_id: id,
                                numdoc: 10,
                            },
                        })
                            .then((response) => {
                                const finish = SessionHistoryInfoHandler.getFinish.call(response);
                                const notZero = finish ? FormatDateInString({ value: finish }) : EMPTY_DATA;

                                this.setState({
                                    notZero,
                                });
                            });
                    });
                })
                .catch(() => {
                    this.setState({ isLoading: false });
                });
        });
    }

    render() {
        const { car, modelsData } = this.props;

        const modelName = modelsData && modelsData[car.model_id]
            ? modelsData[car.model_id].name
            : car.model_id;

        const color = CAR_COLORS_RGB[car.status] || CAR_COLORS_RGB.default;
        const isDelegated = this.state.car?.tags?.find(el => el.tag === FREE_DELEGATION) || false;

        const img = this.state?.car?.views?.[0]?.image_angle_url
            || this.state?.car?.views?.[0]?.image_small_url
            || modelsData[car.model_id]?.image_angle_url
            || modelsData[car.model_id]?.image_small_url;

        return <div className={`${style.car_tooltip} ${this.state.isLoading ? style.loading_tooltip : ''}`}>
            <div className={style.car_tooltip_mainInfo}>
                <CarNumber carInfo={car as typeof CarInfo}/>
                <div className={style.car_tariff}>{car.offer_name && car.offer_name || EMPTY_DATA}</div>
            </div>
            <div className={style.car_tooltip_image_container}>
                <div className={style.car_tooltip_image}>
                    <div className={style.car_tooltip_image_ring} style={{ color: `#${color}` }}/>
                    {
                        this.state.car && img && <img src={img} alt={'Car image'}/>
                    }
                </div>
            </div>
            <div className={style.car_tooltip_model_name}>
                {modelName}
                {isDelegated && <DelegationIcon/>}
            </div>
            <div className={style.car_tooltip_other}>
                Статус: {car.status} / {car.usage}
            </div>
            <div className={style.car_tooltip_other}>
                Последняя: {car.surge?.idle_start ?
                    <FormatDate value={car.surge.idle_start * ONE_SECOND}/> : EMPTY_DATA}
            </div>
            <div className={style.car_tooltip_other}>
                Не нулевая: {this.state.notZero}
            </div>
            <div className={style.car_tooltip_other}>
                Теги: {this.state.tags}
            </div>

            <CarIndicatorsBlock carInfo={this.state.car ?? {}}
                                wrapperStyle={style.car_tooltip_telematics}
                                iconStyle={style.telematics_icon}
                                itemStyle={style.telematics_info_item} />
        </div>;
    }
}
