import * as React from 'react';

import { LSSettingItems } from '../../../../types';
import { Button } from '../../../ui/Button';
import Checkbox from '../../../ui/Checkbox';
import { Link } from '../../../ui/Link';
import { IBlockRules } from '../../../utils/IBlockRules';
import LS from '../../../utils/localStorage/localStorage';
import * as styles from './index.css';
import { getOfferId } from './utils';

export type LegendGroup = {
    name: string;
    id: string;
    cars_count: number;
    group_name?: string;
    isGroupOffer?: boolean;
}[]

interface IMapLegendProps extends IBlockRules {
    legend: {
        offer_name: LegendGroup;
        model_id: LegendGroup;
        status: LegendGroup;
        idle_start: LegendGroup;
    };
    idleOn: boolean;
    filterStatus: (status: string) => void;
    filterModel: (model: string) => void;
    filterOffer: (offer: string) => void;
    filterIdle: (idle: string) => void;
    clearClick: () => void;
    checkIdle: () => void;
    tags_filter: string;
}

interface IMapLegendState {
    selectedStatus: string;
    selectedModel: string;
    selectedOffer: string;
    selectedIdle: string;
}

const OTHER = 'Другое';

export class MapLegend extends React.Component<IMapLegendProps, IMapLegendState> {
    state: IMapLegendState = {
        selectedStatus: '',
        selectedModel: '',
        selectedOffer: '',
        selectedIdle: '',
    };

    ls = new LS();

    componentDidUpdate(prevProps) {
        if (prevProps.legend !== this.props.legend) {
            this.setFilter();
        }
    }

    setFilter() {
        const currentFilters = this.ls.get(LSSettingItems.map_settings).filters;

        this.setState({
            selectedModel: currentFilters?.model_id ?? '',
            selectedStatus: currentFilters?.status ?? '',
            selectedOffer: currentFilters?.offer_name ?? '',
            selectedIdle: currentFilters?.idle_start?.[1]?.toString() ?? '',
        });
    }

    selectFilter(filterFunction, forceClearTags) {
        filterFunction();
        forceClearTags && this.props.clearClick();
        this.setFilter();
    }

    clearAllFilters() {
        const { filterStatus, filterModel, filterOffer, filterIdle } = this.props;

        filterOffer('');
        filterIdle('');
        filterModel('');
        filterStatus('');

        this.setFilter();
    }

    render() {
        const {
            legend,
            filterStatus,
            filterModel,
            filterOffer,
            filterIdle,
            tags_filter,
            idleOn,
            BlockRules,
        } = this.props;
        const { selectedStatus, selectedModel, selectedOffer, selectedIdle } = this.state;
        let offerCount = 0;
        const legendOffer = legend?.offer_name?.reduce((result, offer) => {
            if (offer.group_name) {
                if (!result?.[offer?.group_name]) {
                    result[offer?.group_name] = [];
                }

                result[offer?.group_name].push(offer);
            } else {
                if (!result?.[OTHER]) {
                    result[OTHER] = [];
                }

                result[OTHER].push(offer);
            }

            offerCount += offer?.cars_count;

            return result;
        }, {}) ?? {};
        const tags_filter_array = tags_filter?.split(',') || [];
        let tags_filter_title = tags_filter;
        const maxCount = 5;
        if (tags_filter_array.length > maxCount) {
            tags_filter_title = `${tags_filter_array.slice(0, maxCount).join(', ')} `
                + `... и ещё ${tags_filter_array.length - maxCount}`;
        }

        const carsCount = legend?.status
            ?.reduce((result, content) => result += content?.cars_count, 0);
        const idlesCount = legend?.idle_start
            ?.reduce((result, content) => result += content?.cars_count, 0);

        const showClearAllButton = !!(selectedIdle.length
            || selectedModel.length
            || selectedOffer.length
            || selectedStatus.length);

        return (
            <>
                <div className={`${styles.legend} ${styles.legend_left}`}>
                    {BlockRules?.FilterTariff ? <div>
                        <table>
                            <tbody>
                                <MapLegendTr selected={selectedOffer}
                                             onClick={this.selectFilter
                                                 .bind(this, filterOffer.bind(this, ''), !offerCount)}
                                             content={null}
                                             carsCount={offerCount}/>
                                {Object.keys(legendOffer)?.reverse()?.map((group, index) => {
                                    const groupOfferContent = {
                                        name: group,
                                        id: group === OTHER ? undefined : group,
                                        isGroup: true,
                                        cars_count: legendOffer[group]
                                            .reduce((result, offer) => result += offer.cars_count, 0),
                                    };

                                    return <React.Fragment key={`offer_group-${index}`}>
                                        {groupOfferContent?.cars_count ?
                                            <MapLegendTr isOffer
                                                         selected={selectedOffer}
                                                         onClick={this.selectFilter
                                                             .bind(this, filterOffer.bind(this, groupOfferContent))}
                                                         content={groupOfferContent}
                                                         key={`offer-${index}`}/>
                                            : null}
                                        {legendOffer[group]
                                            ?.filter((content) => content?.cars_count !== 0)
                                            ?.map((content, index) => {
                                                return <MapLegendTr isOffer
                                                                    selected={selectedOffer}
                                                                    onClick={this.selectFilter.bind(this,
                                                                        filterOffer.bind(this, getOfferId(content)))}
                                                                    content={content}
                                                                    key={`offer-${index}`}/>;
                                            })
                                        }
                                    </React.Fragment>;
                                })}
                            </tbody>
                        </table>
                    </div> : null}
                    {legend?.['idle_start'].length && BlockRules?.FilterIdle
                        ? <>
                            <div className={styles.legend_extra}>
                                idle_start
                                <Checkbox checked={idleOn}
                                          onChange={this.props.checkIdle.bind(this)}/>
                            </div>
                            <div>
                                <table>
                                    <tbody>
                                        <MapLegendTr selected={selectedIdle}
                                                     onClick={this.selectFilter.bind(this, filterIdle.bind(this, ''))}
                                                     content={null}
                                                     carsCount={idlesCount}/>
                                        {legend?.['idle_start']
                                            ?.filter((content) => content?.cars_count !== 0)
                                            ?.map((content, index) => {
                                                return <MapLegendTr selected={selectedIdle}
                                                                    onClick={this.selectFilter
                                                                        .bind(this, filterIdle.bind(this, content?.id))}
                                                                    content={content}
                                                                    key={`idle-${index}`}
                                                                    isIdle/>;
                                            },
                                            )}
                                    </tbody>
                                </table>
                            </div>
                        </>
                        : null
                    }
                </div>
                <div className={`${styles.legend} ${styles.legend_right}`}>
                    {showClearAllButton && (
                        <div className={styles.clear_button}>
                            <Button basic
                                    onClick={this.clearAllFilters.bind(this)}>Сбросить все фильтры</Button>
                        </div>
                    )}
                    {tags_filter && (
                        <div className={styles.legend_extra}>
                            <span>
                                <b>Теги:</b> {tags_filter_title}
                            </span>
                            <Link className={styles.link}
                                  onClick={this.props.clearClick.bind(this)}>
                                очистить
                            </Link>
                        </div>
                    )}
                    {BlockRules?.FilterStatus && legend?.['status'].length ? <div>
                        <table>
                            <tbody>
                                <MapLegendTr selected={selectedStatus}
                                             onClick={this.selectFilter.bind(this, filterStatus.bind(this, ''))}
                                             content={null}
                                             carsCount={carsCount}/>
                                {legend?.['status']
                                    ?.filter((content) => content?.cars_count !== 0)
                                    ?.map((content, index) => {
                                        return <MapLegendTr selected={selectedStatus}
                                                            isStatus
                                                            onClick={this.selectFilter
                                                                .bind(this, filterStatus.bind(this, content?.id))}
                                                            content={content}
                                                            key={`status-${index}`}/>;
                                    })}
                            </tbody>
                        </table>
                    </div> : null}
                    {BlockRules?.FilterModel && legend?.['model_id'].length ? <div>
                        <table>
                            <tbody>
                                <MapLegendTr selected={selectedModel}
                                             onClick={this.selectFilter.bind(this, filterModel.bind(this, ''))}
                                             content={null}
                                             carsCount={carsCount}/>
                                {legend?.['model_id']
                                    ?.map((content, index) => {
                                        return <MapLegendTr selected={selectedModel}
                                                            onClick={this.selectFilter
                                                                .bind(this, filterModel.bind(this, content?.id))}
                                                            content={content}
                                                            key={`model-${index}`}/>;
                                    })}
                            </tbody>
                        </table>
                    </div> : null}
                </div>
            </>
        );
    }
}

interface IMapLegendTr {
    onClick: () => void;
    carsCount?: number;
    content: any;
    isStatus?: boolean;
    isOffer?: boolean;
    selected: string;
    isIdle?: boolean;
}

const MapLegendTr = React.memo((props: IMapLegendTr) => {
    const { content, carsCount, onClick, isStatus, selected, isOffer, isIdle } = props;
    const isSelectedItem = selected
        ?.split(',')
        ?.some((selected) => selected == (isOffer ? getOfferId(content) : content?.id ?? ''));
    const isSelected = isSelectedItem
        ? styles.selected
        : '';

    return (
        <tr>
            <td>
                <Link className={`${styles.link} ${isSelected}`}
                      onClick={onClick.bind(null)}>
                    {isStatus && <span className={`${styles.icon} ${styles.status} ${styles[content?.name] ?? ''}`}/>}
                    {isIdle && <span className={`${styles.icon} ${styles.idle} ${styles[`h_${content?.id ?? ''}`]}`}/>}
                    {content?.isGroup ? <u>{content?.name}</u> : content?.name || 'Всего'}
                </Link>
            </td>
            <td>
                {content?.cars_count >= 0 ? content?.cars_count : (carsCount ?? '-')}
            </td>
        </tr>
    );
});
