import React from 'react';

import { LSSettingItems } from '../../../types';
import { defaultTraits, EMPTY_DATA, mileageSensor, ONE_MINUTE } from '../../constants';
import { CacheControl } from '../../ui/CacheControl';
import { ErrorsModal } from '../../ui/ErrorBar';
import { Link } from '../../ui/Link';
import { CacheHelper } from '../../utils/cache';
import { CSSettingsItems, CustomSettings } from '../../utils/customSettings';
import LS from '../../utils/localStorage/localStorage';
import { Request2 } from '../../utils/request';
import { debounce, getDuration, isValidGUIDString, setDocumentTitle } from '../../utils/utils';
import { IStore } from '../App/store';
import CarsFilter from '../CarsFilters';
import { initMap } from '../MainMap/utils';
import { drawPolygonControl } from '../TrackMap/drawPolygonControl';
import { drawCityControl } from './controls/drawCityControl';
import { drawClusterControl } from './controls/drawClusterControl';
import { downloadCarsXLSX, drawDownloadControl } from './controls/drawDownloadControl';
import { drawFuelStationsControl } from './controls/drawFuelStationsControl';
import { drawSearchControl } from './controls/drawSearchControl';
import ServiceControl from './controls/drawServiceControl';
import { drawShowAllControl } from './controls/drawShowAllControl';
import { drawZoomControl } from './controls/drawZoomControl';
import { colorizeIdlePreset } from './helpers/colorizeIdlePreset';
import { colorizePreset } from './helpers/colorizePreset';
import { hideCarTooltip } from './helpers/hideCarTooltip';
import { initOptions } from './helpers/initOptions';
import { setCoordinates } from './helpers/setCoordinates';
import { showCarTooltip } from './helpers/showCarTooltip';
import { WIDTH_MAP_CONTROL } from './helpers/widthMapControl';
import * as style from './index.css';
import MapLegend from './MapLegend';
import { LegendGroup } from './MapLegend/component';
import { getOfferId, offerNameSeparator, shouldDisplayCar } from './MapLegend/utils';
import { MAP_REQUESTS, REQUESTS } from './request';

declare let ymaps: any;

interface IMap2Props extends IStore {
    polyFilters: any;
    location: any;
    carCardOpen: boolean;
}

class Map2 extends React.Component<IMap2Props, any> {
    ls = new LS();
    cs = new CustomSettings();

    state = {
        carsFiltersIsOpen: false,
        isServiceModalOpen: false,
        fuelStations: [],
        hasCache: false,
        cacheTimestamp: null,
        mainError: null,
        error: null,
        showErrorModal: false,
        legend: {
            status: [],
            offer_name: [],
            model_id: [],
            idle_start: [],
        },
        tags_filter: this.isAllowSaveFilters() && (this.ls.get(LSSettingItems.tags_filter) ?? '') || '',
        idleOn: false,
    };

    isAllowSaveFilters() {
        return this.cs.get(CSSettingsItems.allowSaveCarsFilters);
    }

    filters: any = {};

    map: any;
    carObjectManager: any;

    request = new Request2({ requestConfigs: MAP_REQUESTS });
    cache = new CacheHelper({
        id: MAP_REQUESTS[REQUESTS.GET_CARS].api,
        onResponse: this.onResponse.bind(this),
        onCache: this.onCache.bind(this),
        onError: this.changeState.bind(this, 'mainError'),
    });

    carTooltip;

    models;
    cars;
    legend;
    customPolygon;

    fuelStationsCollection;
    polygonsCollection;
    customPolygonCollection;

    downloadButton;
    showAllButton;

    timerCarList;
    mapLegend;

    constructor(props) {
        super(props);
        this.carTooltip = {
            parentElement: React.createRef<HTMLDivElement>(),
            element: null,
        };
        this.mapLegend = React.createRef<any>();
    }

    getTagsFilterFromLocation() {
        const techFilterAll = 'Any';
        const modelsParam = 'models';
        if (this.props.location?.search?.length) {
            const queries = new URLSearchParams(this.props.location.search);
            const tags_filter = queries.get('tags_filter');
            const models = queries.get(modelsParam); //#DRIVEFRONT-940

            if (tags_filter?.length) {
                this.toMapClick(tags_filter);
                queries.delete('tags_filter');
            }

            const modelsArr = models?.split(',') || [];
            if (queries.has(modelsParam)) {

                let filter = modelsArr[0];

                if (!models || modelsArr[0] === techFilterAll) {
                    filter = '';
                }

                this.filterLegendFormUrl(filter);

                queries.delete(modelsParam);
            }

            const rest = queries.toString();
            if (tags_filter?.length || models?.length) {
                location.href = `${location.origin}/#/cars${rest.length ? `?${rest}` : ''}`;
            }
        }
    }

    componentDidMount() {
        this.getTagsFilterFromLocation();

        initMap('map2', (map) => {
            this.map = map;
            this.drawMap();
            this.drawExtra();
        });
    }

    componentDidUpdate(prevProps: Readonly<IMap2Props>, prevState) {
        if (this.state.tags_filter !== prevState.tags_filter) {
            this.getCars();
        }

        if (this.state.idleOn !== prevState.idleOn) {
            this.filterAll();
        }

        if (this.props.carCardOpen !== prevProps.carCardOpen) {
            if (!this.props.carCardOpen) {
                this.getCars();
            } else {
                clearTimeout(this.timerCarList);
            }
        }

        if (this.props.location?.search !== prevProps.location?.search) {
            this.getTagsFilterFromLocation();
        }
    }

    shouldComponentUpdate(nextProps: Readonly<IMap2Props>, nextState: Readonly<any>): boolean {
        return this.state.legend !== nextState.legend || this.props.polyFilters === nextProps.polyFilters;
    }

    componentWillUnmount() {
        this.request.abort();
        clearTimeout(this.timerCarList);
        this.map && this.map.destroy();
    }

    drawMap() {
        initOptions(this.map);

        drawZoomControl(this.map);
        drawSearchControl(this.map);

        const serviceControl = new ymaps.control.Button({
            data: { content: 'Сервис' },
            options: {
                maxWidth: WIDTH_MAP_CONTROL,
            },
        });
        serviceControl.events
            .add('select', () => {
                this.setState({ isServiceModalOpen: true });
                serviceControl.deselect();
            });
        this.map.controls.add(serviceControl, { float: 'right' });

        drawCityControl(this.map);

        this.polygonsCollection = new ymaps.GeoObjectCollection();
        this.map.geoObjects.add(this.polygonsCollection);
        drawPolygonControl({
            polyFilters: this.props.polyFilters,
            BlockRules: this.props.AdminUser?.blockRules,
            map: this.map,
            polygonsCollection: this.polygonsCollection,
            errorHandler: this.changeState.bind(this, 'error'),
            polyFiltersRoles: this.props.AdminUser?.activeRoles
                ? Object.keys(this.props.AdminUser.activeRoles)
                : undefined,
        });

        this.carObjectManager = new ymaps.ObjectManager({
            clusterIconLayout: 'default#pieChart',
            clusterize: true,
            gridSize: 128,
        });
        this.map.geoObjects.add(this.carObjectManager);

        const currentPosition = this.ls.get(LSSettingItems.map_settings).current_position;
        if (currentPosition) {
            this.map.setCenter(currentPosition.center, currentPosition.zoom);
        }

        const boundsHandler = (e: any) => {
            this.ls.set(LSSettingItems.map_settings, {
                ...this.ls.get(LSSettingItems.map_settings),
                current_position: { center: e.get('newCenter'), zoom: e.get('newZoom') },
            });
        };

        const debounceTime = 300;
        this.map.events.add('boundschange', debounce(boundsHandler, debounceTime));

        this.getCars();

        drawClusterControl.call(this);

        const filtersCars = new ymaps.control.Button({
            data: { content: 'Фильтр по машинам' },
            options: {
                maxWidth: WIDTH_MAP_CONTROL,
            },
        });
        filtersCars.events
            .add('select', () => {
                this.setState({ carsFiltersIsOpen: true });
                filtersCars.deselect();
            });
        this.map.controls.add(filtersCars, { float: 'right' });

        this.customPolygonCollection = new ymaps.GeoObjectCollection();
        this.map.geoObjects.add(this.customPolygonCollection);

        const filterPoly = new ymaps.control.Button({
            data: { content: 'Фильтр / Полигон' },
            options: {
                maxWidth: WIDTH_MAP_CONTROL,
            },
        });

        filterPoly.events
            .add('select', () => {
                this.addPolygon();
                filterPoly.deselect();
            });
        this.map.controls.add(filterPoly, { float: 'right' });
    }

    addPolygon() {
        this.customPolygonCollection && this.customPolygonCollection.removeAll();
        this.customPolygon = new ymaps.Polygon([], {}, {
            editorDrawingCursor: 'crosshair',
            minPoints: 4,
            fillColor: '#0000FF60',
            strokeColor: '#0000FF',
            strokeWidth: 2,
            draggable: true,
            zIndex: 99999,
        });

        this.map && this.customPolygonCollection && this.customPolygonCollection.add(this.customPolygon);

        const stateMonitor = new ymaps.Monitor(this.customPolygon.editor.state);
        stateMonitor.add('drawing', (newValue: any) => {
            this.customPolygon.options.set('strokeColor', newValue ? '#FF0000' : '#0000FF');
        });
        this.customPolygon.editor.startDrawing();
    }

    drawExtra() {
        this.fuelStationsCollection = new ymaps.ObjectManager({
            clusterize: true,
            clusterIconLayout: 'default#pieChart',
        });
        this.map.geoObjects.add(this.fuelStationsCollection);
        drawFuelStationsControl(
            this.map,
            this.fuelStationsCollection,
            this.getKeyFromState.bind(this, 'fuelStations'),
            this.changeState.bind(this, 'fuelStations'),
            this.changeState.bind(this, 'error'),
        );
    }

    manageTooltip() {
        const showTooltip = (e) => {
            hideTooltip();

            const element = document.createElement('div');
            element.className = style.tooltip;

            if (this.carTooltip.parentElement.current) {
                this.carTooltip.element = element;
                this.carTooltip.parentElement.current?.appendChild(element);

                const objectId = e.get('objectId');
                const pagePixels = e.get('pagePixels');
                const car = this.cars?.find((car) => car.id === objectId);

                car && showCarTooltip(
                    pagePixels[0], pagePixels[1],
                    car,
                    this.carTooltip,
                    this.models,
                    this.map,
                );
            }
        };

        const hideTooltip = () => {
            this.carTooltip.element && hideCarTooltip(this.carTooltip);
            this.carTooltip.element = null;
        };

        const debounceTime = 100;
        this.carObjectManager.events.add(['mouseleave'], debounce(hideTooltip, debounceTime));
        this.carObjectManager.events.add(['mouseenter'], showTooltip);
        this.carObjectManager.events.add(['click'], (e) => {
            hideTooltip();

            const objectId = e.get('objectId');

            if (isValidGUIDString(objectId)) {
                location.href = `#/cars/${objectId}/info`;
            }
        });
    }

    getCars() {
        if (!this.props.carCardOpen) {
            this.cache.exec({
                request: this.request.exec.bind(this.request, REQUESTS.GET_CARS, {
                    queryParams: {
                        tags_filter: encodeURIComponent(this.state.tags_filter),
                        sensors: mileageSensor,
                        traits: defaultTraits,
                    },
                }),
                requestKey: MAP_REQUESTS[REQUESTS.GET_CARS].api,
            });
        }
    }

    onCache(data) {
        const cacheTimestamp = data?.timestamp || 0;
        this.cars = data?.response?.cars ?? [];
        this.models = data?.response?.models ?? {};

        if (this.cars.length) {
            this.createLegend(data?.response);
            setDocumentTitle(`${this.cars.length} авто`);
        }

        this.setState({
            hasCache: true,
            cacheTimestamp,
        });
    }

    onResponse(data) {
        const cacheTimestamp = +new Date();

        this.setState({
            hasCache: false,
            cacheTimestamp,
        });

        this.models = data.models;
        this.cars = data.cars;

        this.createLegend(data);
        setDocumentTitle(`${this.cars.length} авто`);

        clearTimeout(this.timerCarList);
        this.timerCarList = setTimeout(() => this.getCars(), ONE_MINUTE);
    }

    createLegend(data) {
        const statuses = data?.statuses
            ?.filter((car) => !!car.cars_count)
            ?.map((status) => ({
                name: status.filter_id,
                cars_count: status.cars_count,
                id: status.id,
            })) ?? [];

        const idle_start = data?.surges?.idles
            ?.map((idle) => ({
                name: idle.name,
                cars_count: idle.cars_count,
                id: idle.filter_id,
            })) ?? [];

        const models = Object.keys(data.models)
            ?.map((key) => ({
                name: data?.models?.[key]?.name,
                cars_count: data?.models?.[key]?.cars_count,
                id: key,
            }))
            ?.sort((a, b) => a.name.localeCompare(b.name)) ?? [];

        const offer_name = data?.offers_array ?? [];

        this.legend = {
            status: statuses,
            model_id: models,
            offer_name,
            idle_start,
        };

        this.setState({
            legend: this.legend,
        });

        this.filterAll();
    }

    filterStatus(status) {
        const mapSettings = this.ls.get(LSSettingItems.map_settings);
        const currentFilters = {
            ...mapSettings.filters,
            'status': status,
        };
        this.filters['status'] = status;
        this.ls.set(LSSettingItems.map_settings, {
            ...mapSettings,
            ...this.isAllowSaveFilters() && { filters: currentFilters },
        });
        this.filterMap(currentFilters);

        this.updateLegend(currentFilters);
    }

    filterModel(model) {
        const mapSettings = this.ls.get(LSSettingItems.map_settings);
        const currentFilters = {
            ...mapSettings.filters,
            'model_id': model,
        };
        this.filters['model_id'] = model;
        this.ls.set(LSSettingItems.map_settings, {
            ...mapSettings,
            ...this.isAllowSaveFilters() && { filters: currentFilters },
        });
        this.filterMap(currentFilters);

        this.updateLegend(currentFilters);
    }

    filterIdle(idle) {
        const idleIndex = 2;
        const idle_start: LegendGroup = this.state.legend?.idle_start;
        const prevIdleIndex = idle !== ''
            ? idle_start?.findIndex((idle_obj) => idle_obj?.id > idle)
            : -1;
        const prevIdle = prevIdleIndex === 1
            ? { id: 0 }
            : prevIdleIndex !== -1
                ? idle_start[prevIdleIndex - idleIndex]
                : idle_start?.[idle_start?.length - idleIndex];

        const mapSettings = this.ls.get(LSSettingItems.map_settings);
        const _idle_start = idle !== '' ? [prevIdle?.id, idle] : '';
        const currentFilters = {
            ...mapSettings.filters,
            'idle_start': _idle_start,
        };
        this.filters['idle_start'] = _idle_start;
        this.ls.set(LSSettingItems.map_settings, {
            ...mapSettings,
            ...this.isAllowSaveFilters() && { filters: currentFilters },
        });
        this.filterMap(currentFilters);

        this.updateLegend(currentFilters);
    }

    filterOffer(offer) {
        if (offer.isGroup) {
            offer = this.state.legend.offer_name
                .reduce((acc: string[], legend_offer: any) => {
                    if (legend_offer?.group_name === offer.id) {
                        acc.push(getOfferId(legend_offer));
                    }

                    return acc;
                }, []);

            offer = offer.join(offerNameSeparator);
        }

        const mapSettings = this.ls.get(LSSettingItems.map_settings);
        const currentFilters = {
            ...mapSettings.filters,
            'offer_name': offer,
        };
        this.filters['offer_name'] = offer;
        this.ls.set(LSSettingItems.map_settings, {
            ...mapSettings,
            ...this.isAllowSaveFilters() && { filters: currentFilters },
        });
        this.filterMap(currentFilters);

        this.updateLegend(currentFilters);
    }

    filterAll() {
        const currentFilters = this.isAllowSaveFilters()
            && this.ls.get(LSSettingItems.map_settings).filters || this.filters;

        if (currentFilters) {
            this.filterMap(currentFilters);
            this.updateLegend(currentFilters);
        }
    }

    filterMap(currentFilters) {
        const features: any = [];
        this?.cars?.forEach(car => {
            const currentFiltersKeys = Object.keys(currentFilters);
            const shouldDisplay = shouldDisplayCar(currentFiltersKeys, currentFilters, car);

            if (shouldDisplay) {
                features.push(this.makeCar(car));
            }
        });

        this.map && this.carObjectManager && this.redrawMap(features);
    }

    checkIdle() {
        this.setState((prev: any) => ({
            idleOn: !prev.idleOn,
        }));
    }

    redrawMap(features) {
        const clusterize = this.carObjectManager.options.get('clusterize');
        this.map.geoObjects.remove(this.carObjectManager);
        this.carObjectManager = new ymaps.ObjectManager({
            clusterIconLayout: 'default#pieChart',
            clusterize,
            gridSize: 128,
        });
        this.map.geoObjects.add(this.carObjectManager);
        const FeatureCollection = {
            'type': 'FeatureCollection',
            'features': features,
        };
        this.carObjectManager.add(FeatureCollection);
        this.manageTooltip();

        this.downloadButton && this.map.controls.remove(this.downloadButton);
        if (features.length) {
            this.downloadButton = drawDownloadControl(this.map, this.carObjectManager);
        }

        this.showAllButton && this.map.controls.remove(this.showAllButton);
        if (features.length) {
            this.showAllButton = drawShowAllControl(this.map);
        }
    }

    updateLegend(currentFilters) {
        const legend = this.state.legend;
        const currentFiltersKeys = Object.keys(currentFilters);
        currentFiltersKeys.forEach((filter) => {
            Object.assign(legend, {
                [filter]: this.state.legend[filter].map((filter) => {
                    return {
                        ...filter,
                        cars_count: 0,
                    };
                }),
            });

        });

        this?.cars?.forEach(car => {
            const currentFiltersKeys = Object.keys(currentFilters);
            const shouldDisplay = shouldDisplayCar(currentFiltersKeys, currentFilters, car);

            if (shouldDisplay) {
                currentFiltersKeys?.forEach((filter) => {
                    const temp = legend[filter]?.findIndex(prop => {
                        return filter === 'idle_start' && car?.surge?.[filter]
                            ? prop?.id > getDuration(car?.surge?.[filter])
                            : filter === 'offer_name'
                                ? prop.group_name === car.offer_group_name && prop.name === car.offer_name
                                : prop?.id === car[filter];
                    });

                    if (legend[filter][temp]?.cars_count >= 0) {
                        legend[filter][temp].cars_count += 1;
                    }
                });
            }
        });
        this.setState({
            legend,
        }, () => {
            this.mapLegend && this.mapLegend.current && this.mapLegend.current.setFilter();
        });
    }

    changeState(key, value) {
        this.setState({
            [key]: value,
        });
    }

    getKeyFromState(key) {
        return this.state[key];
    }

    toMapClick(value) {
        this.isAllowSaveFilters() && this.ls.set(LSSettingItems.tags_filter, value);
        this.setState({
            tags_filter: value,
        });
        this.onCarsFiltersClose();
    }

    filterLegendFormUrl(filter) {
        this.filterModel(filter);
    }

    clearClick() {
        this.isAllowSaveFilters() && this.ls.set(LSSettingItems.tags_filter, '');
        this.setState({
            tags_filter: '',
        });
    }

    makeCar(car) {
        const coordinates = setCoordinates({ lon: car?.location?.lon, lat: car?.location?.lat });
        const caption = `${this.models?.[car.model_id]?.name || car.model_id || EMPTY_DATA}: ${car.number || EMPTY_DATA}`;

        return {
            'type': 'Feature',
            'id': car.id,
            'geometry': { 'type': 'Point', coordinates },
            'options': {
                preset: this.state.idleOn ? colorizeIdlePreset(car) : colorizePreset(car),
            },
            'properties': {
                clusterCaption: caption,
                balloonContentBody: `<a href="#/cars/${car.id}/info">перейти на карточку автомобиля</a>`,
                id: car.id,
                car,
            },
        };
    }

    reloadCars() {
        this.setState({
            mainError: null,
        });

        this.getCars();
    }

    onCarsFiltersClose() {
        this.setState({ carsFiltersIsOpen: false, evacuationIsOpen: false });
    }

    onServiceControlClose() {
        this.setState({ isServiceModalOpen: false });
    }

    render() {
        const {
            cacheTimestamp,
            carsFiltersIsOpen,
            isServiceModalOpen,
            hasCache,
            mainError,
            error,
            showErrorModal,
            legend,
            idleOn,
            tags_filter,
        } = this.state;

        return <>
            <div ref={this.carTooltip.parentElement}/>
            <div className={style.component} id={'map2'}>
            </div>
            <MapLegend legend={legend}
                       idleOn={idleOn}
                       checkIdle={this.checkIdle.bind(this)}
                       filterStatus={this.filterStatus.bind(this)}
                       filterModel={this.filterModel.bind(this)}
                       filterOffer={this.filterOffer.bind(this)}
                       filterIdle={this.filterIdle.bind(this)}
                       tags_filter={tags_filter}
                       clearClick={this.clearClick.bind(this)}
                       ref={this.mapLegend}/>
            {cacheTimestamp !== null && <CacheControl hasCache={hasCache}
                                                      newCacheAvailable={false}
                                                      cacheTimestamp={cacheTimestamp || 0}
                                                      className={style.statusControl}
                                                      classNameLoading={style.cacheControlLoading}
                                                      error={mainError}
                                                      reloadData={this.reloadCars.bind(this)}/>}
            {error && (
                <div className={`${style.statusControl} ${style.extraError}`}>
                    Произошла ошибка :(&nbsp;
                    <Link onClick={this.changeState.bind(this, 'showErrorModal', true)}>
                        Подробнее
                    </Link>
                </div>
            )}
            {carsFiltersIsOpen
                ? <CarsFilter onClose={this.onCarsFiltersClose.bind(this)}
                              toMapClick={this.toMapClick.bind(this)}
                              clearClick={this.clearClick.bind(this)}
                              downloadClick={downloadCarsXLSX.bind(this)}
                              customPolygon={this.customPolygon}/>
                : null
            }
            {showErrorModal && (
                <ErrorsModal errors={[error]}
                             onClose={this.changeState.bind(this, 'showErrorModal', false)}/>
            )}
            {isServiceModalOpen
                ? <ServiceControl onClose={this.onServiceControlClose.bind(this)}/>
                : null
            }
        </>;
    }
}

export { Map2 };
