import React from 'react';
import XLSX from 'xlsx';

import { EMPTY_DATA, GEO_POINTS } from '../../../../constants';
import CarInfo, { CarInfoHandler } from '../../../../models/car';
import { Button, ButtonTypes } from '../../../../ui/Button';
import { Window } from '../../../../ui/FullModal';
import Select, { IOptionInfo } from '../../../../ui/Select';
import { TabItem, Tabs } from '../../../../ui/Tabs';
import { Request2 } from '../../../../utils/request';
import { buttonNameDetails } from '../../../../utils/sendLogs/eventTypes/buttonDetails';
import CarNumber from '../../../CarNumber';
import Spin from '../../../Spin';
import { TRAITS } from '../../../Support/SupportEvacuations/types';
import { MAP_REQUESTS, REQUESTS } from '../../request';
import * as style from './index.css';

const SERVICE = 'service';
const TECH_OFFER_NAME = 'Техник';
const UNKNOWN_DISTRICT = 'Неизвестный район';

const AREAS = {
    MSK: GEO_POINTS.MOSCOW.LABEL.area_tags,
    SPB: GEO_POINTS.PITER.LABEL.area_tags,
    KZN: GEO_POINTS.KAZAN.LABEL.area_tags,
    UNSORTED_CARS: 'Другие',
};

const EXCLUDE_TAGS = {
    'Major_47_service': null,
    'installation': null,
    'tsvetochny_service': null,
    'Major_SPB_service': null,
    'skladochnaya_service': null,
    'maintenance': null,
    'diagnostics': null,
    'Mercedes_service': null,
    'BMW_service': null,
    'mechanic_repair': null,
    'windshield_replacement': null,
    'maintenance_tsvetochny': null,
    'maintenance_47k': null,
    'Poresche_service': null,
    'Pobeda_SPB_service': null,
    'Allianz_SPB_service': null,
    'palmira_body_service': null,
};

interface IServiceControlProps {
    onClose: () => void;
}

interface ICarTags<T> {
    [key: string]: T[];
}

interface IServiceControlState {
    isDataLoading: boolean;
    error: Error | null;
    filteredData: typeof CarInfo[];
    currentArea: string;
}

export default class ServiceControl extends React.Component<IServiceControlProps, IServiceControlState> {
    state: IServiceControlState = {
        isDataLoading: false,
        error: null,
        filteredData: [],
        currentArea: AREAS.MSK,
    };
    tabs: TabItem[] = [];
    sortedData: ICarTags<typeof CarInfo> = {};
    carTags: ICarTags<IOptionInfo> = {};
    request = new Request2({ requestConfigs: MAP_REQUESTS });

    componentDidMount(): void {
        this.getData();
        this.getTabs();
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    getData() {
        this.setState({ isDataLoading: true, error: null }, () => {
            this.request.exec(REQUESTS.GET_SAAS_CARS_WITH_DETAILS, {
                queryParams: {
                    traits: TRAITS,
                },
            })
                .then(response => {
                    const carTags: ICarTags<IOptionInfo> = {
                        [AREAS.MSK]: [],
                        [AREAS.SPB]: [],
                        [AREAS.KZN]: [],
                        [AREAS.UNSORTED_CARS]: [],
                    };

                    const sortedData: ICarTags<typeof CarInfo> = {
                        [AREAS.MSK]: [],
                        [AREAS.SPB]: [],
                        [AREAS.KZN]: [],
                        [AREAS.UNSORTED_CARS]: [],
                    };

                    response.cars && response.cars
                        .forEach(car => {
                            if ((car.status === SERVICE || car.offer_name === TECH_OFFER_NAME)
                                && car.location
                                && car.tags.every(tag => !EXCLUDE_TAGS.hasOwnProperty(tag.tag))) {

                                if (car.location?.tags.includes(AREAS.MSK)) {
                                    setCarInfo(car, sortedData, carTags, AREAS.MSK);
                                } else if (car.location?.tags.includes(AREAS.SPB)) {
                                    setCarInfo(car, sortedData, carTags, AREAS.SPB);
                                } else if (car.location?.tags.includes(AREAS.KZN)) {
                                    setCarInfo(car, sortedData, carTags, AREAS.KZN);
                                } else {
                                    setCarInfo(car, sortedData, carTags, AREAS.UNSORTED_CARS);
                                }
                            }
                        });

                    this.sortedData = sortedData;
                    this.carTags = carTags;

                    this.setState({
                        isDataLoading: false,
                        filteredData: sortedData[this.state.currentArea],
                    });
                })
                .catch(error => {
                    this.setState({ error, isDataLoading: false });
                });
        });
    }

    getTabs() {
        this.tabs = Object.values(AREAS).map(area => {
            return { name: area, link: area };
        });
    }

    filterTags(val) {
        const { currentArea } = this.state;

        if (val?.length === 0) {
            this.setState({
                filteredData: this.sortedData[currentArea],
            });
        } else {
            const filteredValues = {};
            val?.forEach(v => filteredValues[v] = null);

            const filteredData = this.sortedData[currentArea]?.filter(car => {
                return car.tags.some(tag => filteredValues.hasOwnProperty(tag.tag));
            }) || {};

            this.setState({ filteredData });
        }
    }

    getRegionSortedData() {
        const result: any = {
            [UNKNOWN_DISTRICT]: [],
        };

        this.state.filteredData.forEach(car => {
            const dist = CarInfoHandler.getDistrict.call(car);

            if (dist && !result[dist]) {
                result[dist] = [];
                result[dist].push(car);
            } else if (!dist) {
                result[UNKNOWN_DISTRICT].push(car);
            } else {
                result[dist].push(car);
            }
        });

        return result;
    }

    onTabSelect(tab: string) {
        this.setState({ currentArea: tab }, () => this.filterTags([]));
    }

    addWs(workbook, cityName) {
        const wsData: string[][] = [];
        wsData.push([
            'number',
            'model',
            'tags',
            'status',
            'fuel_level',
            'usage',
        ],
        ...this.sortedData[cityName].map(car => {
            return [
                car.number,
                car.model_id,
                car.tags.map(tag => tag.tag)?.join(),
                car.status,
                car.telematics?.fuel_level?.toString(),
                car.usage,
            ];
        }));

        const ws = XLSX.utils.aoa_to_sheet(wsData);
        XLSX.utils.book_append_sheet(workbook, ws, cityName);
    }

    onDownloadClick() {
        const workbook = XLSX.utils.book_new();
        this.addWs(workbook, `${AREAS.MSK}`);
        this.addWs(workbook, `${AREAS.SPB}`);
        this.addWs(workbook, `${AREAS.KZN}`);
        this.addWs(workbook, `${AREAS.UNSORTED_CARS}`);

        XLSX.writeFile(workbook, `serviceCars.xlsx`);
    }

    render() {
        const { isDataLoading, error, filteredData, currentArea } = this.state;
        const regionSortedData = filteredData?.length ? this.getRegionSortedData() : {};

        return <Window onClose={this.props.onClose.bind(this)}
                       title={'Сервис'}
                       error={error}
                       className={style.window}>
            {isDataLoading
                ? <Spin/>
                : <div className={style.service}>
                    <div className={style.header}>
                        <Tabs tabs={this.tabs} currentTab={currentArea} selectTab={this.onTabSelect.bind(this)}/>
                        {
                            filteredData?.length
                                ? <Button colorType={ButtonTypes.positive}
                                          onClick={this.onDownloadClick.bind(this)}
                                          ytLog={{ button_name: buttonNameDetails.SERVICE_CARS_DOWNLOAD }}>
                                    Скачать список всех машин
                                </Button>
                                : null
                        }
                    </div>
                    {!filteredData?.length
                        ? <span>Ничего не найдено</span>
                        : <div>
                            <div className={style.filter}>
                                <span>Фильтр по тегам: </span>
                                <Select options={Object.values(this.carTags[currentArea])}
                                        onSelect={this.filterTags.bind(this)}
                                        multiSelect={true}
                                        placeholder={'Теги'}/>
                            </div>
                            {
                                filteredData?.length
                                    ? <div className={style.data}>
                                        {Object.entries(regionSortedData).map((item, i) => {
                                            return <div className={style.tableItem} key={i}>
                                                <span className={style.district}>{item[0]}</span>
                                                <ServiceTable data={item[1] as typeof CarInfo[]}/>
                                            </div>;
                                        })}
                                    </div>
                                    : null
                            }
                        </div>
                    }
                </div>
            }
        </Window>;
    }
}

function setCarInfo(car, data, tags, city) {
    data[city].push(car);
    car.tags?.forEach(tag => {
        if (tag.priority > 0 && !tags[city][tag.tag]) {
            tags[city][tag.tag] = { value: tag.tag };
        }
    });
}

interface IServiceTableProps {
    data: typeof CarInfo[];
}

class ServiceTable extends React.Component<IServiceTableProps> {
    render() {
        const data = this.props.data;

        return <table className={style.table}>
            <thead className={style.table_header}>
                <tr>
                    <td>#</td>
                    <td>number</td>
                    <td>model</td>
                    <td>tags</td>
                    <td>status</td>
                    <td>fuel level</td>
                    <td>usage</td>
                </tr>
            </thead>

            <tbody>
                {data && data.map((car, i) => {
                    return <tr key={i}>
                        <td>{i + 1}</td>
                        <td>{car.number ? <CarNumber value={car.number}/> : EMPTY_DATA}</td>
                        <td>{car.model_id || EMPTY_DATA}</td>
                        <td>
                            <div>{
                                car.tags?.filter(tag => tag.priority)
                                    .map((tag, key) => {
                                        return <span key={key}>
                                            <span className={style.table_priority}>{tag.priority}: </span>
                                            <span>{tag.display_name || tag.tag} </span>
                                        </span>;
                                    })
                            }</div>
                        </td>
                        <td>{car.status || EMPTY_DATA}</td>
                        <td>{car.telematics?.fuel_level ?? EMPTY_DATA}</td>
                        <td>{car.usage || EMPTY_DATA}</td>
                    </tr>;
                })}
            </tbody>
        </table>;
    }
}
