import React from 'react';
import VirtualList from 'react-tiny-virtual-list';

import { debounce } from '../../../utils/utils';
import { MOVE_DIRECTION } from '../constants';
import * as _style from '../index.css';
import EntityItem from './EntityItem';
import { ITEM_SIZE } from './types';

interface IEntityContentProps {
    filter: string;
    data: any;
    setEntity: () => void;
    selectedEntity: any;
    moveSource: string[];
    descriptionKey: string;
    descriptionGetter?: (host: any) => void;
}

interface IEntityContentState {
    rowHeights: number[];
    windowSize: { width: number; height: number };
}

const HEIGHT_DIFF = 260;
const ITEM_PADDING = 10;
const DEBOUNCE = 200;

export class EntityContent extends React.Component<IEntityContentProps, IEntityContentState> {
    state = {
        rowHeights: [],
        windowSize: { width: window.innerWidth, height: window.innerHeight },
    };

    onResize = debounce(() => {
        this.setState({
            windowSize: { width: window.innerWidth, height: window.innerHeight },
        });
    }, DEBOUNCE);

    setRowHeights(height, index) {
        const newRowHeights: number[] = this.state.rowHeights;
        newRowHeights[index] = height + ITEM_PADDING;
        this.setState({
            rowHeights: newRowHeights,
        });
    }

    descriptionIncludes(el, host): boolean {
        const { descriptionKey, filter } = this.props;
        const _host = el?.[1]?.[host] || {};
        const customDescription = this.props?.descriptionGetter?.(_host) || '';

        return _host?.[descriptionKey]?.toUpperCase()?.includes(filter.toUpperCase())
            || customDescription?.toUpperCase()?.includes(filter.toUpperCase());
    }

    componentDidMount() {
        window.addEventListener('resize', this.onResize);
    }

    componentWillUnmount() {
        window.removeEventListener('resize', this.onResize);
    }

    render() {
        let { filter, data, setEntity, selectedEntity, moveSource, descriptionKey, descriptionGetter } = this.props;
        const { windowSize, rowHeights } = this.state;

        data = Object.entries(data).filter((el: any) => {
            if (!filter) {
                return true;
            }

            if (Object.keys(MOVE_DIRECTION).some((host) => this.descriptionIncludes(el, host.toLowerCase()))) {
                return true;
            }

            return el[0].toUpperCase().includes(filter.toUpperCase());
        });

        return <div className={_style.content}>
            <div className={_style.header}>
                <span>#</span>
                <span/>
                <span>P</span>
                <span>T</span>
                <span>ST</span>
                <span>QA</span>
                <span>DM</span>
                <span>DMp</span>
                <span>DMt</span>
                <span>entity_name</span>
                <span>description/display_name</span>
            </div>
            <VirtualList key={this.props.filter}
                         width={'100%'}
                         height={windowSize.height - HEIGHT_DIFF}
                         itemCount={data.length}
                         itemSize={(index) => {
                             return rowHeights[index] ?? ITEM_SIZE;
                         }}
                         renderItem={({ index, style }) => {
                             return <EntityItem setHeight={this.setRowHeights.bind(this)}
                                                descriptionKey={descriptionKey}
                                                descriptionGetter={descriptionGetter}
                                                index={index}
                                                item={data[index]}
                                                style={style}
                                                setEntity={setEntity}
                                                key={index}
                                                moveSource={moveSource}
                                                selectedEntities={selectedEntity}/>;
                         }}/>
        </div>;
    }
}
