import React from 'react';

import { Button, ButtonTypes } from '../../../ui/Button';
import { Window } from '../../../ui/FullModal';
import * as coreStyle from '../../../ui/index.css';
import { Link } from '../../../ui/Link';
import TextArea from '../../../ui/TextArea';
import { Request2 } from '../../../utils/request';
import { ProgressBar } from '../../ProgressBar';
import { IProgressData } from '../../QueryScheduler/QueryScheduler';
import { MIGRATE_REQUESTS } from '../request';

interface IMigrateDialogProps {
    onClose: () => void;
    selectedEntities: any;
    moveSource: string;
    moveDestination: string;
    move: () => void;
    propose?: () => void;
    update: () => void;
}

interface IMigrateDialogState {
    [x: string]: any;

    entities: any[];
    envFrom: string;
    envTo: string;
    uploadingProgress: {
        queue: any[];
        success: any[];
        errors: any[];
    };
    comment: string;
    proposeIsLoading: boolean;
    moveIsLoading: boolean;
}

const TIMEOUT = 600;

export class MigrateDialog extends React.Component<IMigrateDialogProps, IMigrateDialogState> {
    state: IMigrateDialogState = {
        entities: [],
        envFrom: '',
        envTo: '',
        uploadingProgress: {
            queue: [],
            success: [],
            errors: [],
        },
        comment: '',
        proposeIsLoading: false,
        moveIsLoading: false,
    };

    request = new Request2({ requestConfigs: MIGRATE_REQUESTS });

    componentDidMount() {
        const envFrom = this.props.moveSource;
        const envTo = this.props.moveDestination;
        const selectedEntries = Object.entries(this.props?.selectedEntities) ?? [];
        const entities = selectedEntries?.reduce((_p: any[], _c: any[]) => {
            if (_c[1][envFrom]) {
                let meta: any;
                try {
                    meta = _c[1][envFrom].meta;
                    if (typeof meta !== 'object') {
                        meta = JSON.parse(_c[1][envFrom].meta);
                    }
                } catch (e) {
                    meta = {};
                }

                const tag = Object.assign({}, _c[1][envFrom], { meta });

                if (tag.description_index) {
                    delete tag.description_index;
                }

                _p.push(tag);
            }

            return _p;
        }, []);

        this.setState({
            entities,
            envFrom,
            envTo,
        });
    }

    onProgress(data: IProgressData) {
        const { failed, queue, success } = data;
        const uploadingProgress = {
            queue,
            success,
            errors: failed,
        };

        this.setState({ uploadingProgress });
    }

    onSuccess(data: IProgressData) {
        const { failed, queue, success } = data;
        const uploadingProgress = {
            queue,
            success,
            errors: failed,
        };

        this.setState({ uploadingProgress, isBatchUploading: false }, () => {
            if (!failed.length) {
                setTimeout(() => {
                    this.setState({
                        moveIsLoading: false,
                        proposeIsLoading: false,
                    }, () => {
                        this.props.update();
                        this.props.onClose();
                    });
                }, TIMEOUT);
            } else {
                this.setState({
                    moveIsLoading: false,
                    proposeIsLoading: false,
                });
            }
        });
    }

    onButtonClick(buttonIsLoading: string, action: () => void) {
        this.setState({
            [buttonIsLoading]: true,
        }, () => {
            action();
        });
    }

    onEditComment(value) {
        this.setState({
            comment: value,
        });
    }

    render() {
        const errorLength = this.state.uploadingProgress.errors.length;

        return <Window onClose={this.props.onClose.bind(this)}
                       title={`Перенос объектов: ${this.state.envFrom} -> ${this.state.envTo}`}>
            <div>entities: {this.state.entities.length}</div>
            {this.props.propose &&
            <TextArea placeholder={'Комментарий для предложения'}
                      value={this.state.comment}
                      onChange={this.onEditComment.bind(this)}/>
            }
            <ProgressBar allLength={this.state.uploadingProgress?.queue?.length}
                         successLength={this.state.uploadingProgress?.success?.length}
                         errors={this.state.uploadingProgress.errors}/>
            {errorLength
                ? <Link href={`${location.origin}${location.pathname}#/settings/error-log`}
                        target={'_blank'}>
                    Ошибки: {errorLength}
                </Link>
                : null
            }
            <div className={`${coreStyle.button_container} ${coreStyle.full_width}`}>
                <Button onClick={this.props.onClose.bind(this)}
                        colorType={ButtonTypes.negative}>Отмена</Button>
                {this.props.move
                    ? <Button onClick={this.onButtonClick.bind(this, 'moveIsLoading', this.props.move.bind(this,
                        this.state.entities,
                        this.state.envTo,
                        this.onProgress.bind(this),
                        this.onSuccess.bind(this),
                    ))}
                              isLoading={this.state.moveIsLoading}>Перенести</Button>
                    : null
                }
                {this.props.propose
                    ? <Button onClick={this.onButtonClick.bind(this, 'proposeIsLoading', this.props.propose.bind(this,
                        this.state.entities,
                        this.state.envTo,
                        this.onProgress.bind(this),
                        this.onSuccess.bind(this),
                        this.state.comment,
                    ))}
                              isLoading={this.state.proposeIsLoading}>Перенести через предложение</Button>
                    : null
                }
            </div>
        </Window>;
    }
}
