import * as React from 'react';

import { Button } from '../../ui/Button';
import { Collapse2 } from '../../ui/Collapse2';
import { Input } from '../../ui/Input';
import Select from '../../ui/Select';
import { Request2 } from '../../utils/request';
import { SimpleError } from '../SimpleError';
import Spin from '../Spin';
import { MOVE_DIRECTION } from './constants';
import DuplicateDialog from './DuplicateDialog';
import EntityContent from './EntityContent';
import * as _style from './index.css';
import MigrateDialog from './MigrateDialog';
import { MIGRATE_REQUESTS } from './request';

interface IMoveEntityProps {
    getEntityList: (errorHandler?) => void;
    result: any[];
    moveEntities: () => void;
    proposeEntities?: () => void;
    descriptionKey: string;
    descriptionGetter?: () => void;
}

interface IMoveEntityState {
    [x: number]: any;

    filter: string;
    selectedEntities: any;
    moveSource: string[];
    moveDestination: string[];
    moveDialogIsOpen: boolean;
    isLoading: boolean;
    errors: Error[];
    duplicates: string[];
}

export class MoveEntity extends React.Component<IMoveEntityProps, IMoveEntityState> {
    state: IMoveEntityState = {
        filter: '',
        selectedEntities: {},
        moveSource: [MOVE_DIRECTION.PRESTABLE],
        moveDestination: [MOVE_DIRECTION.TESTING],
        moveDialogIsOpen: false,
        isLoading: false,
        errors: [],
        duplicates: [],
    };

    ref: any;
    request = new Request2({ requestConfigs: MIGRATE_REQUESTS });

    constructor(props) {
        super(props);
        this.ref = React.createRef();
    }

    componentDidMount() {
        this.setState({
            isLoading: true,
        }, async () => {
            await this.props.getEntityList(this.requestQueue.bind(this));
            this.setState({
                isLoading: false,
            });
        });
    }

    async requestQueue(queue: Promise<unknown>[]) {
        let response;

        if (Promise.allSettled) {
            const _response = await Promise.allSettled(queue);
            response = _response.map((response) => {
                if (response.status === 'fulfilled') {
                    return response.value;
                }

                this.errorHandler(response.reason);

                return [];

            });
        } else {
            try {
                response = await Promise.all(queue);
            } catch (error) {
                this.errorHandler(error);
            }
        }

        return response;
    }

    errorHandler(error) {
        this.setState((prev) => ({
            errors: [...prev.errors, error],
        }));
    }

    setEntities(entity, state) {
        this.setState((prevState) => {
            const { selectedEntities } = prevState;
            if (state) {
                selectedEntities[entity[0]] = entity[1];
            } else {
                delete selectedEntities[entity[0]];
            }

            return {
                ...prevState,
                selectedEntities,
            };
        });
    }

    onChange(filter) {
        this.setState({
            filter,
        });
    }

    cleanSelectedEntities() {
        this.setState({
            selectedEntities: {},
        });
    }

    onSelect(key, value) {
        let isConfirmed = false;
        if (Object.keys(this.state.selectedEntities).length) {
            isConfirmed = confirm('Есть выбранные элементы. Очистить?');
        }

        if (isConfirmed) {
            this.cleanSelectedEntities();
        }

        this.setState({
            [key]: [value],
        });
    }

    checkForDuplicates() {
        const { selectedEntities } = this.state;
        const duplicates: string[] = [];

        Object.keys(selectedEntities).forEach((entityKey) => {
            if (!!selectedEntities[entityKey]?.[this.state.moveSource[0]]
                && !!selectedEntities[entityKey]?.[this.state.moveDestination[0]]) {

                duplicates.push(entityKey);
                const destinationRevision = selectedEntities[entityKey]?.[this.state.moveDestination[0]]?.revision;
                if (destinationRevision) {
                    selectedEntities[entityKey][this.state.moveSource[0]].revision = destinationRevision;
                }
            }
        });

        this.setState({
            duplicates,
            moveDialogIsOpen: !duplicates.length,
            selectedEntities,
        });
    }

    showDialog(state) {
        this.setState({
            moveDialogIsOpen: state,
        });
    }

    render() {
        const options = Object.keys(MOVE_DIRECTION)
            .map((host) => ({ text: host, value: MOVE_DIRECTION[host] }));

        return (
            <div className={_style.component}>
                {this.state.isLoading
                    ? <Spin/>
                    : <>
                        {this.state.errors?.length && <Collapse2 title={`Ошибки: ${this.state.errors.length}`}>
                            {this.state.errors.map((error, index) => (
                                <SimpleError key={index} error={error} highlightBackendCluster/>
                            ))}
                        </Collapse2> || null}

                        {this.state.moveDialogIsOpen
                        && <MigrateDialog onClose={this.showDialog.bind(this, false)}
                                          moveSource={this.state.moveSource[0]}
                                          moveDestination={this.state.moveDestination[0]}
                                          selectedEntities={this.state.selectedEntities}
                                          update={this.props.getEntityList.bind(this, this.requestQueue.bind(this))}
                                          move={this.props.moveEntities.bind(this)}
                                          propose={this.props?.proposeEntities?.bind(this) ?? undefined}/>
                        }
                        {this.state.duplicates.length
                            ? <DuplicateDialog duplicates={this.state.duplicates}
                                               onClose={() => {
                                                   this.setState({
                                                       duplicates: [],
                                                   });
                                               }}
                                               showDialog={() => {
                                                   this.showDialog(true);
                                                   this.setState({
                                                       duplicates: [],
                                                   });
                                               }}/>
                            : null
                        }
                        <div className={_style.controls}>
                            <Input onChange={this.onChange.bind(this)}
                                   className={_style.filter}
                                   value={this.state.filter}
                                   placeholder={'фильтр'}/>
                            <div>
                                <Select onSelect={this.onSelect.bind(this, 'moveSource')}
                                        containerClassName={_style.select}
                                        initialValues={this.state.moveSource}
                                        placeholder={'начальная точка переноса'}
                                        options={options}/>
                                <span>➡️</span>
                                <Select onSelect={this.onSelect.bind(this, 'moveDestination')}
                                        containerClassName={_style.select}
                                        initialValues={this.state.moveDestination}
                                        placeholder={'конечная точка переноса'}
                                        options={options}/>
                                <Button className={_style.button}
                                        onClick={this.cleanSelectedEntities.bind(this)}
                                        basic
                                        disabled={!Object.keys(this.state.selectedEntities).length}>
                                    Сбросить выбор
                                </Button>
                                <Button className={_style.button}
                                        onClick={this.checkForDuplicates.bind(this)}
                                        disabled={!Object.keys(this.state.selectedEntities).length ||
                                    this.state.moveDestination[0] === this.state.moveSource[0]}>Перенести</Button>
                            </div>
                        </div>
                        <EntityContent descriptionKey={this.props.descriptionKey}
                                       descriptionGetter={this.props.descriptionGetter}
                                       data={this.props.result}
                                       filter={this.state.filter}
                                       moveSource={this.state.moveSource}
                                       selectedEntity={this.state.selectedEntities}
                                       setEntity={this.setEntities.bind(this)}/>
                    </>
                }
            </div>
        );
    }
}
