import moment from 'moment';
import * as React from 'react';
import ReactTooltip from 'react-tooltip';

import { LSSettingItems } from '../../../types';
import { CITIES, DEFAULT_COORDINATES, EMPTY_DATA, ONE_SECOND, UTC_OFFSET } from '../../constants';
import { CarInfoHandler } from '../../models/car';
import { dateLag } from '../../ui/FormatDate';
import { IBlockRules } from '../../utils/IBlockRules';
import { isObjectEqual } from '../../utils/isObjectEqual';
import LS from '../../utils/localStorage/localStorage';
import { Request2 } from '../../utils/request';
import { Copy } from '../Copy';
import { initMap } from '../MainMap/utils';
import { drawFuelStationsControl } from '../Map2/controls/drawFuelStationsControl';
import { WIDTH_MAP_CONTROL } from '../Map2/helpers/widthMapControl';
import { drawClientPolygonControl } from '../TrackMap/drawClientPolygonControl';
import { drawPolygonControl } from '../TrackMap/drawPolygonControl';
import drawAvailableCars from './drawAvaolableCars/index';
import * as style from './index.css';
import { CAR_REQUESTS as requestConfigs, REQUESTS } from './request';
import { drawFixPointElements } from './utils';

declare let ymaps: any;

const TEN_MINUTES = 600;
const ZOOM_POSITION = { left: 2, top: 45 };
const FRACTION_DIGITS = 3;
const SERVICE_LINK_PREFIX = 'https://carsharing.yandex-team.ru/driveservice/point/show?what=';
const TOOLTIP_ID = 'copy_tooltip';

interface IMiniMapInternal {
    polyConfig: any;
    polyFilters: any;
}

interface IMiniMapProps extends IBlockRules, IMiniMapInternal {
    mapId: string;
    carInfo: any;
    showExpandButton?: boolean;
    userRealtimeData: any;
    availableCars?: any;
}

interface IMiniMapState {
    address: string;
    mapIsExpanded: boolean;
    location: any;
    lag: any;
}

export class MiniMap extends React.Component<IMiniMapProps, IMiniMapState> {
    map: any;
    polygonsAreaCollection: any;
    availableCarsCollection: any;
    fixPoint: any;
    radarCollection: any;
    userPosition: any;
    tail: any;
    carCollections: any;
    state: IMiniMapState = {
        address: '...',
        mapIsExpanded: false,
        location: null,
        lag: null,
    };
    access = false;
    request = new Request2({ requestConfigs });
    img: any = new Image();
    externalMarksList = null;
    externalMarksListItems = [];
    saasExtraTimer: any;
    externalMarkBalloon: any = null;
    currentExternalBalloon = '';
    followCar = true;
    fuelStations;
    public static defaultProps: Partial<any> = {
        tail: true,
    };

    ls = new LS();
    firstDraw = true;

    mainPolyControl;
    polygonsControl;
    fuelControl;

    componentDidMount() {
        if (!this.props.BlockRules?.HideSmallMap) { // don't init minimap;
            const map_id = this.props.mapId ? this.props.mapId : 'miniMap';
            initMap(map_id, (map) => {
                this.map = map;
                this.drawMap();
                this.getCarModelImage();
                this.extraDrawLocationBySaas();
                this.drawAvailableCarts();
            });
        }
    }

    componentDidUpdate(prevProps: Readonly<any>): void {
        if (!this.props.BlockRules?.HideSmallMap) { // don't init minimap;
            if (!isObjectEqual(this.props.carInfo, prevProps.carInfo)) {
                this.map && this.extraDrawLocationBySaas();
            }

            if (this.props.carInfo?.id !== prevProps.carInfo?.id) {
                this.getCarModelImage();
                this.map && this.redrawControls();
            }

            if (!isObjectEqual(this.props?.availableCars, prevProps?.availableCars)) {
                this.map && this.drawAvailableCarts();
            }
        }
    }

    drawAvailableCarts() {
        drawAvailableCars({
            availableCarsCollection: this.availableCarsCollection,
            availableCars: this.props.availableCars,
        });
    }

    componentWillUnmount() {
        clearTimeout(this.saasExtraTimer);
        this.request.abort();
    }

    drawMap() {
        const MIN_ZOOM = 2;
        const MAX_ZOOM = 19;
        const CENTER_LAT = 37.648295;
        const CENTER_LON = 55.768326;
        const DEFAULT_ZOOM = 17;

        const map = this.map;

        const zoomControl = new ymaps.control.ZoomControl({
            options: {
                position: ZOOM_POSITION,
                size: 'small',
            },
        });
        map.controls.add(zoomControl);

        map.setCenter([CENTER_LAT, CENTER_LON], DEFAULT_ZOOM);

        map.options.set('minZoom', MIN_ZOOM);
        map.options.set('maxZoom', MAX_ZOOM);
        map.options.set('autoFitToViewport', 'always');
        map.options.set('suppressMapOpenBlock', true);
        map.options.set('copyrightLogoVisible', false);
        map.options.set('copyrightProvidersVisible', false);
        map.options.set('copyrightUaVisible', false);

        this.polygonsAreaCollection = new ymaps.GeoObjectCollection();
        map.geoObjects.add(this.polygonsAreaCollection);

        this.availableCarsCollection = new ymaps.GeoObjectCollection();
        map.geoObjects.add(this.availableCarsCollection);

        this.fixPoint = new ymaps.GeoObjectCollection();
        map.geoObjects.add(this.fixPoint);

        this.radarCollection = new ymaps.GeoObjectCollection();
        map.geoObjects.add(this.radarCollection);

        this.userPosition = new ymaps.GeoObjectCollection();
        map.geoObjects.add(this.userPosition);

        this.tail = new ymaps.GeoObjectCollection();
        map.geoObjects.add(this.tail);

        this.carCollections = new ymaps.GeoObjectCollection();
        map.geoObjects.add(this.carCollections);

        this.fuelStations = new ymaps.ObjectManager({
            clusterize: true,
            clusterIconLayout: 'default#pieChart',
        });
        map.geoObjects.add(this.fuelStations);

        const ExpandMap = new ymaps.control.Button({
            data: { content: 'Развернуть карту' },
            options: {
                maxWidth: WIDTH_MAP_CONTROL,
            },
        });

        const FollowBtn = new ymaps.control.Button({
            data: { content: 'Следить' },
            state: {
                selected: true,
            },
            options: {
                maxWidth: WIDTH_MAP_CONTROL,
            },
        });

        this.externalMarksList = new ymaps.control.ListBox({
            data: {
                content: 'Доп. метки',
            },
        });

        ExpandMap.events
            .add('select', () => {
                ExpandMap.data.set('content', 'Свернуть карту');
                this.setState({ mapIsExpanded: true });
            })
            .add('deselect', () => {
                ExpandMap.data.set('content', 'Развернуть карту');
                this.setState({ mapIsExpanded: false });
            });

        FollowBtn.events
            .add('select', () => {
                this.followCar = true;
            })
            .add('deselect', () => {
                this.followCar = false;
            });

        map.controls.add('routeButtonControl', {
            size: 'large',
            float: 'left',
            floatIndex: 1000,
        });
        map.controls.get('routeButtonControl').routePanel.state.set({
            fromEnabled: true,
            toEnabled: true,
            from: '',
            to: '',
            type: 'masstransit',
        });

        this.externalMarkBalloon = new ymaps.Balloon(map);
        this.externalMarkBalloon.options.setParent(map.options);
        this.externalMarkBalloon.events.add('userclose', () => {
            this.currentExternalBalloon = '';
        });

        this.props.showExpandButton &&
        map.controls.add(ExpandMap, { float: 'left' });

        // map.controls.add(CreateFuelLayerBtn.call(this, ymaps), {float: 'right'});

        map && map.controls.add(this.externalMarksList, { float: 'right' });
        map && map.controls.add('typeSelector', {
            float: 'left',
        });
        const mapSettings = this.ls.get(LSSettingItems.map_settings);
        map.setType(mapSettings.typeSelector || 'yandex#map');
        map.events.add('typechange', (event) => {
            const typeSelector = event.originalEvent.map.getType();
            this.ls.set(LSSettingItems.map_settings, {
                ...mapSettings,
                typeSelector,
            });
        });

        map && map.controls.add('searchControl', {
            float: 'right',
            provider: 'yandex#search',
        });
        map && map.controls.add(FollowBtn, { float: 'right' });
    }

    getTail(props: any) {
        props.id && this.request.exec(REQUESTS.GET_TAIL, { queryParams: { car_id: props.id } })
            .then((response: any) => {
                this.tail && this.tail.removeAll();
                const track = response && response.tracks && response.tracks.length && response.tracks[0].track || [];

                const line: number[][] = [];
                const oddDiv = 2;

                track && track.length && track.forEach((coord: number, index: number) => {
                    index % oddDiv == 0 && line.push(track.slice(index, index + oddDiv));
                });

                const polyline = line && line.length && new ymaps.Polyline(line, {}, {
                    strokeColor: '#ca463e',
                    strokeWidth: 4,
                    strokeStyle: '1 3',
                }) || null;

                polyline && this.tail.add(polyline);

                const IMAGE_SIZE = 16;
                const GET_IMAGE_SIZE = 40;
                const IMAGE_OFFSET = -8;

                const tailPlacemark = line && line.length && new ymaps.Placemark(line[line.length - 1], {}, {
                    iconLayout: 'default#image',
                    iconImageHref: this.getImage(GET_IMAGE_SIZE, GET_IMAGE_SIZE, this.drawTail.bind(this)),
                    iconImageSize: [IMAGE_SIZE, IMAGE_SIZE],
                    iconImageOffset: [IMAGE_OFFSET, IMAGE_OFFSET],
                });

                line && line.length && this.tail.add(tailPlacemark);

            });
    }

    getImage(width: number, height: number, drawFn: (context: any) => void) {
        const deg180 = 180;
        const halfDiv = 2;

        const courseDegrees = this.props.carInfo
            && this.props.carInfo.location
            && this.props.carInfo.location.course;
        const courseRadians = (courseDegrees * Math.PI / deg180);

        const canvas = document.createElement('canvas');
        canvas.width = width;
        canvas.height = height;
        const ctx: any = canvas.getContext('2d');

        ctx.translate(width / halfDiv, height / halfDiv);
        ctx.rotate(courseRadians);

        drawFn(ctx);

        const imgUrl = canvas.toDataURL();

        return imgUrl;
    }

    drawTail(ctx: any) {
        const x = 15;
        const tailColor = '#ca463e';
        ctx.beginPath();
        ctx.moveTo(-x, x);
        ctx.lineTo(0, -x);
        ctx.lineTo(x, x);
        ctx.fillStyle = tailColor;
        ctx.fill();
    }

    drawCar(ctx: any) {
        if (this.img.getAttribute('src') !== 'null'
            && this.img.getAttribute('src') !== 'undefined'
            && this.img.getAttribute('src')) {
            this.drawCarImage(ctx);
        } else {
            this.drawPseudoCarImage(ctx);
        }
    }

    drawPseudoCarImage(ctx: any) {
        const rectWidth = 16;
        const rectHeight = 72;
        const headColor = '#FF0000';
        const halfDiv = 2;

        const rectPosition = {
            x: Math.floor(-rectWidth / halfDiv),
            y: Math.floor(-rectHeight / halfDiv),
        };

        ctx.fillRect(rectPosition.x, rectPosition.y, rectWidth, rectHeight);
        ctx.fillStyle = headColor;
        ctx.fillRect(rectPosition.x, rectPosition.y, rectWidth, rectWidth);
    }

    drawCarImage(ctx: any) {
        const carImageWidth = 88;
        const carImageHeight = 144;
        const halfDiv = 2;

        const carImagePosition = {
            x: Math.floor(-carImageWidth / halfDiv),
            y: Math.floor(-carImageHeight / halfDiv),
        };
        try {
            ctx.drawImage(this.img, carImagePosition.x, carImagePosition.y);
        } catch (err) {
            this.drawPseudoCarImage(ctx);
        }
    }

    getAddress(coord: any[]) {
        ymaps.geocode(coord)
            .then((res: any) => {
                const firstGeoObject = res.geoObjects.get(0);
                this.setState({ address: firstGeoObject.getAddressLine() });
            });
    }

    drawExtraCarUpdate(props: any) {
        const {
            lon = DEFAULT_COORDINATES.LON,
            lat = DEFAULT_COORDINATES.LAT,
            precision, timestamp, type, base_latitude, base_longitude,
        } = props?.location || {};
        const { locations = [] } = props;

        const BaseCoords = base_latitude && base_longitude
        && (base_latitude !== 0 || base_longitude !== 0)
            ? new ymaps.Placemark([base_longitude, base_latitude], {
                hintContent: `Сырые координаты GPS`,
            }, {})
            : null;

        this.setState({ location: props.location, lag: props.lag });
        const precisionCircle = new ymaps.Circle([
            [lon, lat],
            precision,
        ], {
            hintContent: `timestamp: ${timestamp} | type: ${type} | precision: ${precision} `,
        }, {
            fillColor: '#DB709377',
            strokeColor: '#990066',
            strokeOpacity: 0.5,
            strokeWidth: 2,
        });

        const GET_IMAGE_SIZE = 144;
        const IMAGE_SIZE = 72;
        const IMAGE_OFFSET = -36;

        this.map
        && this.carCollections.removeAll()
        && this.carCollections.add(precisionCircle)
        && this.carCollections.add(new ymaps.Placemark([lon, lat], {}, {
            type: 'car',
            iconLayout: 'default#image',
            iconImageHref: this.getImage(GET_IMAGE_SIZE, GET_IMAGE_SIZE, this.drawCar.bind(this)),
            iconImageSize: [IMAGE_SIZE, IMAGE_SIZE],
            iconImageOffset: [IMAGE_OFFSET, IMAGE_OFFSET],
        }));

        this.map &&
        this.followCar &&
        !ymaps.util.bounds.containsPoint(this.map.getBounds(), [lon, lat]) &&
        this.map.setCenter([lon, lat]);

        this.map &&
        this.getAddress([lon, lat]);

        this.map && this.getTail(props);
        this.map && BaseCoords && this.carCollections && this.carCollections.add(BaseCoords);

        if (!this.mainPolyControl && !this.fuelControl) {
            this.redrawControls();
        }

        this.createExternals(locations);

        if (this.firstDraw) {
            const offer = CarInfoHandler.getSessionCurrentOffer.call(this.props.carInfo);
            drawFixPointElements({ offer, collection: this.fixPoint });

            this.firstDraw = false;
            this.map.setBounds(this.map.geoObjects.getBounds());
            this.map.setZoom(this.map.getZoom() - 1);
        }

        const { latitude = null, longitude = null, last_location_update = null } = this.props.userRealtimeData || {};

        if (latitude && longitude && this.map) {
            this.userPosition.removeAll();
            this.userPosition.add(new ymaps.Placemark([longitude, latitude]
                , { hintContent: new Date(last_location_update * ONE_SECOND).toLocaleString() }
                , {
                    preset: 'islands#bluePersonIcon',
                }));
        }
    }

    redrawControls() {
        const isFinished = this.props.carInfo?.session_info?.meta?.finished ?? true;
        const sessionId = this.props.carInfo?.session_info?.meta?.session_id ?? '';

        this.mainPolyControl && this.map.controls.remove(this.mainPolyControl);
        this.polygonsControl && this.map.controls.remove(this.polygonsControl);
        this.fuelControl && this.map.controls.remove(this.fuelControl);

        this.polygonsAreaCollection?.removeAll();
        this.fuelStations?.removeAll();

        if (!isFinished && sessionId) {
            this.mainPolyControl = drawClientPolygonControl({
                sessionId,
                polyConfig: this.props.polyConfig,
                polygonsCollection: this.polygonsAreaCollection,
                map: this.map,
                polyControlPairData: {
                    polyFilters: this.props.polyFilters,
                    BlockRules: this.props.BlockRules,
                },
            }, this);
        } else {
            this.mainPolyControl = drawPolygonControl({
                polyFilters: this.props.polyFilters,
                BlockRules: this.props.BlockRules,
                map: this.map,
                polygonsCollection: this.polygonsAreaCollection,
            });
        }

        this.fuelControl = drawFuelStationsControl(
            this.map,
            this.fuelStations,
        );
    }

    createExternals = (locations: any[]) => {
        this.buildExternalListItem(locations);
        this.buildExternalMarks(locations);
    };
    buildExternalListItem = (locations: any[]) => {
        const externalMarksList: any = this.externalMarksList;
        externalMarksList && externalMarksList.removeAll();
        const externalMarksListItemsInit: any = this.externalMarksListItems;

        const externalMarksListItems: any = locations.map((item: any) => {
            const name = item.name || 'default';
            const selected = externalMarksListItemsInit.some((initItem: any) => {
                return !!(initItem.data._data.content === name && initItem.state._data.selected);
            });

            return new ymaps.control.ListBoxItem({ data: { content: name }, state: { selected } });
        });

        externalMarksListItems.forEach((item: any) => {
            externalMarksList && externalMarksList.add(item);
        });

        this.externalMarksListItems = externalMarksListItems;
    };

    buildExternalMarks = (locations: any[]) => {
        const locationMarks = locations
            .filter((location: any) => {
                const name = location.name || 'default';

                return this.externalMarksListItems.some((initItem: any) => {
                    return !!(initItem.data._data.content === name && initItem.state._data.selected);
                });
            })
            ?.map((location: any) => {
                const name = location.name || 'default';
                const keysRows = Object.entries(location).map(data => {
                    const [key, value] = data;
                    const tr = (key.indexOf('lon') == -1
                        && key.indexOf('lat') == -1
                        && key.indexOf('base') == -1
                        && key.indexOf('type') == -1
                    )
                        ? ('<tr><td>' + key + '</td><td>' + value + '</td></tr>')
                        : '';

                    return tr;
                }).join('');
                const balloonBody = '<h4>' + name + '</h4>' + '<table>' + keysRows + '</table>';

                const mark = new ymaps.Placemark([location.longitude, location.latitude], {
                    iconCaption: name,
                }, {
                    preset: 'islands#blueCircleDotIconWithCaption',
                });

                const precision = new ymaps.Circle([
                    [location.longitude, location.latitude],
                    location.precision,
                ], {
                    hintContent: `timestamp: ${location.timestamp} | type: ${location.type} | precision: ${location.precision} `,
                }, {
                    fillColor: '#425B9970',
                    strokeColor: '#222199',
                    strokeOpacity: 0.5,
                    strokeWidth: 2,
                });

                mark.events.add('click', (e: any) => {
                    this.externalMarkBalloon.setData(balloonBody);
                    if (!this.currentExternalBalloon) {
                        this.externalMarkBalloon.open([location.longitude, location.latitude]);
                    } else {
                        this.externalMarkBalloon.setPosition([location.longitude, location.latitude]);
                    }

                    this.currentExternalBalloon = location.type;
                });

                if (this.currentExternalBalloon === location.type) {
                    this.externalMarkBalloon.setData(balloonBody);
                    this.externalMarkBalloon.setPosition([location.longitude, location.latitude]);
                }

                return { mark, precision };

            });
        this.map && locationMarks.forEach((mark: any) => {
            this.carCollections.add(mark.mark);
            this.carCollections.add(mark.precision);
        });
    };

    getCarModelImage() {
        const props = this.props;
        const imgUrl = props?.carInfo?.views?.[0]?.image_map_url_2x
            || props?.carInfo?.models?.[0]?.image_map_url_2x;

        this.img = new Image();
        this.img.crossOrigin = 'Anonymous';
        this.img.src = imgUrl;

        //this.polygonsAreaCollection && this.polygonsAreaCollection.removeAll();
        //this.ParkingLayerBtn && this.ParkingLayerBtn.state.set('selected', false);
    }

    extraDrawLocationBySaas() {
        this.props.carInfo?.id && this.drawExtraCarUpdate(this.props.carInfo);
    }

    getTechArea() {
        const area = this.state.location
            && this.state.location.areas
            && this.state.location.areas.reduce((prev: any, cur: any) => {
                let _area = '';
                if (cur.tags && cur.tags.indexOf('techzone') > -1) {
                    _area = cur.title;
                }

                return `${prev} ${_area}`;
            }, '');

        return area && area.trim();
    }

    render() {
        const coord = this.state.location ? (this.state.location.lat + ',' + this.state.location.lon) : '';
        const lonlat = this.state.location ? (this.state.location.lon + ', ' + this.state.location.lat) : '';
        const techArea = this.getTechArea();
        const map_id = this.props.mapId ? this.props.mapId : 'miniMap';
        const address = this.state.address;//this.props?.carInfo?.geocoded_location;
        let city = '';
        this.props?.carInfo?.location?.tags.forEach((tag) => {
            if (CITIES[tag]) {
                city = CITIES[tag].short_name;
            }
        });
        const number = CarInfoHandler.getNumber.call(this.props.carInfo);
        const model = CarInfoHandler.getModelName.call(this.props.carInfo);

        const shortCopyItem = `${address} (\`${coord}\`) ${techArea ? '(' + techArea + ')' : ''} `;
        const fullCopyItem = `**${number} ${model}** 
${address} (\`${coord}\`) ${techArea ? '(' + techArea + ')' : ''}
${SERVICE_LINK_PREFIX}${coord} `;
        if (this.props.BlockRules?.HideSmallMap) {
            return <div>Карта временно недоступна :(</div>;
        }

        return <div className={`${style.mini_map} ${style[`map_is_expanded_${this.state.mapIsExpanded}`]}`}>
            <div className={style.address}>
                <ReactTooltip id={TOOLTIP_ID}/>
                <span data-for={TOOLTIP_ID} data-tip={'Адрес + координаты'}>
                    <Copy text={shortCopyItem}>
                        <span className={`${style.map_copy_icon} ${style.icon_map}`}/>
                    </Copy>
                </span>
                <span data-for={TOOLTIP_ID} data-tip={'Сервисная информация'}>
                    <Copy text={fullCopyItem}>
                        <span className={`${style.settings_copy_icon} ${style.icon_map}`}/>
                    </Copy>
                </span>

                {coord &&
                <a target={'_blank'}
                   href={'yandexnavi://show_point_on_map?lat=' + this.state.location.lat + '&lon='
                + this.state.location.lon + '&desc=' + (number)}>
                    {coord}
                </a>}
                |&nbsp;{lonlat &&
            <a href={`https://yandex.ru/maps?ll=${lonlat}&z=18&pt=${lonlat}`}
               target={'_blank'}>
                Я.Карта
            </a>}&nbsp;|&nbsp;
                {techArea}
                {city}, {address}
            </div>
            <div className={style.mini_map_container} id={map_id}>
                {
                    this.state.location
                    && this.state.location.type
                    && <div className={style.debug_info}>
                        <b>Debug info:</b> type: <i>{this.state.location.type}</i> |
                        precision: <i>{this.state.location.precision
                        && this.state.location.precision.toFixed(FRACTION_DIGITS)}м</i> |
                        timestamp: <i>{moment(this.state.location.timestamp * ONE_SECOND).isValid() && moment(this.state.location.timestamp * ONE_SECOND).utcOffset(UTC_OFFSET).format('HH:mm:ss.ms')} ({this.state.location.timestamp})</i>
                    </div>
                }
                {
                    this.state.lag
                    && <div className={style.car_lags}>
                        <LagItem title={'время создания коннекта машины с сервером телематики'}
                                 lag={this.state.lag}
                                 name={'created'}/>
                        <LagItem title={'лаг черного ящика'} lag={this.state.lag} name={'blackbox'}/>
                        <LagItem title={'лаг соединения'} lag={this.state.lag} name={'heartbeat'}/>
                        <LagItem title={'лаг позиции'} lag={this.state.lag} name={'location'}/>
                        <LagItem title={'лаг configurator'} lag={this.state.lag} name={'configurator'} ignoreWarning/>
                    </div>
                }
                {this.props.carInfo
                && <BaseInfo location={this.props.carInfo.location} access={this.access}/>}
            </div>
        </div>;
    }
}

const LagItem = (props: { title: string; lag: any; name: string; ignoreWarning?: boolean }) => {
    const { title, lag, name, ignoreWarning } = props;
    let warning = lag[name] && ((new Date().getTime() - lag[name] * ONE_SECOND) / ONE_SECOND) > TEN_MINUTES;
    if (name === 'created') {
        warning = !warning;
    }

    if (ignoreWarning) {
        warning = false;
    }

    return <div title={title} className={style[`is_warning_${warning}`]}>
        {name}: {lag[name] &&
    <i>{dateLag(lag[name] * ONE_SECOND)}</i> || EMPTY_DATA}
    </div>;
};

const PolygonTagItem = (props: any) => {
    const { tag } = props;
    const className = `${style.list_element} ${tag.includes('allow') ? style.allow : ''} ${tag.includes('deny') ? style.deny : ''}`;

    return <span key={tag} className={className}>{tag}</span>;
};

const BaseInfo = (props: { location: any; access: boolean }) => {
    const { location, access = false } = props;

    return <div className={style.base_info}>
        <table>
            <tbody>
                <tr>
                    <td className={style.key}>polygon tags</td>
                </tr>
                <tr>
                    <td className={style.value}>{
                        location && location.tags
                            ?.map((tag: string) => {
                                let component: any = null;

                                if (access || !access && (tag.indexOf('surge') === -1)) {
                                    component = <PolygonTagItem key={tag} tag={tag}/>;
                                }

                                return component;
                            })
                    }</td>
                </tr>
            </tbody>
        </table>
    </div>;
};
