import { CLASS_FURGON, ELECTRIC, MODEL_LEAF, PONY, SHUTTLE } from '../../constants';
import { CSSettingsItems, CustomSettings } from '../../utils/customSettings';
import { fillColor, stroke } from '../MainMap/utils';

declare let ymaps: any;

export const normalizeCoordinates = (coord) => {
    const object: number[][] = [];
    const oddDiv = 2;
    coord.forEach((item: any, index: number) => {
        index % oddDiv == 0 && object.push(coord.slice(index, index + oddDiv));
    });

    return object;
};

export const drawFixPointElements = (props: { offer: any; collection: any }) => {
    const { offer = {}, collection = {} } = props;
    if (offer && offer.finish_area) {
        const p = offer.finish_area.split(' ').filter((e: any) => e);
        const poly: any[] = normalizeCoordinates(p);

        const myPolygon = new ymaps.Polygon([
            poly, [],
        ], {
            hintContent: 'Fix Point Finish Area',
        }, {
            fillColor: '#00FF0020',
            strokeWidth: 2,
            strokeColor: '#FF0000',
        });

        collection && collection.add(myPolygon);
    }

    if (offer && offer.finish) {
        const finish = new ymaps.Placemark(offer.finish.split(' '),
            {
                iconContent: 'Fix Point Finish',
            },
            {
                preset: 'islands#redStretchyIcon',
            },
        );
        collection && collection.add(finish);
    }
};

export interface DrawPolyParams {
    polygons: any[];
    polygonsCollection: any;
    polyConfig?: { stroke: number; opacity_coeff: number };
    pinCollection?: any;
    action?: (poly: any) => void;
    currentFilter?: string;
}

const DEFAULT_POLY_STROKE = 2;
const DEFAULT_OPACITY_COEFF = 1.5;

export const drawPoly = (props: DrawPolyParams) => {
    const { polygons, polygonsCollection, pinCollection, action, polyConfig } = props;
    const cs = new CustomSettings();
    const isNightMode = cs.get(CSSettingsItems.nightMode);
    polygons.map((item: any, index: number) => {
        const tooltip = item?.area_tooltip || item?.tooltip || {};
        const poly = new ymaps.Polygon([item.area_coords || item.coords], {
            hintContent: `${(item.area_title || item.title || '')
                            || (item.area_id || item.id || '')} - ${item.area_tags || item.area_style?.id || ''}`,
        }, {
            fillColor: item?.area_style?.fill_color ??
                (item.area_tags && fillColor({ area_tags: item.area_tags })),
            //always show stroke for better visibility
            strokeWidth: item?.area_style?.stroke_width || (polyConfig?.stroke ?? DEFAULT_POLY_STROKE),
            strokeAlpha: item?.area_style?.stroke_alpha ?? 1,
            strokeColor: isNightMode && '#fff' || (item?.area_style?.stroke_color ?? stroke(item.area_tags)),
            ...isNightMode && { strokeStyle: 'dash' },
            area_id: item?.area_id,
            revision: item?.revision,
            area_tags: item?.area_tags,
            area_title: item?.area_style?.name || item?.area_title,
            area_index: item?.area_style?.index || item?.area_index || 0,
            area_type: item?.type || item?.area_type || '',
            area_tooltip: JSON.stringify(tooltip),
            area_details: item?.area_details && JSON.stringify(item.area_details) || '{}',
        });

        //adding to options in init breaks Polygons dropdown
        if (item?.area_style?.fill_alpha) {
            poly.options.set(
                'fillOpacity',
                item?.area_style?.fill_alpha / (polyConfig?.opacity_coeff ?? DEFAULT_OPACITY_COEFF),
            );
        }

        polygonsCollection.add(poly);
        if (item?.area_details?.hint_position) {
            const coordinates = item.area_details.hint_position.split(' ');

            const hint = new ymaps.Placemark(coordinates.map((i: any) => +i).reverse(),
                Object.assign({}, {
                    iconContent: 'Hint Position for ' + item.area_id,
                }),
                {
                    preset: 'islands#darkBlueStretchyIcon',
                },
            );
            pinCollection && pinCollection.add(hint);
        }

        action && action(poly);
    });
};

export const polyCargoFilter = (polygons: any[]) => {
    return [
        ...polygons
            .filter((item: any) => {
                const tags = item.area_tags.split(',');

                return tags.some((item: string) => {
                    item = item && (item + '').trim();

                    return ['deny_drop_car', 'allow_riding', 'impoundment_lot'].includes(item)
                        || item.indexOf('cargo') > -1;
                });
            }),
    ];
};

export const polyShuttleFilter = (polygons: any[]) => {
    return [
        ...polygons
            .filter((item: any) => {
                const tags = item.area_tags.split(',');

                return tags.some((item: string) => {
                    item = item && (item + '').trim();

                    return ['deny_drop_car', 'allow_riding'
                        , 'shuttle_allow_drop_car', 'shuttle_deny_drop_car', 'impoundment_lot'].includes(item);
                });
            }),
    ];
};

export const polyElectricFilter = (polygons: any[]) => {
    return [
        ...polygons
            .filter((item: any) => {
                const tags = item.area_tags.split(',');

                return tags.some((item: string) => {
                    item = item && (item + '').trim();

                    return ['deny_drop_car', 'allow_riding'
                        , 'electric_car_allow_drop_car', 'electric_car_deny_drop_car'
                        , 'impoundment_lot'].includes(item);
                });
            }),
    ];
};

export const polyPorsche911Filter = (polygons: any[]) => {
    return [
        ...polygons
            .filter((item: any) => {
                const tags = item.area_tags.split(',');

                return tags.some((item: string) => {
                    item = item && (item + '').trim();

                    return ['deny_drop_car', 'allow_riding'
                        , 'porsche_car_allow_drop_car', 'impoundment_lot'].includes(item);
                });
            }),
    ];
};

export const polyExclusiveFilter = (polygons: any[]) => {
    return [
        ...polygons
            .filter((item: any) => {
                const tags = item.area_tags.split(',');

                return tags.some((item: string) => {
                    item = item && (item + '').trim();

                    return ['deny_drop_car', 'allow_riding'
                        , 'pony_car_allow_drop_car', 'pony_car_deny_drop_car'
                        , 'impoundment_lot'].includes(item);
                });
            }),
    ];
};

export const polyBasicFilter = (polygons: any[]) => {
    return [
        ...polygons
            .filter((item: any) => {
                const tags = item.area_tags.split(',');

                return tags.some((item: string) => {
                    item = item && (item + '').trim();

                    return ['allow_drop_car', 'deny_drop_car', 'allow_riding', 'impoundment_lot'].includes(item);
                });
            }),
    ];
};

export const polySurgeFilter = (polygons: any[]) => {
    return [
        ...polygons
            .filter((item: any) => {
                const tags = item.area_tags.split(',');

                return tags.some((item: string) => {
                    item = item && (item + '').trim();

                    return item.indexOf('surge') === -1;
                });
            }),
    ];
};

export const polyTechGlobalFilter = (polygons: any[]) => {
    return [
        ...polygons
            .filter((item: any) => {
                const tags = item.area_tags.split(',');

                return tags.some((item: string) => {
                    item = item && (item + '').trim();

                    return item.indexOf('techzone') === -1
                        && item.indexOf('global_area') === -1;
                });
            }),
    ];
};

export const drawCurrentCarPoly = (polyResponse, carTags, polygonsCollection) => {
    let polygons = polyResponse && polyResponse.areas || [];

    polygons = polySurgeFilter(polygons);
    polygons = polyTechGlobalFilter(polygons);

    const tags = carTags || [];
    const cargo = tags.filter((_t: any) => _t.tag === CLASS_FURGON);
    const shuttle = tags.filter((_t: any) => _t.tag === SHUTTLE);
    const pony = tags.filter((_t: any) => _t.tag === PONY);
    const electric = tags.filter((_t: any) => [ELECTRIC, MODEL_LEAF].includes(_t.tag));
    /* let porsche_911 = tags.filter((_t: any) => [PORSCHE_911, PANAMERA].includes(_t.tag)); */

    if (cargo.length) {
        polygons = polyCargoFilter(polygons);
    } else if (shuttle.length) {
        polygons = polyShuttleFilter(polygons);
    } else if (pony.length) { // exclusive
        polygons = polyExclusiveFilter(polygons);
    } else if (electric.length) { //electric
        polygons = polyElectricFilter(polygons);
    } /* else if (porsche_911.length) { //electric
        polygons = polyPorsche911Filter(polygons);
    } */ else {
        polygons = polyBasicFilter(polygons);
    }

    drawPoly({
        polygons,
        polygonsCollection,
    });
};
