import React from 'react';

import { ONE_SECOND } from '../../constants';
import { FormatDateInString } from '../../ui/FormatDate';
import { NoInformation } from '../../ui/NoInformation';
import { isObjectEqual } from '../../utils/isObjectEqual';
import { Request2 } from '../../utils/request';
import { PictureGallery } from '../PictureGallery';
import { SimpleError } from '../SimpleError';
import Spin from '../Spin';
import style from './index.css';
import { IPhotoItem, PhotoItem } from './PhotoItem';
import { PictureGalleryControlItem } from './PictureGalleryControItem';
import { requestConfigs, REQUESTS } from './request';

interface IPhotoFilters {
    source: null | string;
}

interface IPhotosPops {
    car_id?: string;
    session_id?: string;
    since?: number;
    until?: number;
    filters?: IPhotoFilters;
    onlyMarked?: boolean;
}

interface IPhotosState {
    isLoading: boolean;
    error: null | Error;
    images: IPhotoItem[];
    selectedPhotoIndex: null | number;
}

export class Photos extends React.Component<IPhotosPops, IPhotosState> {
    state: IPhotosState = {
        isLoading: false,
        error: null,
        images: [],
        selectedPhotoIndex: null,
    };

    request = new Request2({ requestConfigs });

    componentDidMount() {
        this.getData();
    }

    componentDidUpdate(prevProps: Readonly<IPhotosPops>, prevState: Readonly<IPhotosState>, snapshot?: any) {
        if (
            this.props.since !== prevProps.since
            || this.props.until !== prevProps.until
            || this.props.car_id !== prevProps.car_id
            || this.props.session_id !== prevProps.session_id
            || this.props.onlyMarked !== prevProps.onlyMarked
            || !isObjectEqual(this.props.filters, prevProps.filters)
        ) {
            this.getData();
        }
    }

    componentWillUnmount() {
        this.request.abort();
    }

    getData() {
        this.setState({
            isLoading: true,
        }, () => {
            this.request.exec(REQUESTS.GET_PHOTOS, {
                queryParams: {
                    car_id: this.props.car_id,
                    session_id: this.props.session_id,
                    since: this.props.since,
                    until: this.props.until,
                    ...this.props.filters,
                },
            })
                .then(response => {
                    const filteredImages = this.props.onlyMarked
                        ? response.images.filter(el => el.mark_up_list?.length)
                        : response.images;

                    this.setState({
                        isLoading: false,
                        images: (filteredImages ?? []).sort((a, b) => b.created_at - a.created_at),
                    });
                })
                .catch(error => {
                    this.setState({
                        isLoading: false,
                        error,
                    });
                });
        });

    }

    showImg(selectedPhotoIndex) {
        this.setState({
            selectedPhotoIndex,
        });
    }

    click(e) {
        if (e.target.localName === 'img') {
            this.showImg(e.target.dataset.index * 1);
        }
    }

    render() {
        return <div>
            {
                this.state.isLoading && <Spin/>
            }
            {
                this.state.error && <SimpleError error={this.state.error}/>
            }
            {
                !this.state.images?.length && !this.state.isLoading && <NoInformation/>
            }
            <div onClick={this.click.bind(this)} className={`${style.component_grid}`}>
                {
                    this.state.images.map((el, index) => {
                        return <PhotoItem photo={el} index={index} key={index}/>;
                    })
                }
            </div>
            {
                this.state.selectedPhotoIndex !== null
                && <PictureGallery initialIndex={this.state.selectedPhotoIndex}
                                   pictures={this.state.images.map(el => {
                                       return {
                                           link: el.url,
                                           title: FormatDateInString({
                                               value: el.created_at * ONE_SECOND,
                                               withSecond: true,
                                           }),
                                       };
                                   })}
                                   onChange={this.showImg.bind(this)}
                                   controls={<PictureGalleryControlItem photo={
                                       this.state.images[this.state.selectedPhotoIndex]
                                   }/>}
                                   onClose={this.showImg.bind(this, null)}/>
            }
        </div>;
    }
}
