import * as React from 'react';

import { FullModal } from '../../ui/FullModal';
import LS from '../../utils/localStorage/localStorage';
import Img, { convertSrcTestingQa } from '../Img';
import Spin from '../Spin';
import * as style from './index.css';

enum ChangePictureDirection {
    PREV = 'prev',
    NEXT = 'next',
}

interface IPicture {
    link: string;
    url?: string;
    title?: string;
}

const ARROW_LEFT_KEY = 'ArrowLeft';
const ARROW_RIGHT_KEY = 'ArrowRight';

interface IPictureGalleryProps {
    onClose: () => void;
    initialIndex?: number;
    pictures: IPicture[];
    readyScr?: boolean;
    controls?: React.ReactElement;
    onChange?: (index: number) => {};
    video?: boolean;
    withSize?: boolean;
    errors?: (Error | null)[];
}

interface IPictureGalleryState {
    currentIndex: number;
    width?: number;
    height?: number;
}

export class PictureGallery extends React.Component<IPictureGalleryProps, IPictureGalleryState> {
    state: IPictureGalleryState = {
        currentIndex: 0,
        width: 0,
        height: 0,
    };
    galleryItem: React.RefObject<HTMLDivElement>;

    constructor(props: IPictureGalleryProps) {
        super(props);

        const currentIndex = this.props.initialIndex !== undefined ? this.props.initialIndex : 0;
        this.galleryItem = React.createRef();

        this.state = { currentIndex };
    }

    componentDidMount(): void {
        if (this.galleryItem.current) {
            this.galleryItem.current.onkeyup = this.onKeyPress.bind(this);
            this.galleryItem.current.setAttribute('tabIndex', '99');
            this.galleryItem.current && this.galleryItem.current.focus();
        }
    }

    componentDidUpdate(prevProps: Readonly<IPictureGalleryProps>): void {
        if (prevProps.initialIndex !== this.props.initialIndex) {
            const currentIndex = this.props.initialIndex !== undefined ? this.props.initialIndex : 0;
            this.setState({ currentIndex });
        }
    }

    componentWillUnmount() {
        this.galleryItem.current && this.galleryItem.current.blur();
    }

    onKeyPress(e: KeyboardEvent) {
        let direction;
        if (e.key === ARROW_LEFT_KEY) {
            direction = ChangePictureDirection.PREV;
        } else if (e.key === ARROW_RIGHT_KEY) {
            direction = ChangePictureDirection.NEXT;
        }

        this.changePicture(direction);
    }

    changePicture(direction: ChangePictureDirection) {
        if (direction === ChangePictureDirection.PREV) {
            const currentIndex = this.state.currentIndex === 0
                ? this.props.pictures.length - 1
                : this.state.currentIndex - 1;
            this.setState({ currentIndex }, () => {
                this.props.onChange && this.props.onChange(this.state.currentIndex);
            });
        } else if (direction === ChangePictureDirection.NEXT) {
            const currentIndex = this.state.currentIndex === this.props.pictures.length - 1
                ? 0
                : this.state.currentIndex + 1;
            this.setState({ currentIndex }, () => {
                this.props.onChange && this.props.onChange(this.state.currentIndex);
            });
        }
    }

    handleImageLoaded(img) {
        if (this.props.withSize) {
            this.setState({
                width: img.width,
                height: img.height,
            });
        }
    }

    render() {
        const currentPicture = this.props.pictures && this.props.pictures[this.state.currentIndex] || {};

        return <FullModal onClose={this.props.onClose} isMinimal={false}>
            <div ref={this.galleryItem} className={style.gallery} onKeyPress={this.onKeyPress.bind(this)}>
                {
                    this.props.controls
                        ? <div className={style.controls}>{this.props.controls}</div>
                        : null
                }
                {this.props.pictures && this.props.pictures.length && this.props.pictures.length > 1
                    ? <GalleryNavigationArrow changePicture={this.changePicture.bind(this)}
                                              direction={ChangePictureDirection.PREV}/>
                    : <div/>}
                <div className={style.gallery_image_container}>
                    {this.props.video
                        ? <Video src={currentPicture?.link || currentPicture?.url || ''}
                                 readyScr={this.props.readyScr}/>
                        : <Img readyScr={this.props.readyScr}
                               controls={true}
                               error={this.props.errors?.[this.state.currentIndex]}
                               handleImageLoaded={this.handleImageLoaded.bind(this)}
                               src={currentPicture?.link || currentPicture?.url || ''}/>
                    }
                    <div className={style.picture_info_container}>
                        <span className={style.picture_info}>
                            {this.state.currentIndex + 1}/{this.props.pictures && this.props.pictures.length}
                        </span>
                        {currentPicture && currentPicture.title
                            ? <div>
                                {currentPicture && currentPicture.title}
                            </div>
                            : null}
                        {this.props.withSize && this.state.width && ` (size: ${this.state.width}x${this.state.height})`}
                    </div>
                </div>
                {this.props.pictures && this.props.pictures.length && this.props.pictures.length > 1
                    ? <GalleryNavigationArrow changePicture={this.changePicture.bind(this)}
                                              direction={ChangePictureDirection.NEXT}/>
                    : <div/>}
            </div>
        </FullModal>;
    }
}

const GalleryNavigationArrow = React.memo((props: { changePicture: () => {}; direction: ChangePictureDirection }) => {
    const { changePicture, direction } = props;
    const directionClassName = direction === ChangePictureDirection.PREV ? style.left : style.right;

    return <div className={style.navigation}
                onClick={changePicture.bind(null, direction)}>
        <div className={`${style.arrow} ${directionClassName}`}/>
    </div>;
});

interface IVideo {
    readyScr?: boolean;
    title?: string;
    src: string;
}

interface IState {
    isLoading: boolean;
    error: string;
}

export class Video extends React.Component <IVideo, IState> {
    video: HTMLVideoElement;
    rootElement: HTMLElement;
    ls = new LS();

    state = {
        isLoading: true,
        error: '',
    };

    resize() {
        const ratio = this.rootElement.offsetWidth / this.video.offsetWidth;
        if (ratio < 1) {
            this.video.width = this.video.offsetWidth * ratio;
        }

        this.video.classList.remove('visible_false');
    }

    onLoad() {
        this.resize();
        this.setState({ isLoading: false, error: '' });
    }

    onError(e: any) {
        const error = e.nativeEvent.path
        && e.nativeEvent.path[0]
        && e.nativeEvent.path[0].error
            ? e.nativeEvent.path[0].error.message
            : e.currentTarget && e.currentTarget.error
                ? e.currentTarget.error.message
                : 'error';

        this.setState({ isLoading: false, error: error });
    }

    onLoadStart() {
        this.video.classList.add('visible_false');
        this.setState({ isLoading: true, error: '' });
    }

    render() {
        const src = this.props.readyScr ? this.props.src : convertSrcTestingQa(this.props.src, this.ls);

        return <div ref={(div: HTMLDivElement) => {
            this.rootElement = div;
        }}>
            {src
            && <video ref={(video: HTMLVideoElement) => this.video = video}
                      loop
                      controls
                      src={src}
                      onLoadStart={this.onLoadStart.bind(this)}
                      onError={this.onError.bind(this)}
                      onLoadedData={this.onLoad.bind(this)}>
            </video>}
            {this.state.isLoading && this.props.src ? <Spin size={'l'}/> : this.state.error &&
                <div>{this.state.error}</div>}
        </div>;
    }
}
