import moment from 'moment';
import * as React from 'react';

import { Button, ButtonTypes } from '../../ui/Button';
import { Collapse } from '../../ui/Collapse';
import { download } from '../../utils/utils';
import QueryScheduler, { COUNT_FIRST, IProgressData } from '../QueryScheduler/QueryScheduler';
import { SimpleError } from '../SimpleError';
import * as style from './index.css';
import { IProgressBarProps } from './types';

const ALL_LENGTH = 100;

export const ProgressBar = (props: IProgressBarProps) => {
    const { allLength = ALL_LENGTH, successLength = 0, errors = [] } = props;
    const errorsLength = errors.length;
    const errorsIndexes = errors && errors.map(error => error.data) || [];

    const oneBlockWidth = `${allLength ? ((ALL_LENGTH / allLength)) : 0}%`;
    const styleWidth = {
        width: oneBlockWidth,
    };
    const allBlocks = new Array(successLength + errorsLength).fill('');

    return <>
        <div className={style.progress_bar}>
            <div className={style.progress_title}>
                <span className={style.success}>{successLength}/{allLength}</span>
                <span className={style.errors}>{` (${errorsLength})`}</span>
            </div>
            {allBlocks.map((block, index) => {
                return <div key={index}
                            className={errorsIndexes.includes(index) ? style.failed : style.progress}
                            style={styleWidth}/>;
            })}
        </div>
        {errorsLength
            ? <ErrorListCollapse errors={errors}/>
            : null}
    </>;
};

interface IErrorListCollapseProps {
    errors: { data: number; error: Error }[];
}

interface IErrorListCollapseState {
    collapsed: boolean;
}

class ErrorListCollapse extends React.Component<IErrorListCollapseProps, IErrorListCollapseState> {
    state: IErrorListCollapseState = { collapsed: true };

    changeCollapseStatus() {
        this.setState({ collapsed: !this.state.collapsed });
    }

    render() {
        const { errors = [] } = this.props;
        const { collapsed } = this.state;

        return <Collapse onClick={this.changeCollapseStatus.bind(this)}
                         collapsed={collapsed}
                         title={'Список ошибок'}>
            {errors?.map((error, index) => {
                return <SimpleError key={index} error={error?.error || {}}/>;
            })}
        </Collapse>;
    }
}

interface IRetryProgressButtonsProps {
    uploadingProgress: { [key: string]: any };
    fileData: { [key: string]: any }[] | null;
    isWorking?: boolean;
    onProgress: (data: IProgressData) => void;
    onSuccess: (data: IProgressData) => void;
    withoutDownload?: boolean;
    multiplyBatches?: boolean;
}

export class RetryProgressButtons extends React.Component<IRetryProgressButtonsProps, any> {

    state = {
        currentLoadingData: this.props.fileData,
    };

    componentDidMount(): void {
        const currentLoadingData = this.props.fileData;
        this.setState({ currentLoadingData });
    }

    uploadRetryFileData() {
        const { uploadingProgress, onProgress, onSuccess } = this.props;

        const failedRunsIndexes = uploadingProgress.errors
            && uploadingProgress.errors.map((error: any) => error.data) || [];

        const failedRuns = failedRunsIndexes && failedRunsIndexes.length
        && uploadingProgress.queue && uploadingProgress.queue.length
            ? failedRunsIndexes
                .map((failedRunsIndex: number) => uploadingProgress.queue[failedRunsIndex])
            : [];

        const stateCurrentLoadingData: any = this.state.currentLoadingData;
        const currentLoadingData = failedRunsIndexes && failedRunsIndexes.length
            ? failedRunsIndexes
                .map((failedRunsIndex: number) => stateCurrentLoadingData && stateCurrentLoadingData[failedRunsIndex])
            : [];

        const qs = new QueryScheduler({
            queue: failedRuns,
            limit: COUNT_FIRST,
            onProgress: onProgress.bind(this),
            onSuccess: onSuccess.bind(this),
        });

        qs.run();

        this.setState({ currentLoadingData });
    }

    downloadFailed() {
        const { uploadingProgress } = this.props;
        const { currentLoadingData: fileData } = this.state;

        const failedRunsIndexes = uploadingProgress.errors
            && uploadingProgress.errors.map((error: any) => error.data) || [];

        const failedData = failedRunsIndexes.map((failedRunIndex: number) => {
            return fileData && fileData[failedRunIndex];
        });

        let allCsvData = '';

        if (this.props.multiplyBatches) {

            const keys = failedData && failedData[0] && failedData[0][0] && Object.keys(failedData[0][0]);

            let csvData = failedData.map((failedDataArray: any) => {
                return failedDataArray.map((failedDataItem: any) => {
                    return keys.reduce((res: any[], key: string) => {
                        res.push(failedDataItem[key]);

                        return res;
                    }, []).join(';');
                });
            });

            csvData = csvData.reduce((result: string[], csvDataItem: any[]) => {
                result.push(...csvDataItem);

                return result;
            }, []);

            allCsvData = [keys.join(';'), ...csvData].join('\n');

        } else {
            const keys = failedData && failedData[0] && Object.keys(failedData[0]);

            const csvData = failedData.map((failedDataItem: any) => {
                return keys.reduce((res: any[], key: string) => {
                    res.push(failedDataItem[key]);

                    return res;
                }, []).join(';');
            });

            allCsvData = [keys.join(';'), ...csvData].join('\n');
        }

        const fileName = `failedData ${moment().format('DD-MM-YYYY')}.csv`;
        download(allCsvData, fileName, 'text/csv');
    }

    render() {
        const { isWorking } = this.props;

        return <div className={style.retry_progress_buttons}>
            <Button disabled={isWorking}
                    onClick={this.uploadRetryFileData.bind(this)}
                    basic>Загрузить повторно</Button>
            {!this.props.withoutDownload
                ? <Button disabled={isWorking}
                          onClick={this.downloadFailed.bind(this)}
                          className={style.button}
                          colorType={ButtonTypes.positive}
                          basic>Скачать ошибочные</Button>
                : null}
        </div>;
    }
}
