import React from 'react';
import { RouteComponentProps, withRouter } from 'react-router-dom';
import ReactTooltip from 'react-tooltip';

import { Dict } from '../../../../types';
import { EMPTY_DATA } from '../../../constants';
import { Button } from '../../../ui/Button';
import FormatDate from '../../../ui/FormatDate';
import { Window } from '../../../ui/FullModal';
import { JsonModal } from '../../../ui/FullModal/JsonModal';
import { Link } from '../../../ui/Link';
import { NoInformation } from '../../../ui/NoInformation';
import * as tblStyle from '../../../ui/Table/index.css';
import { isValidJSONString } from '../../../utils/isValidJSONString';
import { Copy } from '../../Copy';
import {
    RegistrationFlowItem,
    RegistrationFlowItemAction,
    RegistrationFlowType,
} from '../../InterfaceAdminConfig/RegistrationFlow/schema';
import TagModal from '../../TagModal';
import { OBJECT_TYPES } from '../../TagModal/component';
import { GroupActions } from '../GroupActions';
import { RegistrationTimelineSize } from '../types';
import style from './index.css';

interface ILineProps {
    registration_flow: RegistrationFlowItem[];
    data: Dict<any>;
    size: RegistrationTimelineSize;
    uniqueTag: string;
}

interface ILineState {
    filteredTimeline: RegistrationFlowItem[];
    filteredTimelineObject: Dict<RegistrationFlowItem>;
    checks: Dict<any>;
    selectedId: string | null;
}

interface ICheckDetailsProps {
    onClose: () => void;
    flowItem: RegistrationFlowItem;
    checks: any[];
}

interface IFlowActionsProps extends RouteComponentProps<any> {
    actions: RegistrationFlowItemAction[];
    onCloseModal: () => void;
}

interface CheckDetailsView {
    data: any[];
}

const tooltip = 'check_tooltip';

export default class Line extends React.Component<ILineProps, ILineState> {
    state: ILineState = {
        filteredTimeline: [],
        filteredTimelineObject: {},
        checks: {},
        selectedId: null,
    };

    componentDidMount() {
        const filteredTimeline = (this.props?.registration_flow || []).filter(el => el.enabled)
            .sort((a, b) => a.order - b.order) || [];
        this.setState({
            filteredTimeline,
            filteredTimelineObject: filteredTimeline.reduce((_p, _c) => {
                if (!_p.hasOwnProperty(_c.id)) {
                    _p[_c.id] = _c;
                }

                return _p;
            }, {}),
            checks: this.props?.data?.fields?.reduce((_p, _c) => {
                if (!_p.hasOwnProperty(_c.key)) {
                    let value = {};
                    if (isValidJSONString(_c.value)) {
                        value = JSON.parse(_c.value);
                    }

                    _p[_c.key] = value;
                }

                return _p;
            }, {}) || {},
        }, () => {
            ReactTooltip.rebuild();
        });
    }

    getLast(checkName) {
        const checks = this.state.checks?.[checkName] || [];

        return checks?.sort?.((a, b) => {
            return a.create_ts && b.create_ts
                ? a.create_ts - b.create_ts
                : (a.ts && b.ts) && (a.ts - b.ts);
        })?.[checks.length - 1] || {};
    }

    openModal(selectedId) {
        this.setState({
            selectedId,
        });
    }

    render() {
        const selectedCheckConfig: RegistrationFlowItem = this.state.selectedId
            && this.state.filteredTimelineObject[this.state.selectedId] || {} as RegistrationFlowItem;
        let selectedCheck = this.state.selectedId && this.state.checks[this.state.selectedId];
        if (selectedCheck && Object.keys(selectedCheck)?.length && !Array.isArray(selectedCheck)) {
            selectedCheck = [selectedCheck];
        }

        return <div className={`${style.component_wrapper}`}>
            <div className={`${style.component} ${style[`size_${this.props.size}`]}`}>
                <ReactTooltip id={tooltip} html type={'info'}/>
                <span className={style.title}> Проверки ({this.props.uniqueTag}): </span>
                {
                    this.state.filteredTimeline.map((el, index) => {
                        const tip = `<div>${el.title}</div> <i>${el.id}</i>`;
                        let check = this.state.checks[el.id] || {};
                        if (check && Object.keys(check)?.length && !Array.isArray(check)) {
                            check = [check];
                        }

                        const lastCheck = this.getLast(el.id);
                        const isCommon = el.type === RegistrationFlowType.COMMON;
                        const largeTitle = this.props.size === RegistrationTimelineSize.LARGE
                            ? (`${!isCommon ? getPrecheckType(lastCheck) : ''} ${el.title || el.id}: `)
                            : '';

                        const commonState = !Object.keys(check).length
                            ? 'empty'
                            : checkVerdict(lastCheck)[0];

                        const botState = !Object.keys(check).length
                            ? 'empty'
                            : botCheckVerdict(lastCheck);

                        return <div key={index}
                                    onClick={this.openModal.bind(this, el.id)}
                                    className={`${style.block} ${style.state} ${isCommon
                                        ? style[commonState]
                                        : style[botState]}`}
                                    data-for={tooltip}
                                    data-tip={tip}>{largeTitle}{check?.length || EMPTY_DATA}</div>;
                    })
                }
                {
                    this.state.selectedId && selectedCheckConfig?.type
                    && <CheckDetails flowItem={selectedCheckConfig}
                                     checks={selectedCheck}
                                     onClose={this.openModal.bind(this, null)}/>
                }
            </div>
            <GroupActions flowItem={this.state.filteredTimelineObject} uniqueTag={this.props.uniqueTag}/>
        </div>;
    }
}

const checkVerdict = (check) => {
    if (!check.hasOwnProperty('result')) {
        return ['without_result', null];
    }

    const result = check?.result?.verdicts || check?.result.general || {};
    if (Object.values(result).every(el => el == 'OK')) {
        return ['result_ok', result];
    }

    return ['result_not_ok', result];

};

const botCheckVerdict = (check) => {
    return check?.action === 'enter' ? 'result_ok' : 'result_not_ok';
};

const getPrecheckType = (el) => {
    return el.type === 'full' ? 'FULL' : 'LIGHT';
};

const FlowActions = withRouter((props: IFlowActionsProps) => {
    const { actions, onCloseModal } = props;

    const [selectedAction, selectAction] = React.useState<RegistrationFlowItemAction | null>(null);
    const object_id = props?.match?.params?.client_id;
    const onClose = () => {
        selectAction(null);
        onCloseModal();
    };

    return <>
        {selectedAction && object_id &&
        <TagModal initialData={{ tag: selectedAction.tagName, ...selectedAction.initialData }}
                  title={selectedAction.buttonName}
                  objectId={{ type: OBJECT_TYPES.USER, id: object_id }}
                  onClose={onClose}/>
        }
        <div className={style.buttons}>
            {
                actions.map((el, index) => {
                    return <Button key={index} onClick={selectAction.bind(null, el)}>
                        {el.buttonName}
                    </Button>;
                })
            }
        </div>
    </>;
});

const CheckDetails = (props: ICheckDetailsProps) => {
    return <Window onClose={props.onClose} title={`${props.flowItem.title}: ${props.flowItem.id}`}>
        {props.flowItem.type === RegistrationFlowType.BOT
            ? <CheckBotDetailsView data={props.checks}/>
            : <CheckCommonDetailsView data={props.checks}/>}
        {
            props.flowItem?.actions?.length
                ? <FlowActions actions={props.flowItem?.actions} onCloseModal={props.onClose}/>
                : <pre>Нет доступных действий</pre>
        }
    </Window>;
};

const CheckBotDetailsView = (props: CheckDetailsView) => {
    const [json, showJson] = React.useState(null);

    return <>
        {json && <JsonModal onClose={showJson.bind(null, null)} obj={json}/>}
        {props.data?.length ? <table className={`${tblStyle.table} ${tblStyle.clearFirstOpacity} ${style.table}`}>
            <thead>
                <tr>
                    <th>Время проверки</th>
                    <th>Тип проверки</th>
                    <th>Выбранное действие</th>
                    <th>Подробности</th>
                    <th>Граф</th>
                    <th>Пул</th>
                    <th/>
                </tr>
            </thead>
            <tbody>
                {props.data?.map((el, index) => {
                    const status = botCheckVerdict(el);

                    return <tr key={index}>
                        <td>
                            {el.ts && <FormatDate value={+el.ts} withSecond/>}
                        </td>
                        <td>{getPrecheckType(el)}</td>
                        <td>
                            <span className={`${style.item_status} ${style[status]}`}/>
                            {el.action}
                        </td>
                        <td>{el?.verification_statuses?.comment}</td>
                        <td>{el.nirvana_link && <Link href={`${el.nirvana_link}`}
                                                      target={'_blank'}>нирвана</Link> || EMPTY_DATA} </td>
                        <td><Link onClick={showJson.bind(null, el)}>JSON</Link></td>
                    </tr>;
                })
                }
            </tbody>
        </table>
            : <NoInformation/>
        }
    </>;
};

const CheckCommonDetailsView = (props: CheckDetailsView) => {
    const [json, showJson] = React.useState(null);

    return <>
        {json && <JsonModal onClose={showJson.bind(null, null)} obj={json}/>}
        {props.data?.length ? <table className={`${tblStyle.table} ${tblStyle.clearFirstOpacity} ${style.table}`}>
            <thead>
                <tr>
                    <th>Создано</th>
                    <th>Завершено</th>
                    <th className={style.result}>Результаты</th>
                    <th className={style.id}/>
                    <th/>
                    <th>Граф</th>
                    <th>Пул</th>
                    <th/>
                </tr>
            </thead>
            <tbody>
                {
                    props.data?.map((el, index) => {
                        const [status, results] = checkVerdict(el);

                        return <tr key={index}>
                            <td>
                                {el.create_ts && <div><FormatDate value={el.create_ts} withSecond/></div>}
                                {el.created_at && <div><FormatDate value={el.created_at} withSecond/></div>}
                            </td>
                            <td>
                                <FormatDate value={el.submit_ts} withSecond/>
                            </td>
                            <td>
                                <span className={`${style.item_status} ${style[status]}`}/>
                                {results && JSON.stringify(results) || EMPTY_DATA}</td>
                            <td>
                                <div><strong><i>secret:</i></strong> <Copy>{el.secret}</Copy></div>
                                <div><strong><i>worker_id:</i></strong> <Copy>{el.worker_id}</Copy></div>
                            </td>
                            <td><Link onClick={showJson.bind(null, el)}>JSON</Link></td>
                            <td>{el.nirvana_link &&
                        <Link href={`${el.nirvana_link}`} target={'_blank'}>нирвана</Link> || EMPTY_DATA} </td>
                            <td>{el.pool_id && <Link href={`https://yang.yandex-team.ru/requester/pool/${el.pool_id}`}
                                                     target={'_blank'}>{el.pool_id}</Link> || EMPTY_DATA}</td>
                            <td>{el.assignment_id &&
                        <Link href={`https://yang.yandex-team.ru/task/${el.pool_id}/${el.assignment_id}`}
                              target={'_blank'}>задание</Link> || EMPTY_DATA}</td>

                        </tr>;
                    })
                }
            </tbody>
        </table> : <NoInformation/>}
    </>;
};
