import React from 'react';

import { ITagsWithPropsExport, UserTags } from '../../../decorators/UserTags';
import { isObjectEqual } from '../../utils/isObjectEqual';
import { ErrorSource, logError } from '../Content/initErrorCounter';
import { RegistrationFlowItem } from '../InterfaceAdminConfig/RegistrationFlow/schema';
import { SimpleError } from '../SimpleError';
import Spin from '../Spin';
import style from './index.css';
import Line from './Line';
import { RegistrationTimelineSize } from './types';

interface IRegistrationTimelineProps extends ITagsWithPropsExport {
    userId: string;
    size: RegistrationTimelineSize;
    registration_flow: RegistrationFlowItem[];
}

@UserTags()
export default class RegistrationTimeline extends React.Component<IRegistrationTimelineProps, any> {
    state = {
        error: null,
    };

    componentDidMount() {
        this.getData();
    }

    componentDidUpdate(prevProps: Readonly<IRegistrationTimelineProps>, prevState: Readonly<any>, snapshot?: any) {
        if (this.props.userId !== prevProps.userId) {
            this.getData();
        }
    }

    shouldComponentUpdate(
        nextProps: Readonly<IRegistrationTimelineProps>,
        nextState: Readonly<any>,
        nextContext: any,
    ): boolean {
        return this.props.userId !== nextProps.userId
            || !isObjectEqual(this.props.tags, nextProps.tags)
            || this.props.tagIsLoading !== nextProps.tagIsLoading;
    }

    getData() {
        this.props.userId && this.props?.getTags?.({ object_id: this.props.userId });
    }

    componentDidCatch(error: Error, errorInfo: React.ErrorInfo) {
        logError(error, ErrorSource.ERROR_BOUNDARY, errorInfo);
        this.setState({ error });
    }

    render() {
        return <div className={style.component}>{this.props.tagIsLoading ? <Spin size={'s'}/>
            : this.state.error
                ? <SimpleError error={this.state.error}/>
                : <>
                    <MultiLine flow={this.props.registration_flow} tags={this.props.tags} size={this.props.size}/>
                </>
        }
        </div>;
    }
}

const MultiLine = ({ flow, tags, size }) => {
    const uniqueInitTags = [...new Set(flow.map(el => el.init_tag))];

    return <>{uniqueInitTags.map((uniqueTag: string) => {
        const data = tags?.find(el => el.tag === uniqueTag);
        const registration_flow = flow.filter(el => el?.init_tag === uniqueTag);

        return data && <Line data={data}
                             uniqueTag={uniqueTag}
                             key={uniqueTag}
                             registration_flow={registration_flow}
                             size={size}/> || null;
    }) || null}</>;
};
