import moment from 'moment';
import * as React from 'react';

import { Dict } from '../../../types';
import { ONE_SECOND } from '../../constants';
import { SessionHistoryInfoHandler } from '../../models/session';
import { Request2 } from '../../utils/request';
import BillingViewCurrent from '../Clients/BillingView/BillingViewCurrent';
import BillingViewHistory from '../Clients/BillingView/BillingViewHistory';
import { BillingViewRefundTable } from '../Clients/BillingView/BillingViewRefundTable';
import { SimpleError } from '../SimpleError';
import Spin from '../Spin';
import { SessionContext } from './component';
import * as styles from './index.css';
import { REQUESTS, SESSION_REQUESTS } from './request';

interface ISessionCardBillingProps {
    sessionId: string;
    BlockRules: Dict<any>;
}

interface ISessionCardBillingState {
    isLoading: boolean;
    error: Error | null;
    current: any[];
    history: any[];
    tags: Dict<any>;
    tagsError: Error | null;
}

export default class SessionCardBilling extends React.Component<ISessionCardBillingProps, ISessionCardBillingState> {
    state: ISessionCardBillingState = {
        isLoading: false,
        error: null,
        current: [],
        history: [],
        tags: {},
        tagsError: null,
    };

    request = new Request2({ requestConfigs: SESSION_REQUESTS });

    getData() {
        const { sessionId } = this.props;
        const SINCE_SUBTRACT = 90;

        this.setState({
            isLoading: true,
        }, () => {
            this.request.exec(REQUESTS.GET_BILLING_INFO, {
                queryParams: {
                    session_id: sessionId,
                    since: +(+moment(Date.now()).subtract(SINCE_SUBTRACT, 'd') / ONE_SECOND).toFixed(0),
                    until: +(Date.now() / ONE_SECOND).toFixed(0),
                },
            })
                .then(async (response) => {
                    const history = response?.compiled_history?.sort((a, b) => b?.timestamp - a?.timestamp) ?? [];
                    const current = response?.current?.reverse() ?? [];

                    await this.getRefundTags();

                    this.setState({
                        history,
                        current,
                        isLoading: false,
                    });
                })
                .catch((error) => {
                    this.setState({
                        error,
                        isLoading: false,
                    });
                });
        });
    }

    async getRefundTags() {
        const object_id = SessionHistoryInfoHandler.getUserId.call(this.context);
        if (!object_id) {
            return;
        }

        try {
            const tags = await this.request.exec(REQUESTS.GET_USER_HISTORY_TAGS, {
                queryParams: {
                    object_id,
                    since: Math.round(SessionHistoryInfoHandler.getStart.call(this.context) / ONE_SECOND),
                },
            });
            this.setState({
                tags,
            });
        } catch (tagsError) {
            this.setState({
                tagsError,
            });
        }
    }

    componentDidMount() {
        this.getData();
    }

    componentDidUpdate(prevProps: Readonly<ISessionCardBillingProps>) {
        if (prevProps.sessionId !== this.props.sessionId) {
            this.getData();
        }
    }

    render() {
        const { history, current, error, isLoading, tags, tagsError } = this.state;
        const { sessionId } = this.props;
        const userId = SessionHistoryInfoHandler.getUserId.call(this.context);

        return error
            ? <SimpleError error={error}/>
            : isLoading
                ? <Spin/>
                : <div className={styles.billingTableContainer}>
                    <BillingViewCurrent className={styles.currentBilling}
                                        userId={userId}
                                        data={current}
                                        currentFilter={''}
                                        reloadData={this.getData.bind(this)}/>
                    <BillingViewHistory userId={userId}
                                        data={history}
                                        reloadData={this.getData.bind(this)}/>
                    <BillingViewRefundTable tagsError={tagsError}
                                            tags={tags}
                                            sessionId={sessionId}/>
                </div>;
    }
}

SessionCardBilling.contextType = SessionContext;
