import * as React from 'react';
import ReactTooltip from 'react-tooltip';

import SessionsHistoryItem, { INSURANCE_TYPE, SessionHistoryInfoHandler } from '../../models/session';
import { UserInfoHandler } from '../../models/user';
import { Window } from '../../ui/FullModal';
import { Link } from '../../ui/Link';
import CarNumber from '../CarNumber';
import { Copy } from '../Copy';
import { initMapApi } from '../MainMap/utils';
import { SessionBadges } from '../SessionsList/SessionItem/badges';
import { sessionTooltipId } from '../SessionsList/sessionTooltipId';
import * as styles from './index.css';
import SessionCoordinatesBlock from './SessionCoordinatesBlock';

export const SessionCardHeader = (props: { session: SessionsHistoryItem; tagsObject: Record<string,any> }) => {
    const { session, tagsObject } = props;

    const [isDetailsOpen, openDetails] = React.useState(false);

    const userDetails = SessionHistoryInfoHandler.getUserDetails.call(session);
    const userId = UserInfoHandler.getId.call(userDetails);
    const userName = `${UserInfoHandler.getPrintNameWithoutPn.call(userDetails)}`;

    const carId = SessionHistoryInfoHandler.getCarId.call(session);
    const sessionId = SessionHistoryInfoHandler.getSessionId.call(session);

    const insurance = SessionHistoryInfoHandler.getInsurance.call(session);
    const isLongTerm = SessionHistoryInfoHandler.isLongTerm.call(session);

    return (
        <div className={styles.header}>
            <div>
                <h3>
                    <span className={styles.meta}>Пользователь:</span>
                    <Link className={styles.link}
                          href={`${location.pathname}#/clients/${userId}`}
                          target={'_blank'}>
                        {userName}
                    </Link>
                </h3>
                <h3>
                    <span className={styles.meta}>Автомобиль:</span>
                    <CarNumber carInfo={SessionHistoryInfoHandler.getCar.call(session)}/>
                    <Link className={`${styles.link} ${styles.link_car}`}
                          href={`#/cars/${carId}/info`}
                          target={'_blank'}>
                        {SessionHistoryInfoHandler.getModelName.call(session)}
                    </Link>
                </h3>
                <h3>
                    <span className={styles.meta}>Сессия: </span>
                    <Copy internalTooltipId={'session_id'}>{SessionHistoryInfoHandler.getSessionId.call(session)}</Copy>
                </h3>
                {
                    isLongTerm
                        ? <div>
                            Полная страховка: {!isLongTerm?.Franchise ? <strong>ДА</strong> : 'НЕТ'}
                        </div>
                        : <h4>
                            <span className={styles.meta}>Страховка:</span>
                            <span className={`${styles.insurance_type} ${styles[insurance]}`}>
                                {INSURANCE_TYPE[insurance] || insurance}
                            </span>
                        </h4>
                }
                <div>
                    <ReactTooltip id={sessionTooltipId} html type={'info'}/>
                    <SessionBadges index={0} sessionsData={session} tagsObject={tagsObject}/>
                </div>
            </div>
            <div>
                <h5>
                    <Link className={styles.link}
                          href={`${location.pathname}#/clients/${userId}/bonuses`}
                          target={'_blank'}>
                        Бонуcы пользователя
                    </Link>
                    <span> | </span>
                    <Link className={styles.link}
                          href={`#/tracks?session_id=${sessionId}&car_id=${carId}&status=`}
                          target={'_blank'}>
                        Маршрут
                    </Link>
                </h5>
                <h5>
                    <span className={styles.meta}>Город:</span>
                    {SessionHistoryInfoHandler.getCity.call(session)}
                </h5>

                <SessionCoordinatesBlock session={session} isSessionStart={true}/>

                <SessionCoordinatesBlock session={session} isSessionStart={false}/>

                <h5>
                    <span className={styles.meta}>Длительность:</span>
                    {SessionHistoryInfoHandler.getDuration.call(session)}
                </h5><h5>
                    <span className={styles.meta}>Расстояние:</span>
                    {SessionHistoryInfoHandler.getHrMileage.call(session)}
                </h5>
                <div><Link onClick={openDetails.bind(null, true)}>Детали тарифа</Link></div>
                {isDetailsOpen && <OfferDetails data={SessionHistoryInfoHandler.getOfferDetails.call(session)}
                                                onClose={openDetails.bind(null, false)}/>}
            </div>
        </div>
    );
};

export const OfferDetails = (props) => {
    return <Window onClose={props.onClose} title={'Детали тарифа'} className={styles.offer_details_modal}>
        <iframe srcDoc={props.data} className={styles.offer_details_modal_iframe}/>
    </Window>;
};

export const Address = React.memo((props: { latitude; longitude }) => {
    const [address, setAddress] = React.useState(null);
    // @ts-ignore
    const ymaps = window.ymaps;
    if (!ymaps) {
        const mapScript = initMapApi();
        mapScript.addEventListener('load', () => {
            // @ts-ignore
            window.ymaps.ready(() => {
                // @ts-ignore
                getAddress(window?.ymaps, [props.longitude, props.latitude])
                    .then(address => {
                        setAddress(address);
                    });
            });
        });
    } else {
        getAddress(ymaps, [props.longitude, props.latitude]).then(address => {
            setAddress(address);
        });
    }

    return <div className={styles.address}>{address || '...'}</div>;
});

const getAddress = (ymaps, coords) => {
    return ymaps?.geocode?.(coords)
        ?.then(res => {
            const firstGeoObject = res.geoObjects.get(0);

            const address = `${firstGeoObject.getLocalities().length
                ? firstGeoObject.getLocalities()
                : firstGeoObject.getAdministrativeAreas()},`
                + ` ${firstGeoObject.getThoroughfare() || firstGeoObject.getPremise() || ''}`
                + ` ${firstGeoObject.getPremiseNumber() || ''}`;

            return address;
        });
};
