import * as React from 'react';

import { Dict } from '../../../types';
import { ONE_SECOND } from '../../constants';
import FormatDate from '../../ui/FormatDate';
import { JsonModal } from '../../ui/FullModal/JsonModal';
import { Link } from '../../ui/Link';
import tblStyles from '../../ui/Table/index.css';
import { Request2 } from '../../utils/request';
import { SimpleError } from '../SimpleError';
import Spin from '../Spin';
import { REQUESTS, SESSION_REQUESTS } from './request';

interface IProps {
    sessionId: string;
}

interface IState {
    isLoading: boolean;
    error: null | Error;
    data: any[];
    selectedTag: Dict<any> | null;
}

export default class SessionTagList extends React.Component<IProps, IState> {
    state: IState = {
        isLoading: false,
        error: null,
        data: [],
        selectedTag: null,
    };
    request = new Request2({ requestConfigs: SESSION_REQUESTS });

    componentDidMount() {
        this.getData();
    }

    componentDidUpdate(prevProps): void {
        if (this.props.sessionId !== prevProps.sessionId) {
            this.getData();
        }
    }

    getData() {
        const session_id = this.props.sessionId;

        session_id && this.setState({
            isLoading: true,
        }, () => {
            this.request.exec(REQUESTS.TAG_LIST, { queryParams: { object_id: session_id } })
                .then(response => {
                    this.setState({
                        data: response?.records || [],
                        isLoading: false,
                    });
                })
                .catch(error => {
                    this.setState({ error, isLoading: false });
                });
        });
    }

    showTagDetails(selectedTag) {
        this.setState({ selectedTag });
    }

    render() {
        return <>
            {
                this.state.error && <SimpleError error={this.state.error}/>
            }
            {
                this.state.selectedTag !== null
                    ? <JsonModal obj={this.state.selectedTag}
                                 title={`информация о теге ${this.state.selectedTag?.tag}`}
                                 onClose={this.showTagDetails.bind(this, null)}/> : null
            }
            {
                this.state.isLoading ? <Spin/> :
                    <>
                        <h4>Фидбек (теги на сессии) {!this.state.data.length && 'не найден'}</h4>
                        {this.state.data.length ? <table className={tblStyles.table}>
                            <thead>
                                <tr>
                                    <th>#</th>
                                    <th>дата</th>
                                    <th>описание</th>
                                    <th>комментарий</th>
                                    <th/>
                                </tr>
                            </thead>
                            <tbody>
                                {
                                    this.state.data.map((el, index) => {
                                        return <tr key={index}>
                                            <td>{index + 1}</td>
                                            <td>
                                                <FormatDate value={(el.Timestamp || el.timestamp) * ONE_SECOND}
                                                            withSecond/>
                                            </td>
                                            <td>{el.display_name}</td>
                                            <td>{el.Comment || el.comment}</td>
                                            <td><Link onClick={this.showTagDetails.bind(this, el)}> info </Link></td>
                                        </tr>;
                                    })
                                }
                            </tbody>
                        </table> : ''
                        }
                    </>
            }
        </>;
    }
}
