import * as React from 'react';
import Scrollbars from 'react-custom-scrollbars';

import { ITagDescriptionsPropsExport, TagDescriptions } from '../../../decorators/TagDescriptions';
import SessionsHistoryItem from '../../models/session';
import { Cross } from '../../ui/Cross';
import coreStyles from '../../ui/index.css';
import { hasWebphonePanel } from '../../utils/hasWebphonePanel';
import { Request2 } from '../../utils/request';
import { TabView } from '../../utils/TabView';
import { IStore } from '../App/store';
import { VisitedLog } from '../Content/db';
import { SimpleError } from '../SimpleError';
import Spin from '../Spin';
import styles from './index.css';
import { REQUESTS, SESSION_REQUESTS } from './request';
import { SessionCardHeader } from './SessionCardHeader';
import SessionCardSidebar from './SessionCardSidebar';

interface ISessionCardProps extends IStore, ITagDescriptionsPropsExport{
    sessionId: string;
    tabs: any;
    t: any;
    uri: string;
    location?: any;
}

interface ISessionCardState {
    session: SessionsHistoryItem;
    isLoading: boolean;
    error: Error | null;
}

export const SessionContext = React.createContext({});

@TagDescriptions()
export class SessionCard extends React.Component<ISessionCardProps, ISessionCardState> {
    state: ISessionCardState = {
        session: {},
        isLoading: false,
        error: null,
    };

    visitedLog = new VisitedLog();
    request = new Request2({ requestConfigs: SESSION_REQUESTS });

    getData() {
        const { sessionId } = this.props;
        this.props?.getTags?.();

        this.setState({
            isLoading: true,
        }, () => {
            this.request.exec(REQUESTS.GET_SESSION, {
                queryParams: {
                    session_id: sessionId,
                },
            })
                .then((response) => {
                    const session = {
                        ...response,
                    };

                    this.setState({
                        session,
                        isLoading: false,
                    }, () => this.visitedLog.saveLog(this.state.session, this.props));
                })
                .catch((error) => {
                    this.setState({
                        error,
                        isLoading: false,
                    });
                });
        });
    }

    onClose() {
        const queries = new URLSearchParams(this.props.location
            && this.props.location.search);

        if (queries && queries.get('back')) {
            location.hash = queries.get('back')!;
        } else {
            location.hash = '#';
        }
    }

    componentDidUpdate(prevProps: Readonly<ISessionCardProps>) {
        if (prevProps.sessionId !== this.props.sessionId) {
            this.getData();
        }
    }

    componentDidMount() {
        this.getData();
    }

    render() {
        const { tabs, t, uri, children } = this.props;
        const { session, isLoading, error } = this.state;
        const hasWebphone = hasWebphonePanel(location.href, this.props?.AdminUser?.blockRules);

        return (
            <div className={coreStyles.card_modal}>
                <div className={coreStyles.background}
                     onClick={this.onClose.bind(this)}/>
                <div className={`${coreStyles.card_container} ${!hasWebphone ? coreStyles.hasNoWebphone : ''}`}>
                    {error ? <SimpleError error={error}/> : isLoading ?
                        <Spin/>
                        : (
                            <div className={`${coreStyles.card} ${styles.card}`}>
                                <div className={coreStyles.header}>
                                    <SessionCardHeader tagsObject={this.props.tagsObject}
                                                       session={session}/>
                                </div>
                                <div className={coreStyles.tabs}>
                                    <TabView withoutTitle
                                             tabs={tabs}
                                             t={t}
                                             uri={uri}/>
                                </div>
                                <div className={coreStyles.sidebar}>
                                    <SessionCardSidebar session={session} />
                                </div>
                                <div className={coreStyles.content}>
                                    <Scrollbars autoHide>
                                        <SessionContext.Provider value={session}>
                                            {children}
                                        </SessionContext.Provider>
                                    </Scrollbars>
                                </div>
                            </div>
                        )}
                    <Cross onClick={this.onClose.bind(this)} className={coreStyles.close_icon}/>
                </div>
            </div>
        );
    }
}
