import * as React from 'react';

import { ONE_SECOND } from '../../../constants';
import SessionsHistoryItem, { SessionHistoryInfoHandler } from '../../../models/session';
import { Button, ButtonTypes } from '../../../ui/Button';
import DatePicker from '../../../ui/DatePicker';
import { Window } from '../../../ui/FullModal';
import { Request2 } from '../../../utils/request';
import { rubs } from '../../../utils/rubs';
import { BillingViewRefundModal } from '../../Clients/BillingView/BillingViewRefundModal';
import Spin from '../../Spin';
import { REQUESTS, SESSION_REQUESTS } from '../request';
import * as styles from './BillingCalculator.css';

interface IBillingCalculatorProps {
    session: SessionsHistoryItem;
    onClose: () => void;
    since?: number;
    until?: number;
    BlockRules?: Record<string,any>;
}

interface IBillingCalculatorState {
    [x: number]: number;

    error: Error | null;
    isLoading: boolean;
    showRefundModal: boolean;
    price?: number;
    since: number;
    until: number;
}

export class BillingCalculator extends React.Component<IBillingCalculatorProps, IBillingCalculatorState> {
    state: IBillingCalculatorState = {
        error: null,
        isLoading: false,
        showRefundModal: false,
        since: this.props.since || SessionHistoryInfoHandler.getStart.call(this.props.session),
        until: this.props.until || SessionHistoryInfoHandler.getFinish.call(this.props.session),
    };

    request = new Request2({ requestConfigs: SESSION_REQUESTS });

    onChange(key, value) {
        this.setState({
            [key]: value,
        });
    }

    componentDidMount() {
        if (this.props.since || this.props.until) {
            this.calculate();
        }
    }

    calculate() {
        const { session } = this.props;
        const { since, until } = this.state;
        const sessionId = SessionHistoryInfoHandler.getSessionId.call(session);
        const isCompiled = SessionHistoryInfoHandler.isSessionCompiled.call(session) ? 1 : 0;

        this.setState({
            isLoading: true,
        }, () => {
            this.request.exec(REQUESTS.GET_CALCULATIONS, {
                queryParams: {
                    use_compiled: isCompiled,
                },
                body: {
                    session_id: sessionId,
                    since: +(+since / ONE_SECOND).toFixed(0),
                    until: +(+until / ONE_SECOND).toFixed(0),
                },
            })
                .then((response) => {
                    const { price } = response;

                    this.setState({
                        price,
                        isLoading: false,
                    });
                })
                .catch((error) => {
                    this.setState({
                        error,
                        isLoading: false,
                    });
                });
        });
    }

    showRefundModal() {
        this.setState((prev) => ({
            showRefundModal: !prev.showRefundModal,
        }));
    }

    render() {
        const { error, isLoading, price, showRefundModal, since, until } = this.state;
        const { onClose, session } = this.props;

        const userId = SessionHistoryInfoHandler.getUserId.call(session);
        const sessionId = SessionHistoryInfoHandler.getSessionId.call(session);

        return (
            <>
                {showRefundModal && price && (
                    <BillingViewRefundModal amount={0}
                                            userId={userId}
                                            payment={{
                                                session_id: sessionId,
                                            }}
                                            onClose={this.showRefundModal.bind(this)}/>
                )}
                <Window onClose={onClose}
                        title={'Калькулятор сессий'}
                        error={error}>
                    <div className={styles.dates_container}>
                        <DatePicker className={styles.input}
                                    placeholder={'C'}
                                    value={since}
                                    onChange={this.onChange.bind(this, 'since')}
                                    timeFormat={'HH:mm:ss'}/>
                        <DatePicker className={styles.input}
                                    placeholder={'По'}
                                    value={until}
                                    onChange={this.onChange.bind(this, 'until')}
                                    timeFormat={'HH:mm:ss'}/>
                        <Button onClick={this.calculate.bind(this)}>Рассчитать</Button>
                    </div>
                    {isLoading && (
                        <Spin/>
                    )}
                    {!(price === undefined) && (
                        <div className={styles.result_container}>
                            <p>Заплачено за период:&nbsp;
                                <strong>{rubs(price)}</strong>
                            </p>
                            {
                                (price && this.props.BlockRules?.Refound)
                                    ? <Button onClick={this.showRefundModal.bind(this)}
                                              colorType={ButtonTypes.negative}>Вернуть</Button>
                                    : undefined}
                        </div>
                    )}
                </Window>
            </>
        );
    }
}
