import * as React from 'react';
import { connect } from 'react-redux';

import { Dict } from '../../../../types';
import { ONE_SECOND } from '../../../constants';
import { Button } from '../../../ui/Button';
import { Confirm, Window } from '../../../ui/FullModal';
import { Link } from '../../../ui/Link';
import { isElementHasScroll } from '../../../utils/hasElementScroll';
import { Request2 } from '../../../utils/request';
import { FormConstructor } from '../../FormConstructor';
import { ISchemaItem } from '../../FormConstructor/types';
import Spin from '../../Spin';
import { REQUESTS, SESSION_REQUESTS } from '../request';
import * as styles from './EvacuationForm.css';

interface IEvacuationFormProps {
    onClose: () => void;
    initialData: any;
}

interface IEvacuationFormState {
    isLoadingSchema: boolean;
    schemaLoadingError: Error | null;
    schema: Dict<ISchemaItem> | null;

    isLoadingComments: boolean;
    commentsLoadingError: Error | null;
    comments: string[];

    isSendingForm: boolean;
    formSendingError: Error | null;

    formData: Dict<any>;
}

export class EvacuationFormInternal extends React.Component<IEvacuationFormProps, IEvacuationFormState> {
    state: IEvacuationFormState = {
        isLoadingSchema: false,
        schemaLoadingError: null,
        schema: null,

        isLoadingComments: false,
        commentsLoadingError: null,
        comments: [],

        isSendingForm: false,
        formSendingError: null,

        formData: {},
    };
    request = new Request2({ requestConfigs: SESSION_REQUESTS });
    modal: React.RefObject<any>;

    constructor(props) {
        super(props);
        this.modal = React.createRef();
    }

    componentDidMount() {
        const { initialData } = this.props;

        this.setState({ isLoadingSchema: true, schemaLoadingError: null }, () => {
            this.request.exec(REQUESTS.GET_SCHEME_REPORT_EVACUATION_PLUS, { queryParams: { scheme: true } })
                .then(response => {
                    const schema = response?.request_data?.structure ?? null;
                    this.setState({ isLoadingSchema: false, schema });
                })
                .catch(schemaLoadingError => {
                    this.setState({ isLoadingSchema: false, schemaLoadingError });
                });
        });

        const car_id = initialData?.car_id;

        if (car_id) {
            this.setState({ isLoadingComments: true, commentsLoadingError: null }, () => {
                this.request.exec(REQUESTS.GET_CAR_TAGS, { queryParams: { car_id } })
                    .then(response => {
                        const tags = response?.records ?? [];
                        const evacuationTags = tags.filter(tag => tag.tag === 'evacuation');
                        const comments = evacuationTags.map(evacuationTag => evacuationTag.comment);

                        this.setState({ isLoadingComments: false, comments });
                    })
                    .catch(commentsLoadingError => {
                        this.setState({ isLoadingComments: false, commentsLoadingError });
                    });
            });
        }
    }

    sendFormData() {
        const { onClose } = this.props;
        const { formData } = this.state;

        const orderEndTime = formData?.session?.hasOwnProperty('order_end_time')
            && formData?.session?.order_end_time;

        if (orderEndTime) {
            formData.session.order_end_time = new Date(formData?.session?.order_end_time * ONE_SECOND)
                ?.toLocaleString();
        }

        this.setState({ isSendingForm: true, formSendingError: null }, () => {
            this.request.exec(REQUESTS.REPORT_EVACUATION_PLUS, { body: formData })
                .then(() => {
                    this.setState({ isSendingForm: false });
                    onClose();
                })
                .catch(formSendingError => {
                    this.setState({ isSendingForm: false, formSendingError });
                });
        });
    }

    closeWarning() {
        this.setState({ comments: [] });
    }

    changeForm(formData: Dict<any>) {
        this.setState({ formData });
    }

    render() {
        const { onClose, initialData } = this.props;
        const {
            comments, schema,
            isLoadingSchema, schemaLoadingError,
            isLoadingComments, commentsLoadingError,
            isSendingForm, formSendingError,
        } = this.state;

        const windowBody = this.modal?.current?.containerRef?.current;
        const submitButton = <Button className={styles.button}
                                     onClick={this.sendFormData.bind(this)}
                                     isLoading={isSendingForm}>Отправить</Button>;

        return <Window onClose={onClose.bind(this)}
                       error={schemaLoadingError || formSendingError}
                       title={'Эвакуация'}
                       ref={this.modal}>
            {comments?.length
                ? <Confirm title={'Тикеты'}
                           onClose={onClose.bind(this)}
                           error={commentsLoadingError}
                           accept={this.closeWarning.bind(this)}
                           isWorking={isLoadingComments}
                           question={
                               <div>
                                   {...comments.map(url =>
                                       <div key={url}>
                                           <Link href={url} target={'_blank'}>{url}</Link>
                                       </div>,
                                   )}
                               </div>
                           }/>
                : null
            }
            {isLoadingSchema
                ? <Spin/>
                : <>
                    {
                        isElementHasScroll(windowBody)
                            ? submitButton
                            : null
                    }
                    <FormConstructor schema={schema ?? {}}
                                     onChange={this.changeForm.bind(this)}
                                     initialData={initialData}/>
                    {submitButton}
                </>}
        </Window>;
    }
}

export const EvacuationForm = connect(() => {
    return {};
})(EvacuationFormInternal);
