import * as React from 'react';

import { EMPTY_DATA } from '../../../constants';
import { IOfferProto, ISessionBill } from '../../../models/session';
import { Confirm } from '../../../ui/FullModal';
import { Input } from '../../../ui/Input';
import { Request2 } from '../../../utils/request';
import { rubs } from '../../../utils/rubs';
import { REQUESTS, SESSION_REQUESTS } from '../request';
import * as styles from './FixCompensateModal.css';

interface IFixCompensateModalProps {
    onClose: () => void;
    offerProto: IOfferProto;
    bill: ISessionBill[];
    userId: string;
    carNumber: string;
    sessionId: string;
}

export const FixCompensateModal = (props: IFixCompensateModalProps) => {
    const { onClose, bill, offerProto, userId, carNumber, sessionId } = props;
    const [error, setError] = React.useState<Error | null>(null);
    const [isLoading, setLoading] = React.useState(false);
    const [link, onChangeLink] = React.useState('');
    const request = new Request2({ requestConfigs: SESSION_REQUESTS });

    let compensationInCoins = 0;
    const offerProtoPrice = offerProto?.PackOffer?.PackPrice;
    const totalInCoins = bill.find((bill) => bill?.type === 'total')?.cost;
    const parkingInCoins = bill.find((bill) => bill?.type === 'old_state_parking')?.cost ?? 0;
    const reservationInCoins = bill.find((bill) => bill?.type === 'old_state_reservation')?.cost ?? 0;

    if (totalInCoins && offerProtoPrice) {
        compensationInCoins = totalInCoins - parkingInCoins - reservationInCoins - offerProtoPrice;
    } else {
        setError(new Error('Невозможно рассчитать компенсацию'));
    }

    const accept = (compensationInCoins: number) => {
        const body = {
            amount: compensationInCoins,
            comment: 'Пересчитываем фикс из минут',
            car_number: carNumber,
            object_id: userId,
            tag: 'bb_service_problem',
            priority: 0,
            session_id: sessionId,
            links: [{ uri: link, type: 'other' }],
        };

        setLoading(true);
        request.exec(REQUESTS.ATTACH_USER_TAG, {
            body,
        })
            .then(() => {
                setLoading(false);
                onClose();
            })
            .catch((error) => {
                setLoading(false);
                setError(error);
            });
    };

    const question = <div>
        {/* eslint-disable-next-line max-len */}
        <p>Компенсировать пересчет в минутный тариф из-за того, что пользователь завершил на границе зеленой зоны и тариф пересчитался из-за погрешности?</p>
        <p><strong>Итого:</strong> {totalInCoins ? rubs(totalInCoins) : EMPTY_DATA}</p>
        <p><strong>Бронь: </strong> {rubs(reservationInCoins)}</p>
        <p><strong>Ожидание: </strong> {rubs(parkingInCoins)}</p>
        <p><strong>Первоначальная стоимость: </strong> {rubs(offerProtoPrice)}</p>
        <p><strong>Начислить бонусов:</strong> {compensationInCoins ? rubs(compensationInCoins) : EMPTY_DATA}</p>
        <Input placeholder={'Ссылка на тикет/чат'}
               value={link}
               onChange={onChangeLink.bind(null)}/>
    </div>;

    return <Confirm isWorking={isLoading}
                    className={styles.fixCompensateModal}
                    title={'Компенсация пересчета в минутный тариф'}
                    question={question}
                    error={error}
                    accept={accept.bind(null, compensationInCoins)}
                    acceptTitle={`Начислить ${rubs(compensationInCoins)} бонусов`}
                    onClose={onClose.bind(null)}/>;
};
