import * as React from 'react';
import ReactTooltip from 'react-tooltip';

import { EMPTY_DATA, ONE_SECOND } from '../../../constants';
import { TagRecord, TagRecordHandler } from '../../../models/tag';
import { Button } from '../../../ui/Button';
import FormatDate, { FormatDateInString } from '../../../ui/FormatDate';
import { Window } from '../../../ui/FullModal';
import { Input } from '../../../ui/Input';
import { Link } from '../../../ui/Link';
import Select from '../../../ui/Select';
import * as tblStyles from '../../../ui/Table/index.css';
import TextArea from '../../../ui/TextArea';
import { isValidJSONString } from '../../../utils/isValidJSONString';
import { Request2 } from '../../../utils/request';
import CarNumber from '../../CarNumber';
import { ShowTagDetails } from '../../Clients/UserInfoView/ShowTagDetails';
import Spin from '../../Spin';
import { REQUESTS, SESSION_REQUESTS } from '../request';
import * as styles from './ViolationsForm.css';

interface IViolationsFormProps {
    onClose: () => void;
    userId: string;
    sessionId: string;
}

interface IViolationsFormState {
    isLoading: boolean;
    error: Error | null;
    tags: TagRecord[];
    allTags: any;
    currentTags: any;
    sessions: any;
    selectedTag: string[];
    selectedSession: string[];
    userComment: string;

    isWorking: boolean;
    score: number;
    links: string[];

    showTagDetails: boolean;
    tagData: any;

    [key: string]: any;
}

const linksCount = 2;

export class ViolationsForm extends React.Component<IViolationsFormProps, IViolationsFormState> {
    state: IViolationsFormState = {
        isLoading: false,
        error: null,
        tags: [],
        allTags: [],
        sessions: [],
        selectedTag: [],
        selectedSession: [],
        currentTags: [],
        userComment: '',
        score: 0,
        links: [],
        isWorking: false,
        showTagDetails: false,
        tagData: {},
    };

    request = new Request2({ requestConfigs: SESSION_REQUESTS });

    getData() {
        const { userId, sessionId } = this.props;

        userId && sessionId && this.setState({
            isLoading: true,
        }, () => {
            Promise.all([
                this.request.exec(REQUESTS.GET_SESSION, {
                    queryParams: {
                        user_id: userId,
                    },
                }),
                this.request.exec(REQUESTS.GET_USER_HISTORY_TAGS, {
                    queryParams: {
                        object_id: userId,
                        duration: '90d',
                    },
                }),
                this.request.exec(REQUESTS.GET_TAGS),
                this.request.exec(REQUESTS.GET_USER_TAGS, {
                    queryParams: {
                        object_id: userId,
                    },
                }),
            ])
                .then(response => {
                    const TAGS_RESPONSE = 2;
                    const USERS_TAGS_RESPONSE = 3;

                    const _constBonus = response[TAGS_RESPONSE].records
                        && response[TAGS_RESPONSE].records.reduce((_p: string[], _c: any) => {
                            if (_c.type === 'user_problem_tag' && _c.name) {
                                return [..._p, _c.name];
                            }

                            return _p;

                        }, []) || [];

                    const tags = response[1].records
                        && response[1].records.reduce((_p: TagRecord[], _c: TagRecord) => {
                            if (_constBonus.includes(TagRecordHandler.getTagName.call(_c))) {
                                return [..._p, _c];
                            }

                            return _p;
                        }, []) || [];

                    const allTags = response[TAGS_RESPONSE].records
                        && response[TAGS_RESPONSE].records
                            .filter((_tag: any) => _constBonus.includes(_tag.name))
                            .reduce((_p: any, _c: any) => {
                                let _meta = _c.meta;
                                if (_meta) {
                                    _meta = isValidJSONString(_meta)
                                        ? JSON.parse(_meta)
                                        : typeof(_meta) === 'object' ? _meta : {};
                                }

                                return Object.assign({}, _p, {
                                    [_c.name || '']: {
                                        ..._c,
                                        _meta,
                                    },
                                });
                            }, {}) || [];

                    const currentTags = response[USERS_TAGS_RESPONSE].records
                        && response[USERS_TAGS_RESPONSE].records.reduce((_p: any[], _c: any) => {
                            if (_constBonus.includes(_c?.tag)) {
                                return [..._p, _c];
                            }

                            return _p;
                        }, []) || [];
                    this.setState({
                        isLoading: false,
                        error: null,
                        tags,
                        allTags,
                        currentTags,
                        sessions: response[0].sessions || [],
                        selectedSession: [sessionId] || [],
                        score: currentTags.reduce((_p: number, _c: any) => {
                            const _tag = allTags[_c.tag] || {};
                            if (_tag._meta && _tag._meta.point) {
                                _p += _tag._meta.point;
                            }

                            return _p;
                        }, 0),
                    });
                })
                .catch((error) => {
                    this.setState({
                        error,
                        isLoading: false,
                    });
                });
        });
    }

    componentDidMount() {
        this.getData();
    }

    onSelectTag(value: string) {
        let obj: any = null;
        const tag = this.state.allTags[value?.toString()] || {};
        if (tag._meta && tag._meta && tag._meta.amount) {
            obj = {
                amount: tag._meta.amount,
                disabledAmount: true,
                selectedTag: value ? [value] : [],
            };
        }

        if (obj) {
            this.setState(obj);
        } else {
            this.setState({ selectedTag: value ? [value] : [], disabledAmount: false });
        }
    }

    onSelectSession(value: string) {
        this.setState({ selectedSession: value ? [value] : [] });
    }

    onChange(key: string, e: any) {
        this.setState({
            [key]: e,
        });
    }

    onChangeLinks(index: number, value: any) {
        const links = this.state.links;
        links[index] = value;
        this.setState({ links });
    }

    showTagDetails(tagData: any, e: KeyboardEvent) {
        this.setState({
            showTagDetails: true,
            tagData,
        });
    }

    closeTagDetails() {
        this.setState({
            showTagDetails: false,
            tagData: {},
        });
    }

    accept() {
        const { userId, onClose } = this.props;
        const { userComment, selectedSession, selectedTag, links } = this.state;
        const tag = this.state.allTags[this.state.selectedTag.toString()] || {};
        const session = this.state.sessions.filter((_el: any) => _el.segment
            && _el.segment.meta && _el.segment.meta.session_id == this.state.selectedSession)[0];
        const car = session && session.car || {};

        this.setState({ isWorking: true }, () => {
            this.request.exec(REQUESTS.ATTACH_USER_TAG, {
                body: {
                    'car_number': car.number,
                    'comment': userComment,
                    'object_id': userId,
                    'priority': tag.default_priority && +tag.default_priority || 0,
                    'session_id': selectedSession.toString(),
                    'tag': selectedTag.toString(),
                    'links': links.map(uri => {
                        return { type: 'other', uri };
                    }),
                },
            })
                .then(() => {
                    this.setState({ isWorking: false }, () => {
                        onClose();
                    });
                })
                .catch((error) => {
                    this.setState({
                        isWorking: false,
                        error,
                    });
                });
        });
    }

    render() {
        const {
            error,
            score,
            isLoading,
            allTags,
            selectedTag,
            sessions,
            selectedSession,
            links,
            tags,
        } = this.state;
        const { onClose } = this.props;

        return (
            <Window error={error}
                    title={'Нарушение'}
                    onClose={onClose.bind(this)}
                    className={styles.modal}>
                <div className={styles.container}>
                    {this.state.showTagDetails && <ShowTagDetails onClose={this.closeTagDetails.bind(this)}
                                                                  tagData={this.state.tagData}/>
                    }
                    {isLoading ? <Spin/> : (
                        <>
                            <div>
                                <h4>Накоплено баллов: {score}</h4>
                                <div>
                                    <Select placeholder={'Причина'}
                                            options={Object.entries(allTags).map((_el: any[]) => {
                                                return {
                                                    text: _el[1].display_name || _el[1].name,
                                                    value: _el[0],
                                                    description: _el[1].comment,
                                                };
                                            })}
                                            onSelect={this.onSelectTag.bind(this)}
                                            initialValues={selectedTag}/>
                                    <div>
                                        {allTags[selectedTag.toString()]
                                        && allTags[selectedTag.toString()].comment || EMPTY_DATA}
                                    </div>
                                    <Select placeholder={'Поездка'}
                                            options={[{ text: EMPTY_DATA, value: null }]
                                                .concat(sessions.map((_el: any) => {
                                                    const car = _el.car && _el.car.number || EMPTY_DATA;
                                                    const { session_id = '', start = 0, finish = 0 } = _el.segment
                                                && _el.segment.meta
                                                || _el.segment;

                                                    const text = `${FormatDateInString({
                                                        value: start * ONE_SECOND,
                                                        onlyDate: true,
                                                    })} `
                                                    + `${FormatDateInString({
                                                        value: start * ONE_SECOND,
                                                        onlyTime: true,
                                                    })}  -`
                                                    + `${FormatDateInString({
                                                        value: finish * ONE_SECOND,
                                                        onlyTime: true,
                                                    })}, ${car}`;

                                                    return {
                                                        text,
                                                        value: session_id,
                                                    };
                                                }))}
                                            onSelect={this.onSelectSession.bind(this)}
                                            initialValues={selectedSession}/>
                                    <TextArea placeholder={'Комментарий'}
                                              value={this.state.userComment}
                                              onChange={this.onChange.bind(this, 'userComment')}>
                                    </TextArea>
                                    {new Array(linksCount).fill('')
                                        .map((el: any, index: number) => {
                                            return (
                                                <Input value={links[index]}
                                                       key={index}
                                                       className={'link-item'}
                                                       placeholder={`Ссылка на фото или видео #${index}`}
                                                       onChange={this.onChangeLinks.bind(this, index)}/>
                                            );
                                        })
                                    }

                                    <div className={styles.buttonContainer}>
                                        <Button isLoading={this.state.isWorking}
                                                onClick={onClose.bind(this)}
                                                basic>Отмена</Button>
                                        <Button isLoading={this.state.isWorking}
                                                onClick={this.accept.bind(this)}>Применить</Button>
                                    </div>
                                </div>
                            </div>
                            <div>
                                <div className={styles.currentTags}>
                                    <h4>Активные теги</h4>
                                    <table className={`${tblStyles.table} ${tblStyles.clearHover}`}>
                                        <thead>
                                            <tr>
                                                <th>#</th>
                                                <th>Тег/Описание</th>
                                                <th>Комментарий</th>
                                                <th colSpan={2}/>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            {this.state.currentTags.map((item: any, index: number) => {
                                                return (
                                                    <tr key={index}>
                                                        <td>{index + 1}</td>
                                                        <td>
                                                            <strong>{item.display_name}</strong>
                                                            <div>
                                                                {item.tag !== item.display_name ? item.tag : ''}
                                                            </div>
                                                        </td>
                                                        <td>{item.comment}</td>
                                                        <td>
                                                            {item.session_id && (
                                                                <Link href={`#/session/${item.session_id}`}>
                                                                сессия
                                                                </Link>
                                                            ) || EMPTY_DATA}
                                                        </td>
                                                        <td>
                                                            <Link onClick={this.showTagDetails.bind(this, item)}>
                                                            info
                                                            </Link>
                                                        </td>
                                                    </tr>
                                                );
                                            })}
                                        </tbody>
                                    </table>
                                </div>
                                <div className={styles.history}>
                                    <h4>История</h4>
                                    <table className={tblStyles.table}>
                                        <tbody>
                                            {tags.map((item, index) => {
                                                const tagName = TagRecordHandler.getTagName.call(item);
                                                const { point = 0 } = allTags[tagName]
                                            && allTags[tagName]._meta || {};
                                                const tagId = TagRecordHandler.getTagId.call(item);
                                                const carNumber = TagRecordHandler.getTagCarNumber.call(item);

                                                return [
                                                    <tr key={tagId + '_0'}>
                                                        <td className={'index'}>
                                                            {index + 1}
                                                        </td>
                                                        <td>{TagRecordHandler.getAction.call(item)}:</td>
                                                        <td>
                                                            <FormatDate withSecond={true}
                                                                        value={TagRecordHandler.getTagTimestamp
                                                                            .call(item)}/>
                                                        </td>
                                                        <td>{point} баллов</td>
                                                        <td>
                                                            {carNumber
                                                        && <CarNumber value={carNumber}/>
                                                        || EMPTY_DATA}
                                                        </td>
                                                    </tr>,
                                                    <tr key={tagId + '_1'} className={'sub_row'}>
                                                        <td colSpan={2}/>
                                                        <td data-tip
                                                            colSpan={3}
                                                            data-for={tagId}>
                                                            <strong>
                                                                {TagRecordHandler.getTagDisplayName.call(item)
                                                            || tagName}
                                                            </strong>
                                                            <ReactTooltip id={tagId}
                                                                          aria-haspopup='true'
                                                                          place="top">
                                                                {TagRecordHandler.getTagComment.call(item)
                                                                    || EMPTY_DATA}
                                                            </ReactTooltip>
                                                        </td>
                                                    </tr>];
                                            })}
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </>
                    )}
                </div>
            </Window>
        );
    }
}
