import React, { ReactElement } from 'react';
import { renderToString } from 'react-dom/server';
import ReactTooltip from 'react-tooltip';

import { DEDICATED_FLEET_TAG, MARKET_FLEET_TAG, ONE_SECOND } from '../../../constants';
import {
    DELEGATION_INDEX,
    DELEGATION_TYPE,
    DELEGATION_TYPE_DESCRIPTION,
    InsuranceNotification,
    SessionHistoryInfoHandler,
} from '../../../models/session';
import FormatDate from '../../../ui/FormatDate';
import {
    DedicatedFleet,
    IconBLE,
    IconCorp,
    IconCountCircle,
    IconLongTerm,
    IconTaxi,
    IconVoyage,
    IconYac,
    MarketIcon,
} from '../../../ui/Icons';
import style from '../index.css';
import { sessionTooltipId } from '../sessionTooltipId';
import { ISessionBadgesProps } from './types';

export const SessionBadges = React.memo((props: ISessionBadgesProps) => {

    const { sessionsData, index, tagsObject = {}, tagIsLoading, internalTooltip } = props;
    const items: BadgesContainerItem[] = [];
    const isContainsBluetoothAction = SessionHistoryInfoHandler.isContainsBluetoothAction.call(sessionsData, index);
    const isCorp = SessionHistoryInfoHandler.isOfferCorp.call(sessionsData, index);
    const isOfferYAC = SessionHistoryInfoHandler.isOfferYAC.call(sessionsData, index);
    const traceTags = SessionHistoryInfoHandler.getTraceTags.call(sessionsData, index);
    const idFeeDropZone = SessionHistoryInfoHandler.idFeeDropZone.call(sessionsData, index);

    const transferredFrom = SessionHistoryInfoHandler.getTransferredFrom.call(sessionsData, index);
    const delegationType = SessionHistoryInfoHandler.getDelegationType.call(sessionsData, index);
    const transferredType = SessionHistoryInfoHandler.getTransferredType.call(sessionsData, index);
    const computedDelegationType = delegationType || DELEGATION_INDEX[transferredType];

    let delegationDescription = DELEGATION_TYPE_DESCRIPTION[computedDelegationType];

    const isTaxi = SessionHistoryInfoHandler.isTaxi.call(sessionsData, index);
    const isLongTerm = SessionHistoryInfoHandler.isLongTerm.call(sessionsData, index);
    const prevSessionFinish = SessionHistoryInfoHandler.getFinish.call(sessionsData, index + 1);
    const currentSessionStart = SessionHistoryInfoHandler.getStart.call(sessionsData, index);
    const currentSessionFinish = SessionHistoryInfoHandler.getFinish.call(sessionsData, index);
    const nextSessionStart = SessionHistoryInfoHandler.getStart.call(sessionsData, index - 1);
    const beforeEvolution = SessionHistoryInfoHandler.getBeforeEvolution.call(sessionsData, index);
    const insuranceNotifications: InsuranceNotification[] = SessionHistoryInfoHandler.getInsuranceNotifications
        .call(sessionsData, index);

    const isServMode = SessionHistoryInfoHandler.getServicingDuration.call(sessionsData, index);
    const voyageId = SessionHistoryInfoHandler.getVoyageId.call(sessionsData, index);
    const isDedicatedFleet = SessionHistoryInfoHandler.getFleetMark.call(sessionsData, index, DEDICATED_FLEET_TAG);
    const isMarketFleet = SessionHistoryInfoHandler.getFleetMark.call(sessionsData, index, MARKET_FLEET_TAG);

    if (prevSessionFinish && prevSessionFinish > currentSessionStart
        || nextSessionStart && nextSessionStart < currentSessionFinish) {
        items.push({
            component: <BadgeLabel label={'2x'} className={style.double}/>,
            tip: 'Возможно двойная аренда',
        });
    }

    if (isLongTerm) {
        items.push({ component: <IconLongTerm/>, tip: 'Долгосрок' });
    }

    if (isContainsBluetoothAction) {
        items.push({ component: <IconBLE/>, tip: 'BLE' });
    }

    if (isTaxi) {
        items.push({ component: <IconTaxi/>, tip: 'Яндекс.GO (такси)' });
    }

    if (isCorp) {
        if (isOfferYAC) {
            items.push({ component: <IconYac/>, tip: 'Яндекс.Плюс (кешбек)' });
        } else {
            items.push({ component: <IconCorp/>, tip: 'Корпоративная поездка' });
        }
    }

    if (idFeeDropZone) {
        items.push({
            component: <BadgeLabel label={'Min'} className={style.fee}/>,
            tip: 'Зона с минимальной стоимостью завершения',
        });
    }

    if (beforeEvolution) {
        items.push({
            component: <BadgeLabel label={'эво'} className={style.evo}/>,
            tip: 'Эволюция тарифа',
        });
    }

    if (computedDelegationType) {
        let label, className;

        if (computedDelegationType === DELEGATION_TYPE.free) {
            label = 'ПОДВ';
            className = `${style.free}`;
        } else {
            if (!transferredFrom) {
                label = 'ПЕР';
                className = `${style.delegation}`;
                delegationDescription = delegationDescription.replace('передача руля', 'передаёт руль');
            } else {
                label = 'ПРИЁМ';
                className = `${style.delegation_from}`;
                delegationDescription = delegationDescription.replace('передача руля', 'принимает руль');
            }
        }

        items.push({
            component: <BadgeLabel label={label} className={className}/>,
            tip: delegationDescription,
        });
    }

    if (traceTags?.includes('scoring_trace_tag')) {
        items.push({
            component: <BadgeLabel label={' ⚡ '} className={style.scoring_trace}/>,
            tip: 'Неаккуратная поездка',
        });
    }

    if (traceTags?.includes('speeding_trace_tag')) {
        items.push({
            component: <BadgeLabel label={'🚨'} className={style.speeding_trace_tag}/>,
            tip: 'Превышение скорости',
        });
    }

    if (isServMode) {
        items.push({
            component: <BadgeLabel label={'🛠️СР'} className={style.serv_mode}/>,
            tip: 'Сервисный режим',
        });
    }

    if (voyageId) {
        items.push({ component: <IconVoyage voyageId={voyageId}/>, tip: 'Вояж (подробная информация по клику)' });
    }

    if (isDedicatedFleet) {
        items.push({ component: <DedicatedFleet/>, tip: 'Выделенный парк' });
    }

    if (isMarketFleet) {
        items.push({ component: <MarketIcon/>, tip: ' Maркет' });
    }

    if (traceTags?.length) {
        const tip = `<b>Фидбек</b>: ${renderToString(traceTags
            .map((i, index) => <div key={index}>{tagsObject?.[i]?.comment || i}</div>))}` +
            (tagIsLoading ? `<pre>Загрузка переводов...</pre>` : '');
        items.push({ component: <IconCountCircle count={traceTags.length}/>, tip });
    }

    if (insuranceNotifications?.length) {
        const insuranceItems = insuranceNotifications.map((el, index) => {
            return <div key={index}>
                <FormatDate value={el.start * ONE_SECOND} withSecond/> - <FormatDate value={el.finish * ONE_SECOND}
                                                                                     withSecond/>
            </div>;
        });

        const tip = `<b>Страховая</b>: ${renderToString(<>{insuranceItems}</>)}`;

        items.push({
            component: <IconCountCircle className={style.insurance} count={insuranceNotifications.length}/>,
            tip,
        });
    }

    return items.length
        ? <div className={`${style.session_item_badges}`}>
            {
                internalTooltip && <ReactTooltip id={sessionTooltipId} type={'info'} html/>
            }
            <BadgesContainer items={items}
                             tooltipId={sessionTooltipId}/>
        </div>
        : null;
});

const BadgeLabel = React.memo((props: { label: string; className: string }) => {
    const { label, className } = props;

    return <span className={`${style.session_item_badges_label} ${className}`}>{label}</span>;
});

interface BadgesContainerItem {
    tip: string;
    component: ReactElement | HTMLElement;
}

const BadgesContainer = React.memo((props: { items: BadgesContainerItem[]; tooltipId?: string }) => {
    const { items, tooltipId } = props;

    return <>
        {
            items.map((item, index) => {
                return <span key={index}
                             data-for={tooltipId || sessionTooltipId}
                             data-tip={item.tip}
                             className={style.session_item_badge}>{item.component}</span>;
            })
        }
    </>;
});
