import React from 'react';
import { renderToString } from 'react-dom/server';

import { EMPTY_DATA } from '../../../constants';
import { SessionHistoryInfoHandler } from '../../../models/session';
import { UserInfoHandler } from '../../../models/user';
import { CopyIcon } from '../../../ui/CopyIcon';
import FormatDate from '../../../ui/FormatDate';
import { Link } from '../../../ui/Link';
import CarNumber from '../../CarNumber';
import { SessionBill } from '../../SessionCard/SessionBill';
import { SessionOffer } from '../../SessionCard/SessionOffer';
import style from '../index.css';
import { sessionTooltipId } from '../sessionTooltipId';
import { SessionBadges } from './badges';
import { ISessionItemProps } from './types';

export const SessionItem = (props: ISessionItemProps) => {
    const { sessionsData, index, isUser, BlockRules, showProto, _style, tagsObject, tagIsLoading } = props;

    return <div style={_style}
                className={`${style.session_item_wrapper} ${!isUser ? style.session_item_wrapper_car : ''}`}>
        <div className={style.session_item}>
            <SessionStatus index={index} sessionsData={sessionsData}/>
            <SessionStart index={index} sessionsData={sessionsData}/>
            <SessionFinish index={index} sessionsData={sessionsData}/>
            <SessionDuration index={index} sessionsData={sessionsData}/>
            <SessionCity index={index} sessionsData={sessionsData}/>
            {
                isUser
                    ? <SessionCarModel index={index} sessionsData={sessionsData}/>
                    : <SessionUserName index={index} sessionsData={sessionsData}/>
            }
            {
                isUser
                    ? <SessionCarCardLinks index={index} sessionsData={sessionsData}/>
                    : <SessionUserCardLinks index={index} sessionsData={sessionsData}/>

            }
            {
                isUser
                    ? <SessionCarNumber index={index} sessionsData={sessionsData}/>
                    : <SessionCarNumberEmpty/>

            }
            <SessionOfferName index={index} sessionsData={sessionsData} BlockRules={BlockRules}/>
            <SessionOfferControls index={index} sessionsData={sessionsData} showProto={showProto}/>
            <SessionBadges index={index}
                           sessionsData={sessionsData}
                           tagsObject={tagsObject}
                           tagIsLoading={tagIsLoading}/>
            <SessionTotalPrice index={index} sessionsData={sessionsData}/>
            <SessionMileage index={index} sessionsData={sessionsData}/>
        </div>
    </div>;
};

export const SessionStatus = React.memo((props: ISessionItemProps) => {
    const { sessionsData, index } = props;
    const isLastSession = SessionHistoryInfoHandler.getFinish.call(sessionsData, index);

    return <div className={`${style.session_item_main_status} ${!isLastSession ? style.session_item_current : ''}`}/>;
});

const SessionStart = React.memo((props: ISessionItemProps) => {
    const { sessionsData, index } = props;
    const start = SessionHistoryInfoHandler.getStart.call(sessionsData, index);

    return <div className={`${style.session_item_start}`}>{
        start && <FormatDate value={start} withSecond/>
    }</div>;
});

export const SessionFinish = React.memo((props: ISessionItemProps) => {
    const { sessionsData, index } = props;
    const finish = SessionHistoryInfoHandler.getFinish.call(sessionsData, index);
    const lastEventState = SessionHistoryInfoHandler.getLastEventsStatus.call(sessionsData, index);

    return <div className={`${style.session_item_finish}`}>{
        finish
            ? <FormatDate value={finish} withSecond/>
            : <div className={`${style.session_item_event_status}`}>{lastEventState}</div>
    }</div>;
});

export const SessionDuration = React.memo((props: ISessionItemProps) => {
    const { sessionsData, index } = props;
    const duration = SessionHistoryInfoHandler.getDuration.call(sessionsData, index);

    return <div className={`${style.session_item_duration}`}>{duration}</div>;
});

export const SessionCity = React.memo((props: ISessionItemProps) => {
    const { sessionsData, index } = props;
    const city = SessionHistoryInfoHandler.getCity.call(sessionsData, index);

    return <div className={style.session_item_city}>{city || EMPTY_DATA}</div>;
});

const SessionCarModel = React.memo((props: ISessionItemProps) => {
    const { sessionsData, index } = props;
    const model = SessionHistoryInfoHandler.getModelName.call(sessionsData, index);

    return <div className={style.session_item_model_user}>{model}</div>;
});

const SessionUserName = React.memo((props: ISessionItemProps) => {
    const { sessionsData, index } = props;
    const user = SessionHistoryInfoHandler.getUserDetails.call(sessionsData, index);
    const printName = UserInfoHandler.getPrintNameWithoutPn.call(user);

    return <div className={style.session_item_model_user}>{printName}</div>;
});

const SessionUserCardLinks = React.memo((props: ISessionItemProps) => {
    const { sessionsData, index } = props;
    const userId = SessionHistoryInfoHandler.getUserId.call(sessionsData, index);
    const user = SessionHistoryInfoHandler.getUserDetails.call(sessionsData, index);
    const user_name = UserInfoHandler.getPrintName.call(user);
    const phoneNumber = UserInfoHandler.getSimplePhone.call(user);

    const copyText = `${user_name}; ${phoneNumber}`;

    return <div className={style.session_item_card_links}>
        <CopyIcon text={copyText}
                  externalTooltipId={sessionTooltipId}/>
        <span> - </span>
        <Link href={`#/clients/${userId}/info`}>кц</Link>
        <span> - </span>
        <Link href={`#/clients/${userId}/sessions`}>сессии</Link>
    </div>;
});

const SessionCarCardLinks = React.memo((props: ISessionItemProps) => {
    const { sessionsData, index } = props;
    const carId = SessionHistoryInfoHandler.getCarId.call(sessionsData, index);
    const back = encodeURIComponent(location.hash);

    return <div className={style.session_item_card_links}>
        <Link href={`#/cars/${carId}/info?back=${back}`}>кц</Link>
        <span> - </span>
        <Link href={`#/cars/${carId}/tags?back=${back}`}>теги</Link>
        <span> - </span>
        <Link href={`#/cars/${carId}/sessions?back=${back}`}>сессии</Link>
    </div>;
});

const SessionCarNumber = React.memo((props: ISessionItemProps) => {
    const { sessionsData, index } = props;
    const car = SessionHistoryInfoHandler.getCar.call(sessionsData, index);

    return <div className={style.session_item_car_number}>
        <CarNumber carInfo={car} externalTooltipId={sessionTooltipId}/>
    </div>;
});

const SessionCarNumberEmpty = React.memo((props) => {
    return <div className={style.session_item_car_number}/>;
});

export const SessionOfferName = React.memo((props: ISessionItemProps) => {
    const { sessionsData, index, BlockRules } = props;
    const offerName = SessionHistoryInfoHandler.getOfferName.call(sessionsData, index);
    const current_offer = SessionHistoryInfoHandler.getCurrentOffer.call(sessionsData, index) ?? {};
    const offer_proto = SessionHistoryInfoHandler.getOffer.call(sessionsData, index);
    const total = SessionHistoryInfoHandler.getTotalPrice.call(sessionsData, index);

    const sessionOffer = renderToString(
        <SessionOffer BlockRules={BlockRules}
                      current_offer={current_offer}
                      key={index}
                      offer_proto={offer_proto}
                      total={total}
                      className={style.custom_tooltip}/>,
    );

    return <div className={style.session_item_offer_name}
                data-tip={sessionOffer}
                data-place={'left'}
                data-for={sessionTooltipId}>{offerName}</div>;
});

const SessionOfferControls = React.memo((props: ISessionItemProps) => {
    const { sessionsData, index, showProto } = props;
    const sessionId = SessionHistoryInfoHandler.getSessionId.call(sessionsData, index);
    const bill = SessionHistoryInfoHandler.getBill.call(sessionsData, index);

    return <div className={style.session_item_offer_controls}>
        <span className={style.session_item_offer_controls_bill}
              data-for={sessionTooltipId}
              data-tip={renderToString(<SessionBill bill={bill}/>)}>чек</span>
        <span> - </span>
        <Link href={`#/session/${sessionId}/billing?back=${encodeURIComponent(location.hash)}`}>info</Link>
        <span> - </span>
        <Link onClick={showProto}>proto</Link>
    </div>;
});

export const SessionTotalPrice = React.memo((props: ISessionItemProps) => {
    const { sessionsData, index } = props;
    const total = SessionHistoryInfoHandler.getTotalPrice.call(sessionsData, index);

    return <div className={style.session_item_total_price}>{total}</div>;
});

export const SessionMileage = React.memo((props: ISessionItemProps) => {
    const { sessionsData, index } = props;
    const sessionId = SessionHistoryInfoHandler.getSessionId.call(sessionsData, index);
    const mileage = SessionHistoryInfoHandler.getMileage.call(sessionsData, index);
    const hr_mileage = SessionHistoryInfoHandler.getHrMileage.call(sessionsData, index);
    const coordinatesDiff = SessionHistoryInfoHandler.getCoordinatesDiff.call(sessionsData, index);
    const small = coordinatesDiff && !mileage;

    const link = <Link href={`#/tracks?session_id=${sessionId}&status=`}>
        {(small || mileage) ? (small ? '<1 км' : hr_mileage) : EMPTY_DATA}
    </Link>;

    return <div className={`${style.session_item_mileage} ${mileage ? style.session_item_mileage_hr : ''}`}>
        {link}
    </div>;
});
