import React from 'react';
import ReactTooltip from 'react-tooltip';
import XLSX from 'xlsx';

import { GetSessions, IGetSessionsPropsExport } from '../../../decorators/GetSessions';
import { ITagDescriptionsPropsExport, TagDescriptions } from '../../../decorators/TagDescriptions';
import { Dict } from '../../../types';
import { EMPTY_DATA, ONE_SECOND, USER_INFO_CSV_ROW } from '../../constants';
import SessionsHistoryItem, { ISessionInfo, SessionHistoryInfoHandler } from '../../models/session';
import { UserInfoHandler } from '../../models/user';
import { FormatDateInString } from '../../ui/FormatDate';
import { JsonModal } from '../../ui/FullModal/JsonModal';
import { Link } from '../../ui/Link';
import { NoInformation } from '../../ui/NoInformation';
import { getCookie } from '../../utils/cookie';
import { isObjectEqual } from '../../utils/isObjectEqual';
import { Request2 } from '../../utils/request';
import { ytLogs } from '../../utils/sendLogs';
import { EVENT_TYPES } from '../../utils/sendLogs/eventTypes/eventTypes';
import { deepCopy, getUserInfoForCSV } from '../../utils/utils';
import { SimpleError } from '../SimpleError';
import Spin from '../Spin';
import IdentifyForShowing from './IdentifyForShowing';
import style from './index.css';
import { REQUESTS, SESSIONS_LIST_REQUESTS } from './request';
import { SessionItem } from './SessionItem';
import SessionsListHeader from './SessionsListHeadr';
import { sessionTooltipId } from './sessionTooltipId';

interface ISessionsListProps extends IGetSessionsPropsExport, ITagDescriptionsPropsExport {
    userId?: string;
    carId?: string;
    whoami?: string;
    BlockRules: Dict<any>;
    location: any;
}

interface IPerformItems {
    records?: [];
    user_tags?: [];
}

interface ISessionsListState {
    [key: string]: any;

    isLoading: boolean;
    showProto: string | null;
    error: Error | null;
    previousPages: number[];
    ts_supreme_pagination: number;
    sessionsData: SessionsHistoryItem;
    since: number;
    until: number;
    isWorking: boolean;
    page: number;
    hasIdentified: boolean;
    performItems: IPerformItems | null;
    performItemsIsLoading: boolean;
}

const NUMDOC = 50;

export const sliceEnd = 3;

@GetSessions({ numdoc: NUMDOC })
@TagDescriptions()
export class SessionsList extends React.Component<ISessionsListProps, ISessionsListState> {
    request = new Request2({ requestConfigs: SESSIONS_LIST_REQUESTS });
    state: ISessionsListState = {
        isLoading: true,
        error: null,
        sessionsData: {} as SessionsHistoryItem,
        showProto: null,
        since: 0,
        until: 0,
        isWorking: false,
        previousPages: [],
        ts_supreme_pagination: 0,
        page: 0,
        hasIdentified: false,
        performItems: null,
        performItemsIsLoading: false,
    };
    ref: React.RefObject<HTMLDivElement>;
    log = ytLogs.getInstance();

    constructor(props) {
        super(props);
        this.ref = React.createRef();
    }

    componentDidMount() {
        if (this.checkIdentify()) {
            this.checkPerformClient();
        }

        if (!this.checkIdentify()) {
            this.getData();
            this.props?.getTags?.();
        }
    }

    componentDidUpdate(
        prevProps: Readonly<ISessionsListProps>,
        prevState: Readonly<ISessionsListState>,
        snapshot?: any,
    ) {
        if (!isObjectEqual(prevProps.sessionsData, this.props.sessionsData)) {
            ReactTooltip.rebuild();
        }

        if (this.props.userId !== prevProps.userId || this.state.hasIdentified !== prevState.hasIdentified) {
            if (this.checkIdentify()) {
                this.checkPerformClient();
            }

            if (!this.checkIdentify()) {
                this.getData();
            }
        }
    }

    getData() {
        this.props?.getSessions?.({
            user_id: this.props.userId,
            car_id: this.props.carId,
            ts_since: this.state.since,
            until: this.state.ts_supreme_pagination
                && Math.floor(this.state.ts_supreme_pagination / ONE_SECOND)
                || this.state.until,
        });
    }

    componentWillUnmount() {
        this.request.abort();
    }

    showProto(showProto) {
        this.setState({
            showProto,
        });
    }

    onChangeDates(field, value) {
        this.setState({
            [field]: value,
            page: 0,
            previousPages: [],
            ts_supreme_pagination: 0,
        }, () => {
            this.getData();
        });
    }

    cutSessions() {
        const sessionsData: any = deepCopy(this.props.sessionsData || {});
        const sessions: any[] = [];
        const SESSION_LIMIT = 4; //https://st.yandex-team.ru/DRIVEFRONT-965#605c668707effe485d54c34b
        if (this.isClientSessions() && !this.props.BlockRules?.ViewAllSession) {
            const length = sessionsData?.sessions?.length;
            for (let index = 0; index < length; index++) {
                const diff = SessionHistoryInfoHandler.getCoordinatesDiff.call(sessionsData, index);
                const mileage = SessionHistoryInfoHandler.getMileage.call(sessionsData, index);
                const flag = ((diff && !mileage) || mileage);
                sessions.push(sessionsData.sessions[index]);
                if (flag && index >= SESSION_LIMIT) {
                    const currentStart = SessionHistoryInfoHandler.getStart.call(sessionsData, index);
                    const prevFinish = SessionHistoryInfoHandler.getFinish.call(sessionsData, index + 1);
                    if (currentStart < prevFinish) {
                        sessions.push(sessionsData.sessions[index + 1]);
                    }

                    break;
                }
            }

            sessionsData.sessions = sessions;
        }

        return sessionsData;
    }

    downloadCarSessions() {
        const sessionsData: any = this.cutSessions();

        if (sessionsData.sessions.length) {
            const wb = XLSX.utils.book_new();
            const ws_data: any = [];
            ws_data.push(
                ['#', 'начало', 'конец', 'продожительность', 'пользователь', 'тариф', 'город', 'пробег', 'итого'],
                ...sessionsData?.sessions.map((sessions, index) => {
                    const userDetails = SessionHistoryInfoHandler.getUserDetails.call(sessionsData, index);
                    const current_offer = SessionHistoryInfoHandler.getCurrentOffer.call(sessionsData, index);
                    const mileage = SessionHistoryInfoHandler.getMileage.call(sessionsData, index);
                    const coordinatesDiff = SessionHistoryInfoHandler.getCoordinatesDiff.call(sessionsData, index);

                    return [
                        index + 1,
                        FormatDateInString({ value: SessionHistoryInfoHandler.getStart.call(sessionsData, index) }),
                        FormatDateInString({ value: SessionHistoryInfoHandler.getFinish.call(sessionsData, index) }),
                        SessionHistoryInfoHandler.getDuration.call(sessionsData, index),
                        UserInfoHandler.getPrintName.call(userDetails),
                        current_offer.name || current_offer.short_name || EMPTY_DATA,
                        SessionHistoryInfoHandler.getCity.call(sessionsData, index),
                        `${SessionHistoryInfoHandler.getHrMileage.call(sessionsData, index)}` +
                        `${coordinatesDiff && !mileage ? ' (<1 км)' : ''}`,
                        SessionHistoryInfoHandler.getTotalPrice.call(sessionsData, index),
                    ];
                }),
            );

            const carNumber = SessionHistoryInfoHandler.getCarNumber.call(sessionsData, 0);

            const ws = XLSX.utils.aoa_to_sheet(ws_data);
            XLSX.utils.book_append_sheet(wb, ws, 'сессии');
            XLSX.writeFile(wb, `сессии_${carNumber ?? this.props.carId}.xlsx`);
        }
    }

    downloadPassportInfo() {
        this.setState({
            isWorking: true,
        }, () => {
            const sessions = SessionHistoryInfoHandler.getData.call(this.cutSessions())?.sessions || [];
            const order: any = sessions?.[0];
            const carNumber = order?.car.number;
            const carModel = order?.car.model_id;

            const usersId = sessions?.map((item: any) => {
                return item?.user_details.id;
            });
            this.props.userId && usersId.push(this.props.userId);
            const users = [...new Set(usersId)];

            Promise.all(users?.map((user_id: any) => {
                return this.request.exec(REQUESTS.GET_USER_INFO, {
                    queryParams: { user_id },
                });
            }))
                .then(usersData => {
                    const wb = XLSX.utils.book_new();
                    const ws_data: any = [];
                    ws_data.push(
                        USER_INFO_CSV_ROW.split(';'),
                        ...usersData.map(userData => {
                            return getUserInfoForCSV(userData).split(';');
                        }),
                    );
                    const ws = XLSX.utils.aoa_to_sheet(ws_data);

                    XLSX.utils.book_append_sheet(wb, ws, 'пдн');
                    if (usersData.length === 1) { //only one person
                        XLSX.writeFile(wb, `${ws_data?.[1]?.slice(0, sliceEnd)?.join(' ')}.xlsx`);
                    } else {
                        XLSX.writeFile(wb, `${carModel}_${carNumber}.xlsx`);
                    }

                    this.setState({
                        isWorking: false,
                    });
                })
                .catch((error) => {
                    this.setState({
                        sessionError: error,
                        isWorking: false,
                    });
                });

        });
    }

    getNextPage() {
        const { previousPages } = this.state;
        const sessionsData = this.cutSessions();

        const currentSessions = SessionHistoryInfoHandler.getData.call(sessionsData)?.sessions;
        const ts_supreme_pagination = SessionHistoryInfoHandler.getFinish
            .call(sessionsData, currentSessions.length - 1);
        const ts_supreme_pagination_history = SessionHistoryInfoHandler.getFinish.call(sessionsData, 0);

        previousPages.push(ts_supreme_pagination_history);

        currentSessions && this.setState((prev) => ({
            ts_supreme_pagination,
            previousPages,
            page: prev.page + 1,
        }), () => {
            this.getData();
        });
    }

    getPreviousPage() {
        const { previousPages } = this.state;
        const ts_supreme_pagination = previousPages.pop();

        this.setState((prev) => ({
            ts_supreme_pagination: prev.page - 1 === 0 ? 0 : ts_supreme_pagination ?? 0,
            previousPages,
            page: prev.page - 1,
        }), () => {
            this.getData();
        });
    }

    checkIdentify() {
        return this.props.BlockRules?.NeedIdentifySessions
            && !this.state.hasIdentified
            && !!this.props.userId;
    }

    checkPerformClient() {
        this.setState({
            performItemsIsLoading: true,
        }, () => {
            this.request.exec(REQUESTS.GET_USER_TAGS_BY_PERFORMER, {
                queryParams: { performer: this.props.whoami },
            },
            )
                .then(performItems => {
                    this.setState({
                        performItems,
                        performItemsIsLoading: false,
                    }, () => {
                        if (this.isPerformThatUserNow()) {
                            this.setState({
                                hasIdentified: true,
                            });
                        }
                    });
                })
                .catch(error => {
                    this.setState({
                        error,
                        performItemsIsLoading: false,
                    });
                });
        });
    }

    isPerformThatUserNow() {
        return this.state.performItems?.user_tags?.some((el: any) => el.object_id === this.props.userId);
    }

    onIdentify(supportAnswers) {
        this.setState({
            hasIdentified: true,
        }, () => {
            const data = {
                event_type: EVENT_TYPES.CUSTOM_LOG,
                meta: {
                    supportAnswers,
                    performItems: this.state.performItems,
                    location: location.href,
                    yl: getCookie('yandex_login'),
                    type: 'IDENTIFY_BEFORE_SHOWING_SESSIONS',
                },
            };
            this.log.send({ data });
        });
    }

    isClientSessions() {
        return this.props.location.pathname.includes('/clients/');
    }

    render() {
        const { sessionIsLoading, sessionError, BlockRules } = this.props;
        const sessionsData = this.cutSessions();
        const sessions: ISessionInfo[] = SessionHistoryInfoHandler.getData.call(sessionsData)?.sessions || [];
        const has_more = SessionHistoryInfoHandler.hasMore.call(sessionsData);
        const needToIdentify = this.checkIdentify();
        const showComment = !BlockRules.ViewAllSession && this.isClientSessions();

        return <div className={style.component}>
            {
                needToIdentify && this.state.performItemsIsLoading
                    ? <div><Spin size={'s'}/> Верифкация пользователя...</div>
                    : needToIdentify && !this.state.hasIdentified
                        ? <IdentifyForShowing onIdentify={this.onIdentify.bind(this)}/>
                        : <>
                            <ReactTooltip id={sessionTooltipId} type={'info'} html/>
                            <div className={style.header}>
                                <SessionsListHeader userId={this.props.userId || null}
                                                    carId={this.props.carId || null}
                                                    isWorking={this.state.isWorking}
                                                    downloadPassportInfo={this.downloadPassportInfo.bind(this)}
                                                    downloadCarSessions={this.downloadCarSessions.bind(this)}
                                                    onChangeDates={this.onChangeDates.bind(this)}/>
                            </div>

                            <div className={style.content} ref={this.ref}>

                                {sessionIsLoading ? <Spin/> : sessionError ?
                                    <SimpleError error={sessionError}/>
                                    : sessions.length ?
                                        <>
                                            {
                                                sessions.map((_, index) => {
                                                    return <SessionItem key={index}
                                                                        BlockRules={this.props.BlockRules}
                                                                        sessionsData={sessionsData || {}}
                                                                        index={index}
                                                                        tagsObject={this.props.tagsObject}
                                                                        tagIsLoading={this.props.tagIsLoading}
                                                                        showProto={this.showProto
                                                                            .bind(this, `${index}`)}
                                                                        isUser={!!this.props.userId}/>;
                                                })
                                            }
                                            {

                                                this.props.BlockRules?.ViewAllSession &&
                                            <div className={style.pagination}>
                                                {this.state.previousPages.length ? (
                                                    <Link className={style.page_link}
                                                          onClick={this.getPreviousPage.bind(this)}>
                                                        Назад
                                                    </Link>
                                                ) : undefined}
                                                {has_more ? (
                                                    <Link className={style.page_link}
                                                          onClick={this.getNextPage.bind(this)}>
                                                        Ещё
                                                    </Link>
                                                ) : undefined}
                                            </div>
                                            }

                                            {showComment
                                                ? <div className={style.info}>
                                                    -= Отображаются 5 поездок с треком =-
                                                </div>
                                                : null
                                            }
                                        </>
                                        : <NoInformation/>
                                }

                            </div>
                            {

                                this.state.showProto &&
                            <JsonModal onClose={this.showProto.bind(this, null)}
                                       obj={sessions?.[+this.state.showProto]?.offer_proto || {}}
                                       title={`offer_proto`}/>

                            }
                        </>
            }
        </div>;
    }
}
