import * as React from 'react';

import { CancelButton } from '../../../../../ui/Button';
import { Confirm, Window } from '../../../../../ui/FullModal';
import { Link } from '../../../../../ui/Link';
import * as styleTable from '../../../../../ui/Table/index.css';
import { Request2 } from '../../../../../utils/request';
import { SimpleError } from '../../../../SimpleError';
import Spin from '../../../../Spin';
import { REQUESTS, SETTINGS_REQUESTS } from '../../request';
import { IRolesInfo } from '../../types';
import * as style from './index.css';

interface IRolesByActionsProps {
    selectedAction: string | null;
    onClose: () => void;
    isWorking: boolean;
    rolesInfo: IRolesInfo;
    getRoles: () => void;
}

interface IRolesByActionsState {
    confirmIsOpen: boolean;
    isWorking: boolean;
    error: Error | null;
    question: string;
    accept: () => void;
}

export default class RolesByActions extends React.Component<IRolesByActionsProps, IRolesByActionsState> {
    state: IRolesByActionsState = {
        confirmIsOpen: false,
        isWorking: false,
        error: null,
        question: '',
        accept: () => {
        },
    };
    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

    componentDidMount() {
        if (!Object.keys(this.props.rolesInfo.actionRoles)?.length) {
            this.props.getRoles();
        }
    }

    remove(role_id: string) {
        const { selectedAction } = this.props;

        this.setState({
            error: null,
            confirmIsOpen: true,
            isWorking: false,
            question: `Отвязать роль ${role_id} от экшена ${selectedAction}?`,
            accept: () => {
                this.setState({
                    isWorking: true,
                }, () => {
                    this.request.exec(REQUESTS.DELETE_ROLES_ACTIONS,
                        { queryParams: { role: role_id, actions: selectedAction } })
                        .then(() => {
                            this.setState({
                                isWorking: false,
                                confirmIsOpen: false,
                            }, () => this.props.getRoles());
                        })
                        .catch((error) => {
                            this.setState({
                                error,
                                isWorking: false,
                            });
                        });
                });
            },
        });
    }

    confirm(state: boolean) {
        this.setState({
            confirmIsOpen: state,
        });
    }

    getParentRoles(roles) {
        const parentRoles: string[] = [];
        roles.forEach(role => {
            if (this.props.rolesInfo.parentRoles[role.role_id]) {
                parentRoles.push(...this.props.rolesInfo.parentRoles[role.role_id]);
            }
        });

        return parentRoles;
    }

    render() {
        const { selectedAction, onClose, rolesInfo } = this.props;

        const roles = this.props.selectedAction ? rolesInfo.actionRoles[this.props.selectedAction] : [];
        const parentRoles = roles?.length ? this.getParentRoles(roles) : [];

        return <Window onClose={onClose?.bind(this)}
                       title={`Роли экшена ${selectedAction}`}>
            <div className={style.roles_by_action}>
                {rolesInfo.rolesError
                    ? <SimpleError error={rolesInfo.rolesError} data={{ label: 'Ошибка при загрузке ролей' }}/>
                    : null
                }

                {rolesInfo.isRolesLoading
                    ? <Spin/>
                    : <>
                        {roles?.length
                            ? <>
                                <h4>Роли экшена</h4>
                                <table className={styleTable.table}>
                                    <thead>
                                        <tr>
                                            <th>#</th>
                                            <th>ID роли</th>
                                            <th>Описание роли</th>
                                            <th/>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        {roles.map((role, index) => {
                                            return <tr key={role.role_id || index}>
                                                <td>{index + 1}</td>
                                                <td>{role.role_id}</td>
                                                <td>{role.role_description}</td>
                                                <td>
                                                    <Link onClick={this.remove.bind(this, role.role_id)}>
                                                    Отвязать
                                                    </Link>
                                                </td>
                                            </tr>;
                                        })
                                        }
                                    </tbody>
                                </table>
                            </>
                            : <h4>Нет ролей</h4>
                        }

                        {parentRoles?.length
                            ? <>
                                <h4>Родительские роли</h4>
                                <table className={styleTable.table}>
                                    <thead>
                                        <tr>
                                            <th>#</th>
                                            <th>ID роли</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        {parentRoles.map((role, index) => {
                                            return <tr key={role + index}>
                                                <td>{index + 1}</td>
                                                <td>{role}</td>
                                            </tr>;
                                        })
                                        }
                                    </tbody>
                                </table>
                            </>
                            : <h4>Нет родительских ролей</h4>
                        }
                    </>
                }
                {
                    this.state.confirmIsOpen
                        ? <Confirm question={this.state.question}
                                   error={this.state.error}
                                   onClose={this.confirm.bind(this, false)}
                                   isWorking={this.state.isWorking}
                                   accept={this.state.accept}/>
                        : null
                }
                <div className={style.controls}>
                    <CancelButton onClick={onClose?.bind(this)}/>
                </div>
            </div>
        </Window>;
    }
}
