import * as React from 'react';

import { Dict } from '../../../../../types';
import { Link } from '../../../../ui/Link';
import { IPlaneTreeItem } from '../../../../ui/VirtualTreeList';
import { DiffLabel } from '../../../DiffLabel/component';
import { IAction, IRolesInfo } from '../types';
import * as style from './index.css';
import RolesByActions from './RolesByActions/component';

const ENTITIES_MAX_COUNT = 3;
const DIFF_MAX_COUNT = 3;

enum ACTION_STATUSES {
    WARNING = 'warning',
    ENABLED = 'enabled',
    DISABLED = 'disabled'
}

interface ITreeListItemProps {
    item: IPlaneTreeItem<IAction>;
    actionsTypesWithCount: Dict<number>;
    rolesInfo: IRolesInfo;
    getRoles: () => void;
    isForUpdate?: boolean;
}

interface ITreeListItemState {
    rolesByActions: boolean;
    selectedAction: string | null;
    isWorking: boolean;
}

export default class ActionTreeListItem extends React.Component<ITreeListItemProps, ITreeListItemState> {
    state: ITreeListItemState = {
        rolesByActions: false,
        selectedAction: null,
        isWorking: false,
    };

    getRolesByAction(action_id: string) {
        this.setState({
            rolesByActions: true,
            selectedAction: action_id,
        });
    }

    onClose() {
        this.setState({
            rolesByActions: false,
            selectedAction: null,
        });
    }

    render() {
        const { item, actionsTypesWithCount, rolesInfo, getRoles, isForUpdate } = this.props;
        const { rolesByActions, selectedAction, isWorking } = this.state;
        const { data, meta } = item;
        const { action_id: id = '', action_description: description = '', action_meta = {} } = data || {};
        const { options, parentsIds = [], childrenCount, filteredChildrenCount } = meta || {};

        const { oldParent } = options ?? {};
        const status = options && (options.unlinked || oldParent)
            ? ACTION_STATUSES.WARNING
            : data?.enabled
                ? ACTION_STATUSES.ENABLED
                : ACTION_STATUSES.DISABLED;

        const isActionType = actionsTypesWithCount?.hasOwnProperty(id)
            && parentsIds?.length === 0;

        const adm_entities = action_meta?.adm_entities || [];
        const showEntities = adm_entities.length > ENTITIES_MAX_COUNT
            ? adm_entities.slice(0, ENTITIES_MAX_COUNT).join(', ') + ', ...'
            : adm_entities.join(', ');
        const href = `#/settings/actions${isForUpdate ? '-update' : ''}/${encodeURIComponent(id)}`;

        return <div className={style.tree_node_container}>
            <div className={`${style.tree_node} ${!isActionType ? style.leaf : ''}`}>
                <div className={style.node_action_name}>
                    {!isActionType
                        ? <div className={`${style.status} ${style[status]}`}
                               title={status === ACTION_STATUSES.WARNING ? 'Возможно роль зациклена' : status}/>
                        : null}
                    {!isActionType
                        ? <Link className={style.description} href={href}>
                            {description.trim() ? description : id}
                        </Link>
                        : id}
                    {isActionType
                        ? <div className={style.action_group_count}>
                            {filteredChildrenCount
                                ? filteredChildrenCount || null
                                : childrenCount || null}
                        </div>
                        : null}
                    {data?.propositions?.length
                        ? <Link className={style.propositions}
                                href={`${href}/0`}>Предложения: {data.propositions.length}</Link>
                        : null}
                </div>
                <div className={style.node_action_info}>
                    {data?.action_revision
                        ? <div className={style.node_action_revision}>Rev.: {data?.action_revision}</div>
                        : null}
                    {!isActionType
                        ? <div className={style.node_action_id}>
                            <span>{id} </span>
                            <span className={style.entities}>
                                {adm_entities.length ? ' | ' : null}
                                {showEntities}
                            </span>
                        </div>
                        : null}
                </div>
                <div className={style.node_action_diff}>
                    {
                        !isActionType && !actionsTypesWithCount.hasOwnProperty(item?.meta?.parent?.data?.action_id)
                            ? <DiffLabel maxCount={DIFF_MAX_COUNT}
                                         child={item.data}
                                         parent={item.meta?.parent && item.meta?.parent.data}/>
                            : null
                    }
                </div>
                {!isActionType
                    ? <div className={style.node_action_roles}>
                        <Link onClick={this.getRolesByAction.bind(this, data?.action_id)}>Роли</Link>
                    </div>
                    : null}
            </div>
            {
                rolesByActions
                    ? <RolesByActions rolesInfo={rolesInfo}
                                      getRoles={getRoles.bind(this)}
                                      selectedAction={selectedAction}
                                      onClose={this.onClose.bind(this)}
                                      isWorking={isWorking}/>
                    : null
            }
        </div>;
    }
}
