import * as React from 'react';
import { RouteComponentProps } from 'react-router';

import { Dict } from '../../../../../types';
import { Button, ButtonTypes, CancelButton, DeleteButton } from '../../../../ui/Button';
import { Confirm, Window } from '../../../../ui/FullModal';
import * as coreStyle from '../../../../ui/index.css';
import { Link } from '../../../../ui/Link';
import Select from '../../../../ui/Select';
import { ConstantsKey } from '../../../../utils/fetchConstants';
import { Request2 } from '../../../../utils/request';
import { deepCopy } from '../../../../utils/utils';
import { Copy } from '../../../Copy';
import { FormConstructor } from '../../../FormConstructor';
import { SimpleError } from '../../../SimpleError';
import Spin from '../../../Spin';
import { IPropositions } from '../../Roles/component';
import PropositionConfirmModal from '../PropositionConfirmModal';
import { REQUESTS, SETTINGS_REQUESTS } from '../request';
import * as style from './index.css';

enum PropositionsActions {
    CONFIRM = 'confirm',
    REJECT = 'reject'
}

export const CREATE_NEW_ACTION_ID = 'create-new-action';

export const PROPOSITIONS_ACTIONS: Dict<{ text: string; question: string; answer: string; action: string }> = {
    CONFIRM: {
        text: 'Принять',
        question: 'Принять предложение?',
        answer: 'Действие было принято',
        action: PropositionsActions.CONFIRM,
    },
    REJECT: {
        text: 'Отклонить',
        question: 'Отклонить предложение?',
        answer: 'Действие было отклонено',
        action: PropositionsActions.REJECT,
    },
};

interface IConstants {
    constants: (traits: ConstantsKey[], ids: string) => Promise<any>;
}

interface IAddActionModalProps extends RouteComponentProps<{
    action_id: string;
    proposition_index: string;
}>, IConstants {
    updateAllActions: (success: boolean) => void;
    isForUpdate?: boolean;
}

interface IAddActionModalState {
    actionsIsLoading: boolean;
    action_type: string;
    initAction: Dict<any> | null;
    parentAction: Dict<any>;
    propositions: IPropositions[];
    isConfirmAddPropositionOpen: boolean;
    confirmModalIsOpen: boolean;
    action_id: string;
    iface: Dict<any>;
    action_types: { text: string; value: string }[];
    propositionIndex: number;
    question: string;
    isWorking: boolean;
    accept: () => void;
    mainData: Dict<any>;
    error: Error | null;
    deleteConfirmIsOpen: boolean;
    isActionDeprecated: boolean;
    isFormValid: boolean;
    isFormChanged: boolean;

    actionTypesIsLoading: boolean;
    actionSchemeIsLoading: boolean;
    showCopyModal: boolean;
}

export default class AddActionModal extends React.Component<IAddActionModalProps, IAddActionModalState> {
    state: IAddActionModalState = {
        actionsIsLoading: true,
        action_type: '',
        initAction: null,
        parentAction: {},
        propositions: [],
        isConfirmAddPropositionOpen: false,
        confirmModalIsOpen: false,
        action_id: '',
        iface: {},
        action_types: [],
        propositionIndex: -1,
        question: '',
        isWorking: false,
        accept: () => {},
        mainData: {},
        error: null,
        deleteConfirmIsOpen: false,
        isActionDeprecated: false,
        isFormValid: false,
        isFormChanged: false,
        showCopyModal: false,

        actionTypesIsLoading: true,
        actionSchemeIsLoading: false,
    };
    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

    componentDidMount(): void {
        this.getData();
    }

    componentDidUpdate(prevProps: Readonly<IAddActionModalProps>): void {
        if (this.props.location.pathname !== prevProps.location.pathname) {
            this.getData();
        }
    }

    isNewAction() {
        return this.props?.match?.params?.action_id === CREATE_NEW_ACTION_ID;
    }

    getData() {
        this.getActionTypes(() => {
            this.getActionData();
        });
    }

    getActionData() {
        const props = this.props;
        const action_id = props?.match?.params?.action_id ? decodeURIComponent(props?.match?.params?.action_id) : '';
        const propositionIndex: any = props?.match?.params?.proposition_index || -1;
        const isNewAction = this.isNewAction();

        !isNewAction
        && this.request.exec(REQUESTS.GET_ACTIONS, { queryParams: { action_id: encodeURIComponent(action_id) } })
            .then(responseActions => {
                let initAction = [...responseActions?.report, ...responseActions?.report_deprecated]
                    ?.filter(role => role.action_id === action_id)[0];
                const parentAction = Object.assign({}, initAction);

                if (propositionIndex >= initAction.propositions.length) {
                    console.error(`proposition #${propositionIndex} doesn't exist`);
                } else if (initAction.propositions.length && propositionIndex !== -1) {
                    initAction = initAction.propositions[propositionIndex];
                }

                this.setState({
                    action_type: initAction?.action_type,
                    initAction: initAction,
                    propositions: initAction.propositions,
                    parentAction,
                    propositionIndex,
                    action_id,
                }, () => {
                    this.getActionScheme();
                });
            })
            .catch((error) => {
                this.setState({ error, actionsIsLoading: false });
            });

    }

    getActionScheme() {
        this.setState({ actionSchemeIsLoading: true }, () => {
            this.props.constants([ConstantsKey.IFACE], this.state.action_type)
                .then(responseConstants => {
                    this.setState({
                        iface: responseConstants?.iface,
                        actionSchemeIsLoading: false,
                    });
                })
                .catch(error => {
                    this.setState({
                        actionSchemeIsLoading: false,
                    });
                });
        });
    }

    getActionTypes(callback) {
        this.props.constants([ConstantsKey.ACTION_TYPES], '')
            .then(responseConstants => {
                this.setState({
                    action_types: [...responseConstants?.action_types]
                        .sort((a, b) => a.localeCompare(b))
                        .map(item => ({ text: item, value: item })),
                    actionTypesIsLoading: false,
                });
            })
            .then(error => {
                this.setState({
                    actionTypesIsLoading: false,
                }, () => {
                    callback?.();
                });
            });
    }

    onSelectActionType(action_type: string) {
        this.setState({ action_type }, () => {
            this.getActionScheme();
        });
    }

    onFormChange(mainData: Dict<any>, isFormValid: boolean, isFormChanged: boolean) {
        this.setState({ mainData, isActionDeprecated: mainData?.deprecated ?? false, isFormValid, isFormChanged });
    }

    onClose(success = false) {
        if (success) {
            this.props.updateAllActions(true);
        }

        location.hash = `/settings/actions${this.props.isForUpdate ? '-update' : ''}/`;
    }

    openDeleteConfirm() {
        this.setState({ deleteConfirmIsOpen: true });
    }

    closeDeleteConfirm() {
        this.setState({ deleteConfirmIsOpen: false });
    }

    createActionHandler() {
        const updatedData = deepCopy(this.state.mainData);

        this.setState({
            isWorking: true,
            error: null,
        }, () => {
            this.request.exec(REQUESTS.ADD_ACTION, { body: { ...updatedData, action_type: this.state.action_type } })
                .then(() => {
                    this.setState({
                        isWorking: false,
                    });
                    this.onClose(true);
                })
                .catch((error) => {
                    this.setState({
                        isWorking: false,
                        error,
                    });
                });
        });
    }

    openAddPropositionConfirm() {
        this.setState({ isConfirmAddPropositionOpen: true });
    }

    closePropositionConfirm(isSuccess: boolean) {
        this.setState({ isConfirmAddPropositionOpen: false });
        if (isSuccess) {
            this.onClose(true);
        }
    }

    handleProposition(type: string) {
        this.setState({
            confirmModalIsOpen: true,
            question: PROPOSITIONS_ACTIONS[type.toUpperCase()].question,
            accept: () => {
                this.setState({
                    isWorking: true,
                }, () => {
                    const proposition_id = this.state?.parentAction
                        ?.propositions[this.state.propositionIndex]?.proposition_id;
                    const action = `PROPOSITION_${type.toUpperCase()}`;

                    this.request.exec(REQUESTS[action], {
                        body: {
                            proposition_ids: [proposition_id],
                        },
                    })
                        .then(() => {
                            this.setState({
                                isWorking: false,
                                confirmModalIsOpen: false,
                            }, () => {
                                this.onClose(true);
                            });
                        })
                        .catch((error) => {
                            const SELF_CONFIRM_ERROR_KEY = 'self confirm is denied';
                            if (error?.data?.error_details?.special_info?.session_info
                                .hasOwnProperty(SELF_CONFIRM_ERROR_KEY)) {
                                error = new Error('Нельзя подтверждать свои предложения');
                            }

                            this.setState({ error, isWorking: false });
                        });
                });
            },
        });
    }

    deleteActionHandler(action: string) {
        this.setState({
            error: null,
            deleteConfirmIsOpen: true,
            question: 'Удалить action?',
            accept: () => {
                this.setState({
                    isWorking: true,
                }, () => {

                    this.setState({ isWorking: true });
                    this.request.exec(REQUESTS.REMOVE_ACTIONS, { queryParams: { actions: action } })
                        .then(() => {
                            this.onClose(true);
                        })
                        .catch((error) => {
                            this.setState({ error, isWorking: false });
                        });
                });
            },
        });
    }

    closeConfirm() {
        this.setState({
            confirmModalIsOpen: false,
            deleteConfirmIsOpen: false,
        });
    }

    triggerCopyModal(shouldShow) {
        this.setState({
            showCopyModal: shouldShow,
        });

        if (!shouldShow) {
            this.onClose(true);
        }
    }

    render() {
        const { isActionDeprecated, initAction, mainData, action_id, action_type, iface } = this.state;
        const isNewAction = this.isNewAction();
        const schema = action_type && iface?.[action_type.toString()] || null;

        const actionId = mainData?.action_id ?? action_id;
        const actionPropositionConfirmationLink = `${location.origin}${location.pathname}`
            + `#/lite-confirmations?type=action&id=${encodeURIComponent(actionId)}`;

        return <Window className={this.state.action_type ? '' : style.empty_action_window}
                       onClose={this.onClose.bind(this)}
                       closeWithConfirm={this.state.isFormChanged}
                       title={`Добавить action ${isActionDeprecated ? '(DEPRECATED)' : ''}`}>
            <div className={style.add_action_modal}>
                <div>
                    <Select options={this.state.action_types}
                            placeholder={'action_type'}
                            onSelect={this.onSelectActionType.bind(this)}
                            disabled={!isNewAction}
                            isLoading={this.state.actionTypesIsLoading}
                            initialValues={this.state.action_type ? [this.state.action_type] : []}/>
                    <div className={style.action_form}>
                        {
                            this.state.actionSchemeIsLoading && <Spin size={'l'}/>
                        }
                        {schema
                            ? <>
                                <FormConstructor onChange={this.onFormChange.bind(this)}
                                                 schema={deepCopy(schema)}
                                                 initialData={initAction ?? null}/>
                                <div>
                                    {
                                        this.state.propositions?.length
                                            ? <>
                                                <div>Доступные предложения
                                                    ({this.state.propositions.length}): {this.state.propositions
                                                    .map((item, index) => {
                                                        return <span key={index}>
                                                            <Link href={`#/settings/actions${this.props.isForUpdate ? '-update' : ''}`
                                                                + `/${encodeURIComponent(this.state.action_id)}`
                                                                + `/${index}`}
                                                                  key={index}>{index}
                                                            </Link>&nbsp;
                                                        </span>;
                                                    })}
                                                </div>
                                                <Copy className={style.copyPropose}
                                                      text={actionPropositionConfirmationLink}>
                                                    Копировать ссылку на предложения
                                                </Copy>
                                            </>
                                            : ''
                                    }
                                    {
                                        this.state.propositionIndex !== -1
                                        && <div>
                                            <Link href={`#/settings/actions${this.props.isForUpdate ? '-update' : ''}`
                                            + `/${encodeURIComponent(this.state.action_id)}`}>
                                                Базовое действие
                                            </Link>
                                        </div>
                                    }
                                </div>
                                {this.state.error ? <SimpleError error={this.state.error}/> : null}
                                <div className={coreStyle.button_container}>
                                    <Button disabled={this.state.propositionIndex !== -1}
                                            onClick={this.openAddPropositionConfirm.bind(this)}>Заявка</Button>
                                    <Button colorType={ButtonTypes.positive}
                                            disabled={this.state.propositionIndex === -1}
                                            onClick={this.handleProposition
                                                .bind(this, PROPOSITIONS_ACTIONS.CONFIRM.action)}>
                                        {PROPOSITIONS_ACTIONS.CONFIRM.text}
                                    </Button>
                                    <Button colorType={ButtonTypes.negative}
                                            disabled={this.state.propositionIndex === -1}
                                            onClick={this.handleProposition
                                                .bind(this, PROPOSITIONS_ACTIONS.REJECT.action)}>
                                        {PROPOSITIONS_ACTIONS.REJECT.text}
                                    </Button>
                                    {
                                        this.state.confirmModalIsOpen
                                            ? <Confirm accept={this.state.accept.bind(this)}
                                                       onClose={this.closeConfirm.bind(this)}
                                                       isWorking={this.state.isWorking}
                                                       error={this.state.error}
                                                       question={this.state.question}/>
                                            : null
                                    }
                                </div>
                                <div className={coreStyle.button_container}>
                                    <CancelButton onClick={this.onClose.bind(this)}/>
                                    <DeleteButton onClick={this.deleteActionHandler.bind(this, this.state.action_id)}
                                                  disabled={this.state.propositionIndex !== -1}/>
                                    <Button isLoading={this.state.isWorking}
                                            onClick={this.createActionHandler.bind(this)}
                                            disabled={this.state.propositionIndex !== -1}>
                                        Сохранить
                                    </Button>
                                </div>
                            </>
                            : null
                        }
                    </div>
                    {this.state.isConfirmAddPropositionOpen
                    && <PropositionConfirmModal newTariffInfo={{
                        diff: null,
                        newTariff: { ...this.state.mainData, action_type: this.state.action_type },
                    }}
                                                triggerCopyModal={this.triggerCopyModal.bind(this, true)}
                                                onClose={this.closePropositionConfirm.bind(this)}/>}
                    {this.state.deleteConfirmIsOpen
                        ? <Confirm accept={this.state.accept.bind(this)}
                                   onClose={this.closeConfirm.bind(this)}
                                   isWorking={this.state.isWorking}
                                   error={this.state.error}
                                   question={this.state.question}/>
                        : null}
                    {this.state.showCopyModal
                        && <Window title={'Копировать ссылку на страницу для подтверждения предложения'}
                                   onClose={this.triggerCopyModal.bind(this, false)}>
                            <Copy customTimeout={400}>
                                {actionPropositionConfirmationLink}
                            </Copy>
                        </Window>}
                </div>
            </div>
        </Window>;
    }
}
