import * as React from 'react';
import { Route, Switch } from 'react-router-dom';

import { Dict } from '../../../../types';
import { Button } from '../../../ui/Button';
import Checkbox from '../../../ui/Checkbox';
import { Input } from '../../../ui/Input';
import { TabItem, Tabs } from '../../../ui/Tabs';
import { IPlaneTreeItem, VirtualTreeList } from '../../../ui/VirtualTreeList';
import { Request2 } from '../../../utils/request';
import { deepCopy } from '../../../utils/utils';
import { IRole } from '../../Clients/UserRolesView/types';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { ActionsTree } from './ActionsTree/component';
import { Node } from './ActionsTree/constants';
import ActionTreeListItem from './ActionTreeListItem/component';
import { CREATE_NEW_ACTION_ID } from './AddActionModal/component';
import AddActionModal from './AddActionModal/index';
import { FRONT_OPTIONS_KEY, ROOT_KEY } from './constants';
import * as style from './index.css';
import { REQUESTS, SETTINGS_REQUESTS } from './request';
import { IAction, IActionProposition } from './types';

const ROOT = { action_id: ROOT_KEY };
const PROPOSITIONS_ONLY = 'Предложения для новых экшенов';
const ALL_GROUPING_TAG_TAB = 'all_grouping_tag';
const GROUPING_TAG_SPLITTER = ',';

interface ITMPAction extends IAction {
    parent?: string;
    propositions: IActionProposition[];
    [FRONT_OPTIONS_KEY]?: { moved?: boolean; count?: number; oldParent?: string };
}

interface IActionsState {
    isRolesLoading: boolean;
    rolesError: Error | null;
    parentRoles: Record<string, string[]>;
    actionRoles: Record<string, IRole[]>;
    actionsIsLoading: boolean;
    actions: IAction[];
    actionsTypes: string[];
    tree: ActionsTree;
    actionsTypesWithCount: Dict<number>;
    actionIdFilter: string;
    attributeTabs: TabItem[];
    currentAttributeTab: string | null;
    loadingError: Error | null;
    isFilterDeprecated: boolean;
    isFullActions: boolean;
}

export default class Actions extends React.Component<{}, IActionsState> {
    state: IActionsState = {
        isRolesLoading: false,
        rolesError: null,
        parentRoles: {},
        actionRoles: {},
        actionsIsLoading: false,
        actions: [],
        actionsTypes: [],
        tree: {} as ActionsTree,
        actionsTypesWithCount: {},
        actionIdFilter: '',
        attributeTabs: [],
        currentAttributeTab: null,
        loadingError: null,
        isFilterDeprecated: true,
        isFullActions: false,
    };
    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

    vTree: React.RefObject<any>;

    constructor(props) {
        super(props);
        this.vTree = React.createRef();
    }

    componentDidMount(): void {
        this.getActions();
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    componentDidUpdate(prevProps: Readonly<{}>, prevState: Readonly<IActionsState>, snapshot?: any) {
        if (prevState.isFullActions !== this.state.isFullActions) {
            this.getActions();
        }

        if(prevState.isFilterDeprecated !== this.state.isFilterDeprecated) {
            this.vTree.current.getPlaneTree(false);
        }
    }

    getActions(success = false) {
        this.setState({ actionsIsLoading: !success, loadingError: null }, () => {
            this.request.exec(REQUESTS.GET_ACTIONS, {
                queryParams: {
                    report: !this.state.isFullActions ? 'compact' : null,
                },
            })
                .then(response => {
                    const activeActions: IAction[] = response?.report || [];
                    const deprecatedActions: IAction[] = response?.report_deprecated || [];
                    let actions = [...activeActions, ...deprecatedActions];
                    actions = actions.sort((a, b) => a.action_id.localeCompare(b.action_id));

                    const actionsTypes = Array.from(new Set(actions.map(action => {
                        if (!action.action_type && action.is_proposition_only) {
                            return PROPOSITIONS_ONLY;
                        }

                        return action.action_type;
                    })))
                        .sort();

                    const actionsTypesWithCount = actions.reduce((result: Dict<number>, action) => {
                        const action_type = action.action_type || PROPOSITIONS_ONLY;
                        if (!result.hasOwnProperty(action_type)) {
                            result[action_type] = 1;
                        } else {
                            ++result[action_type];
                        }

                        return result;
                    }, {});

                    const tree = new ActionsTree(ROOT);

                    // START: Normalize action array for making tree
                    const normalizedActions: ITMPAction[] = [];

                    const isParent = (tmpAction: IAction) => (searchAction: IAction) => {
                        return tmpAction.parent === searchAction.action_id;
                    };

                    const tmpActions: IAction[] = actions
                        .map(action => deepCopy(action));

                    const BORDER = 10;
                    while (tmpActions.length) {
                        const tmpAction: ITMPAction = tmpActions.splice(0, 1)[0];

                        if (!tmpAction.hasOwnProperty('parent')) {
                            normalizedActions.unshift(tmpAction);
                            continue;
                        }

                        if (normalizedActions.some(isParent(tmpAction))) {
                            normalizedActions.push(tmpAction);
                            continue;
                        }

                        if (tmpActions.some(isParent(tmpAction))) { // если есть в хвосте потенциальные родители

                            if (tmpAction.hasOwnProperty(FRONT_OPTIONS_KEY)) {
                                if (tmpAction?.__front_options?.moved) {
                                    if (tmpAction?.__front_options.count
                                        && tmpAction.__front_options.count < BORDER) {
                                        tmpAction.__front_options.count++;
                                        tmpActions.push(tmpAction);
                                    } else {
                                        const oldParent = tmpAction.parent;
                                        tmpAction.__front_options = { oldParent };
                                        tmpAction.parent = undefined;
                                        normalizedActions.splice(0, 0, tmpAction);
                                    }
                                }
                            } else {
                                tmpAction.__front_options = { moved: true, count: 1 };
                                tmpActions.push(tmpAction);
                            }
                        } else {
                            const oldParent = tmpAction.parent;
                            tmpAction.__front_options = { oldParent };
                            tmpAction.parent = undefined;
                            normalizedActions.splice(0, 0, tmpAction);
                        }
                    }
                    // END:Normalize action array for making tree

                    normalizedActions.forEach(action => {
                        tree.add(action, action.parent || ROOT.action_id, action[FRONT_OPTIONS_KEY]);
                    });

                    tree._root.children = actionsTypes.map(actionType => {
                        const node: Node = {
                            children: [],
                            data: { action_id: actionType },
                            options: {},
                            parent: null,
                        };
                        let children = tree._root.children
                            .filter(childNode => {
                                if (!node.data?.action_type
                                    && childNode.data?.is_proposition_only
                                    && actionType === PROPOSITIONS_ONLY) {
                                    return true;
                                }

                                return childNode.data && childNode.data.action_type === actionType;
                            })
                            .sort((childNode1, childNode2) => {
                                return childNode1.data?.action_id.localeCompare(childNode2.data?.action_id);
                            });
                        children = children.map(childNode => {
                            childNode.parent = node;

                            return childNode;
                        });

                        node.children = children;

                        return node;
                    });

                    const attributeTabs = this.getAttributesTabsFromTree(tree._root.children);

                    this.setState({
                        actionsIsLoading: false,
                        actions,
                        actionsTypes,
                        tree,
                        actionsTypesWithCount,
                        attributeTabs,
                        currentAttributeTab: ALL_GROUPING_TAG_TAB,
                    }, () => {
                        success && this.vTree.current.getPlaneTree(false);
                    });
                })
                .catch(loadingError => {
                    this.setState({ loadingError, actionsIsLoading: false });
                });
        });
    }

    getAttributesTabsFromTree(tree: Node[]): TabItem[] {
        const getAttributesFromTree = (nodesArr: Node[]) => {
            return nodesArr.reduce((result: string[], node) => {
                const grouppingTagsString = node?.data?.action_meta?.groupping_tags ?? null;
                if (grouppingTagsString) {
                    const groupingTags = node.data?.action_meta.groupping_tags
                        ?.replace(/\*/ig, GROUPING_TAG_SPLITTER);
                    result.push(...groupingTags.split(GROUPING_TAG_SPLITTER));
                }

                const nodeChildren = node?.children ?? null;
                if (nodeChildren) {
                    result.push(...getAttributesFromTree(nodeChildren));
                }

                return result;
            }, []);
        };

        let attributes = [ALL_GROUPING_TAG_TAB, ...getAttributesFromTree(tree)];
        attributes = Array.from(new Set(attributes));

        return attributes.map(attribute => {
            return {
                name: attribute,
                link: attribute,
                active: true,
            };
        });
    }

    getRoles() {
        this.setState({ isRolesLoading: true }, () => {
            this.request.exec(REQUESTS.GET_ROLES)
                .then(response => {
                    const rolesArray = response.report;
                    const parentRoles = {};
                    const actionRoles = {};

                    rolesArray.forEach(role => {
                        if (parentRoles[role.role_id]) {
                            parentRoles[role.role_id].push(...role.parent_role_ids);
                        } else {
                            parentRoles[role.role_id] = role.parent_role_ids;
                        }

                        if (role.actions.length) {
                            role.actions.forEach(action => {
                                if (actionRoles[action.action_id]) {
                                    actionRoles[action.action_id].push(role);
                                } else {
                                    actionRoles[action.action_id] = [role];
                                }
                            });
                        }
                    });

                    this.setState({ parentRoles, actionRoles, isRolesLoading: false });
                })
                .catch(rolesError => {
                    this.setState({
                        rolesError,
                        isRolesLoading: false,
                    });
                });
        });
    }

    onFilterChange(actionIdFilter: string) {
        this.setState({ actionIdFilter });
    }

    treeFilter(): ((node: IPlaneTreeItem<IAction>) => boolean) | null {
        const { actionIdFilter, currentAttributeTab } = this.state;

        if (actionIdFilter || currentAttributeTab !== ALL_GROUPING_TAG_TAB) {
            const filterLowercase = actionIdFilter.toLowerCase();

            return (action: IPlaneTreeItem<IAction>) => {
                const { data } = action;
                const { action_id = '', action_description = '', action_meta = {} } = data;

                const isFilteredById: boolean = actionIdFilter
                    ? action_id?.toLowerCase()?.includes?.(filterLowercase)
                    || action_description?.toLowerCase()?.includes?.(filterLowercase)
                    || action_meta?.tag_name?.toLowerCase()?.includes?.(filterLowercase)
                    : true;

                const isFilteredByGrouppingTag: boolean = currentAttributeTab !== ALL_GROUPING_TAG_TAB
                    ? action_meta?.groupping_tags?.replace?.(/\*/ig, GROUPING_TAG_SPLITTER)
                        ?.split(GROUPING_TAG_SPLITTER)
                        ?.includes(currentAttributeTab) ?? false
                    : true;

                return isFilteredById && isFilteredByGrouppingTag;
            };
        }

        return null;

    }

    selectTab(currentAttributeTab: string) {
        this.setState({ currentAttributeTab });
    }

    onAddActionClick() {
        location.hash = `/settings/actions/${CREATE_NEW_ACTION_ID}`;
    }

    treeBuilder(): IPlaneTreeItem<any>[] {
        const { isFilterDeprecated, actionsTypesWithCount } = this.state;
        const buildTree = (children: any[], parent = null, nestingLevel = 0, parentsIds: string[] = []) => {
            if (children.length) {
                let result = children;
                if (isFilterDeprecated) {
                    result = result.filter(child => {
                        const { data } = child;
                        const { deprecated } = data;

                        return !deprecated;
                    });
                }

                result = result.sort((child1, child2) => {
                    const name1 = child1?.data?.action_description || child1?.data?.action_id || '';
                    const name2 = child2?.data?.action_description || child2?.data?.action_id || '';

                    return name1.localeCompare(name2);
                });

                result = result.map(child => {
                    const { data, options } = child;
                    const { deprecated } = data;
                    const parentId = child?.parent?.data?.action_id ?? null;
                    const path = parentId
                        ? [...parentsIds, parentId, data?.action_id]
                        : [...parentsIds, data?.action_id];

                    return {
                        data,
                        meta: {
                            id: path.join('.'),
                            children: buildTree(child.children, child, nestingLevel + 1, parentsIds),
                            nestingLevel,
                            parent,
                            options,
                            className: deprecated ? style.deprecated : '',
                            parentsIds: parentId ? [...parentsIds, parentId] : [...parentsIds],
                            active: !actionsTypesWithCount?.hasOwnProperty(data?.action_id),
                        },
                    };
                });

                return result;
            }

            return null;

        };

        const { tree } = this.state;
        const rootChildren = tree?._root?.children ?? [];

        return buildTree(rootChildren) ?? [];
    }

    onFilterDeprecatedChange(isFilterDeprecated: boolean) {
        this.setState({ isFilterDeprecated });
    }

    setFullActions(isFullActions: boolean) {
        this.setState({ isFullActions });
    }

    render() {
        const {
            loadingError, actionsIsLoading, actionIdFilter,
            attributeTabs, currentAttributeTab, actionsTypesWithCount, isFilterDeprecated, isFullActions,
            isRolesLoading, rolesError, actionRoles, parentRoles,
        } = this.state;

        const rolesInfo = {
            isRolesLoading, rolesError, actionRoles, parentRoles,
        };

        return <div className={style.actions}>
            {loadingError
                ? <SimpleError error={loadingError} data={{ label: 'Ошибка при загрузке экшенов' }}/>
                : null}
            {actionsIsLoading
                ? <Spin/>
                : <>
                    <div className={style.filters}>
                        <div className={style.controls}>
                            <Input autoFocus
                                   className={style.action_id_filter}
                                   value={actionIdFilter}
                                   onChange={this.onFilterChange.bind(this)}
                                   placeholder={'Фильтр'}/>
                            <Button onClick={this.onAddActionClick.bind(this)}
                                    basic
                                    className={style.add_action_button}>Добавить action</Button>
                            <div className={style.deprecated_checkbox}>
                                <div className={style.title}>Скрывать deprecated:</div>
                                <Checkbox onChange={this.onFilterDeprecatedChange.bind(this)}
                                          checked={isFilterDeprecated}/>
                            </div>
                            <div className={style.deprecated_checkbox}>
                                <div className={style.title}>Полная информация об экшенах (медленно!!!)</div>
                                <Checkbox onChange={this.setFullActions.bind(this)}
                                          checked={isFullActions}/>
                            </div>
                        </div>
                        <Tabs tabs={attributeTabs}
                              collapsed
                              collapsedTitle={`groupping_tags: ${attributeTabs.length}`}
                              currentTab={currentAttributeTab || ALL_GROUPING_TAG_TAB}
                              selectTab={this.selectTab.bind(this)}/>

                    </div>
                    <VirtualTreeList treeBuilder={this.treeBuilder.bind(this)}
                                     ref={this.vTree}
                                     treeFilter={this.treeFilter.bind(this)}
                                     isFilterValueExist={!!actionIdFilter}
                                     initCollapsedLevels={[0]}
                                     treeListItem={[ActionTreeListItem, {
                                         actionsTypesWithCount,
                                         rolesInfo,
                                         getRoles: this.getRoles.bind(this),
                                     }]}/>
                </>}
            <Switch>
                <Route path="/settings/actions/:action_id/:proposition_index?"
                       render={() => <AddActionModal updateAllActions={this.getActions.bind(this, true)}/>}/>
            </Switch>
        </div>;
    }
}
