import diff from 'deep-diff';
import * as React from 'react';

import { Button, ButtonTypes, CancelButton } from '../../../../ui/Button';
import { Confirm, Window } from '../../../../ui/FullModal';
import * as coreStyle from '../../../../ui/index.css';
import { ConstantsKey } from '../../../../utils/fetchConstants';
import { Request2 } from '../../../../utils/request';
import { deepCopy } from '../../../../utils/utils';
import { FormConstructor } from '../../../FormConstructor';
import Spin from '../../../Spin';
import { REQUESTS } from '../../Actions/request';
import { ProposeCommentModal } from '../../ProposeCommentModal';
import { SETTINGS_REQUESTS } from '../request';

interface IConstants {
    constants: (traits: ConstantsKey[], ids: string) => Promise<any>;
}

interface IEditActionsModalProps extends IConstants {
    onClose: () => void;
    action_type: string;
    selectedActions: string[];
}

interface IEditActionsModalState {
    iface: Record<string, any>;
    actionSchemeIsLoading: boolean;
    initialData: Record<string, any>;
    currentData: Record<string, any>;
    fullActionsData: Record<string, any>[] | null;
    isFormValid: boolean;
    isFormChanged: boolean;
    confirmIsOpen: boolean;
    showCommentModal: boolean;
    errors: {
        message: Error | null;
        actions: string[];
    } | null;
    isAddingPropositions: boolean;
}

const FIELD_TYPES = {
    default: 'default',
    id: 'id',
    key: 'key',
};

export default class EditActionsModal extends React.Component<IEditActionsModalProps, IEditActionsModalState> {
    state: IEditActionsModalState = {
        iface: {},
        actionSchemeIsLoading: true,
        initialData: {},
        currentData: {},
        fullActionsData: null,
        isFormValid: false,
        isFormChanged: false,
        confirmIsOpen: false,
        showCommentModal: false,
        errors: null,
        isAddingPropositions: false,
    };
    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });
    booleanFields = new Set();

    componentDidMount() {
        this.getActionSchema();
    }

    getActionSchema() {
        const { action_type } = this.props;
        this.props.constants([ConstantsKey.IFACE], action_type)
            .then(responseConstants => {
                this.setState({
                    iface: responseConstants?.iface,
                    actionSchemeIsLoading: false,
                });
            })
            .catch(error => {
                this.setState({
                    actionSchemeIsLoading: false,
                });
            });
    }

    getActionsData(callback) {
        this.request.exec(REQUESTS.GET_ACTIONS)
            .then(responseActions => {
                this.setState({
                    fullActionsData: responseActions?.report,
                }, () => {
                    callback?.();
                });
            })
            .catch((error) => {
                this.setState({ errors: { message: error, actions: ['getActionsData'] } });
            });

    }

    onFormChange(currentData: Record<string, any>, isFormValid: boolean, isFormChanged: boolean) {
        this.setState({ currentData, isFormValid, isFormChanged });

        if (!isFormChanged) {
            this.setState({ initialData: currentData });
        }
    }

    getFieldType(path: string[]) {
        const { action_type } = this.props;
        const { iface } = this.state;
        const schema = iface?.[action_type] || {};

        let element = { ...schema };
        path.map((step) => {
            if (element[step]?.type === 'structure') {
                element = element[step].structure;
            } else if (element[step]?.type === 'array_types') {
                element = element[step].array_type;
            }
        });

        if (Object.keys(element).includes('id')) {
            return FIELD_TYPES.id;
        }

        if (Object.keys(element).includes('key')) {
            return FIELD_TYPES.key;
        }

        return FIELD_TYPES.default;
    }

    getUniqueFields(difference) {
        const uniqueFields: Record<string, string> = {};

        difference.map((el) => {
            const fieldType = this.getFieldType(el.path);

            if (fieldType !== FIELD_TYPES.default) {
                uniqueFields[el.path.join(',')] = fieldType;
            }
        });

        return uniqueFields;
    }

    prepareData(difference, data, uniqueFields) {
        const formattedDiff = { ...data };

        difference.map((el) => {
            const path = el.path.join(',');
            let element = formattedDiff;
            let currentData = { ...this.state.currentData };

            el.path.map((step, i) => {
                if (i === el.path.length - 1) {
                    if (el.item && Object.keys(uniqueFields).includes(path)) {
                        const index = element[step]
                            ?.findIndex((item) => item[uniqueFields[path]] === el.item.lhs[uniqueFields[path]]);

                        if (index !== -1) {
                            element[step][index] = el.item.lhs;
                        } else {
                            element[step] = element[step] ? [...element[step], el.item.lhs] : [el.item.lhs];
                        }
                    } else {
                        const value = this.booleanFields.has(step)
                            ? el.lhs === 'true'
                            : el.lhs;
                        element[step] = el.item ? currentData[step] : value;
                    }
                } else {
                    element[step] = element[step] ? { ...element[step] } : {};
                }

                element = element[step];
                currentData = currentData[step];
            });
        });

        return formattedDiff;
    }

    showConfirm(confirmIsOpen) {
        this.setState({ confirmIsOpen, errors: null });
    }

    setComment(comment) {
        if (!this.state.fullActionsData) {
            this.getActionsData(() => {
                this.sendProposeRequests(comment);
            });
        } else {
            this.sendProposeRequests(comment);
        }
    }

    sendProposeRequests(comment: string) {
        const { fullActionsData, currentData, initialData } = this.state;
        const { selectedActions, action_type } = this.props;
        const difference = diff.diff(currentData, initialData);
        const uniqueFields = this.getUniqueFields(difference);
        let requests: Promise<unknown>[] = [];
        const dataToSend = {};

        fullActionsData?.map((el) => {
            if (selectedActions.includes(el.action_id)) {
                dataToSend[el.action_id] = el;
            }
        });

        this.setState({ isAddingPropositions: true, errors: null }, () => {
            const { selectedActions } = this.props;
            const errors: any = {};

            requests = selectedActions.map((action) => {
                const body = this.prepareData(difference, dataToSend[action], uniqueFields);

                return this.request.exec(REQUESTS.ADD_ACTION_PROPOSE, {
                    queryParams: { comment: comment || `Предложение для группы ${action_type}` },
                    body,
                });
            });

            Promise.allSettled(requests)
                .then((results) => {
                    results.map((res, i) => {
                        if (res.status === 'rejected') {
                            errors.message = res.reason.data?.error_details
                                ? JSON.stringify(res.reason.data.error_details)
                                : res.reason;
                            errors.actions = errors.actions
                                ? [...errors.actions, selectedActions[i]]
                                : [selectedActions[i]];
                        }
                    });

                    this.setState({
                        isAddingPropositions: false,
                        errors: errors.message ? errors : null,
                    }, () => {
                        if (!this.state.errors) {
                            this.showCommentModal(false);
                            this.props.onClose();
                        }
                    });
                });
        });
    }

    accept() {
        this.showConfirm(false);
        this.showCommentModal(true);
    }

    showCommentModal(showCommentModal: boolean) {
        this.setState({ showCommentModal });
    }

    updateSchema(schema: Record<string, any>) {
        const newSchema: Record<string, any> = deepCopy(schema) ?? {};

        Object.entries(newSchema)?.map((el) => {
            delete newSchema[el[0]].default;

            if (el[1].type === 'structure') {
                newSchema[el[0]].structure = this.updateSchema(el[1].structure);
            }

            if (el[1]?.required) {
                newSchema[el[0]].default = ' ';
            }

            if (el[1].type === 'bool') {
                newSchema[el[0]] = {
                    ...el[1],
                    type: 'variants',
                    display_name: `${el[1].display_name} (boolean)`,
                    variants: [{ value: 'true', text: 'активен' }, { value: 'false', text: 'неактивен' }],
                    default: undefined,
                };
                this.booleanFields.add(el[0]);
            }
        });

        return newSchema;
    }

    render() {
        const { onClose, action_type, selectedActions } = this.props;
        const {
            iface, actionSchemeIsLoading, isFormChanged, confirmIsOpen,
            showCommentModal, errors, isAddingPropositions,
        } = this.state;

        const schema = iface?.[action_type]
            ? this.updateSchema(iface?.[action_type])
            : null;
        delete schema?.action_id;

        return <Window onClose={onClose.bind(this)}
                       title={`Редактировать группу ${action_type}`}>
            {actionSchemeIsLoading && <Spin size={'l'}/>}
            {schema && <FormConstructor schema={schema}
                                        onChange={this.onFormChange.bind(this)}/>}
            <div className={coreStyle.button_container}>
                <CancelButton onClick={onClose.bind(this)}/>
                <Button disabled={!isFormChanged}
                        colorType={ButtonTypes.positive}
                        onClick={this.showConfirm.bind(this, true)}>Заявка</Button>
                {confirmIsOpen
                    ? <Confirm accept={this.accept.bind(this)}
                               onClose={this.showConfirm.bind(this, false)}
                               error={''}
                               question={`Добавить предложение к экшенам: ${selectedActions?.join(', ')}?`}/>
                    : null}
                {showCommentModal
                    ? <ProposeCommentModal setComment={this.setComment.bind(this)}
                                           onClose={this.showCommentModal.bind(this, false)}
                                           error={errors && new Error(`${errors?.actions.join(', ')} — ${errors?.message}`)}
                                           isWorking={isAddingPropositions}/>
                    : null}
            </div>
        </Window>;
    }
}
