import * as React from 'react';
import { Route, Switch } from 'react-router-dom';

import { Button } from '../../../ui/Button';
import Select, { IOptionInfo } from '../../../ui/Select';
import { IPlaneTreeItem, VirtualTreeList } from '../../../ui/VirtualTreeList';
import { ConstantsKey } from '../../../utils/fetchConstants';
import { Request2 } from '../../../utils/request';
import { deepCopy } from '../../../utils/utils';
import { IRole } from '../../Clients/UserRolesView/types';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { ActionsTree } from '../Actions/ActionsTree/component';
import { Node } from '../Actions/ActionsTree/constants';
import AddActionModal from '../Actions/AddActionModal';
import { FRONT_OPTIONS_KEY, ROOT_KEY } from '../Actions/constants';
import { IAction, IActionProposition } from '../Actions/types';
import ActionsUpdateListItem from './ActionsUpdateListItem';
import { DEFAULT_ACTION } from './constants';
import EditActionsModal from './EditActionsModal';
import * as style from './index.css';
import { REQUESTS, SETTINGS_REQUESTS } from './request';

const ROOT = { action_id: ROOT_KEY };
const PROPOSITIONS_ONLY = 'Предложения для новых экшенов';

interface IConstants {
    constants: (traits: ConstantsKey[], ids: string) => Promise<any>;
}

interface ITMPAction extends IAction {
    parent?: string;
    propositions: IActionProposition[];
    [FRONT_OPTIONS_KEY]?: { moved?: boolean; count?: number; oldParent?: string };
}

interface IActionsUpdateState {
    isRolesLoading: boolean;
    rolesError: Error | null;
    parentRoles: Record<string, string[]>;
    actionRoles: Record<string, IRole[]>;
    actionsIsLoading: boolean;
    actions: IAction[];
    selectedActions: Record<string, boolean>;
    actionsTypes: IOptionInfo[];
    tree: ActionsTree;
    actionsTypesWithCount: Record<string, number>;
    selectedActionType: string;
    loadingError: Error | null;
    isModalOpen: boolean;
}

export default class ActionsUpdate extends React.Component<IConstants, IActionsUpdateState> {
    state: IActionsUpdateState = {
        isRolesLoading: false,
        rolesError: null,
        parentRoles: {},
        actionRoles: {},
        actionsIsLoading: false,
        actions: [],
        selectedActions: {},
        actionsTypes: [],
        tree: {} as ActionsTree,
        actionsTypesWithCount: {},
        selectedActionType: DEFAULT_ACTION,
        loadingError: null,
        isModalOpen: false,
    };
    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

    vTree: React.RefObject<any>;

    constructor(props) {
        super(props);
        this.vTree = React.createRef();
    }

    componentDidMount(): void {
        this.getActions();
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    getActions(success = false) {
        this.setState({ actionsIsLoading: !success, loadingError: null }, () => {
            this.request.exec(REQUESTS.GET_ACTIONS, {
                queryParams: { report: 'compact' },
            })
                .then(response => {
                    const activeActions: IAction[] = response?.report || [];
                    const deprecatedActions: IAction[] = response?.report_deprecated || [];
                    let actions = [...activeActions, ...deprecatedActions];
                    actions = actions.sort((a, b) => a.action_id.localeCompare(b.action_id));

                    const actionsTypes = Array.from(new Set(actions.map(action => {
                        if (!action.action_type && action.is_proposition_only) {
                            return PROPOSITIONS_ONLY;
                        }

                        return action.action_type;
                    }))).sort();

                    const actionsTypesWithCount = actions.reduce((result: Record<string, number>, action) => {
                        const action_type = action.action_type || PROPOSITIONS_ONLY;
                        if (!result.hasOwnProperty(action_type)) {
                            result[action_type] = 1;
                        } else {
                            ++result[action_type];
                        }

                        return result;
                    }, {});

                    const tree = new ActionsTree(ROOT);

                    // START: Normalize action array for making tree
                    const normalizedActions: ITMPAction[] = [];

                    const isParent = (tmpAction: IAction) => (searchAction: IAction) => {
                        return tmpAction.parent === searchAction.action_id;
                    };

                    const tmpActions: IAction[] = actions
                        .map(action => deepCopy(action));

                    const BORDER = 10;
                    while (tmpActions.length) {
                        const tmpAction: ITMPAction = tmpActions.splice(0, 1)[0];

                        if (!tmpAction.hasOwnProperty('parent')) {
                            normalizedActions.unshift(tmpAction);
                            continue;
                        }

                        if (normalizedActions.some(isParent(tmpAction))) {
                            normalizedActions.push(tmpAction);
                            continue;
                        }

                        if (tmpActions.some(isParent(tmpAction))) { // если есть в хвосте потенциальные родители

                            if (tmpAction.hasOwnProperty(FRONT_OPTIONS_KEY)) {
                                if (tmpAction?.__front_options?.moved) {
                                    if (tmpAction?.__front_options.count
                                        && tmpAction.__front_options.count < BORDER) {
                                        tmpAction.__front_options.count++;
                                        tmpActions.push(tmpAction);
                                    } else {
                                        const oldParent = tmpAction.parent;
                                        tmpAction.__front_options = { oldParent };
                                        tmpAction.parent = undefined;
                                        normalizedActions.splice(0, 0, tmpAction);
                                    }
                                }
                            } else {
                                tmpAction.__front_options = { moved: true, count: 1 };
                                tmpActions.push(tmpAction);
                            }
                        } else {
                            const oldParent = tmpAction.parent;
                            tmpAction.__front_options = { oldParent };
                            tmpAction.parent = undefined;
                            normalizedActions.splice(0, 0, tmpAction);
                        }
                    }
                    // END:Normalize action array for making tree

                    normalizedActions.forEach(action => {
                        tree.add(action, action.parent || ROOT.action_id, action[FRONT_OPTIONS_KEY]);
                    });

                    tree._root.children = actionsTypes.map(actionType => {
                        const node: Node = {
                            children: [],
                            data: { action_id: actionType },
                            options: {},
                            parent: null,
                        };
                        let children = tree._root.children
                            .filter(childNode => {
                                if (!node.data?.action_type
                                    && childNode.data?.is_proposition_only
                                    && actionType === PROPOSITIONS_ONLY) {
                                    return true;
                                }

                                return childNode.data && childNode.data.action_type === actionType;
                            })
                            .sort((childNode1, childNode2) => {
                                return childNode1.data?.action_id.localeCompare(childNode2.data?.action_id);
                            });
                        children = children.map(childNode => {
                            childNode.parent = node;

                            return childNode;
                        });

                        node.children = children;

                        return node;
                    });

                    this.setState({
                        actionsIsLoading: false,
                        actions,
                        actionsTypes: actionsTypes.map((el) => ({ value: el })),
                        tree,
                        actionsTypesWithCount,
                    }, () => {
                        success && this.vTree.current.getPlaneTree(false);
                    });
                })
                .catch(loadingError => {
                    this.setState({ loadingError, actionsIsLoading: false });
                });
        });
    }

    getRoles() {
        this.setState({ isRolesLoading: true }, () => {
            this.request.exec(REQUESTS.GET_ROLES)
                .then(response => {
                    const rolesArray = response.report;
                    const parentRoles = {};
                    const actionRoles = {};

                    rolesArray.forEach(role => {
                        if (parentRoles[role.role_id]) {
                            parentRoles[role.role_id].push(...role.parent_role_ids);
                        } else {
                            parentRoles[role.role_id] = role.parent_role_ids;
                        }

                        if (role.actions.length) {
                            role.actions.forEach(action => {
                                if (actionRoles[action.action_id]) {
                                    actionRoles[action.action_id].push(role);
                                } else {
                                    actionRoles[action.action_id] = [role];
                                }
                            });
                        }
                    });

                    this.setState({ parentRoles, actionRoles, isRolesLoading: false });
                })
                .catch(rolesError => {
                    this.setState({
                        rolesError,
                        isRolesLoading: false,
                    });
                });
        });
    }

    onSelectChange(selectedActionType: string) {
        this.setState({ selectedActionType: selectedActionType || DEFAULT_ACTION, selectedActions: {} });
    }

    treeFilter(): ((node: IPlaneTreeItem<IAction>) => boolean) | null {
        const { selectedActionType } = this.state;
        const filter = selectedActionType || DEFAULT_ACTION;

        return (action: IPlaneTreeItem<IAction>) => {
            return action?.data?.action_type === filter;
        };
    }

    treeBuilder(): IPlaneTreeItem<any>[] {
        const { actionsTypesWithCount } = this.state;
        const buildTree = (children: any[], parent = null, nestingLevel = 0, parentsIds: string[] = []) => {
            if (children.length) {
                let result = children.filter(child => {
                    const { data } = child;
                    const { deprecated } = data;

                    return !deprecated;
                });

                result = result.sort((child1, child2) => {
                    const name1 = child1?.data?.action_description || child1?.data?.action_id || '';
                    const name2 = child2?.data?.action_description || child2?.data?.action_id || '';

                    return name1.localeCompare(name2);
                });

                result = result.map(child => {
                    const { data, options } = child;
                    const { deprecated } = data;
                    const parentId = child?.parent?.data?.action_id ?? null;
                    const path = parentId
                        ? [...parentsIds, parentId, data?.action_id]
                        : [...parentsIds, data?.action_id];

                    return {
                        data,
                        meta: {
                            id: path.join('.'),
                            children: buildTree(child.children, child, nestingLevel + 1, parentsIds),
                            nestingLevel,
                            parent,
                            options,
                            className: deprecated ? style.deprecated : '',
                            parentsIds: parentId ? [...parentsIds, parentId] : [...parentsIds],
                            active: !actionsTypesWithCount?.hasOwnProperty(data?.action_id),
                        },
                    };
                });

                return result;
            }

            return null;

        };

        const { tree } = this.state;
        const rootChildren = tree?._root?.children ?? [];

        return buildTree(rootChildren) ?? [];
    }

    onCheckboxChange(value) {
        const selectedActions = { ...this.state.selectedActions, ...value };
        this.setState({ selectedActions });
    }

    showModal(showModal) {
        this.setState({ isModalOpen: showModal });
    }

    render() {
        const {
            loadingError, actionsIsLoading, selectedActionType, actionsTypesWithCount, isModalOpen,
            isRolesLoading, rolesError, actionRoles, parentRoles, actionsTypes, selectedActions,
        } = this.state;

        const rolesInfo = { isRolesLoading, rolesError, actionRoles, parentRoles };

        return <div className={style['actions-update']}>
            {loadingError
                ? <SimpleError error={loadingError} data={{ label: 'Ошибка при загрузке экшенов' }}/>
                : null}
            {actionsIsLoading
                ? <Spin/>
                : <>
                    <div className={style.filters}>
                        <div className={style.controls}>
                            <Select className={style.action_id_filter}
                                    options={actionsTypes}
                                    initialValues={[DEFAULT_ACTION]}
                                    onSelect={this.onSelectChange.bind(this)}
                                    placeholder={'Фильтр'}/>
                            <Button disabled={!Object.values(selectedActions).filter((el) => el).length}
                                    onClick={this.showModal.bind(this, true)}>
                                Редактировать
                            </Button>
                        </div>
                    </div>
                    <VirtualTreeList treeBuilder={this.treeBuilder.bind(this)}
                                     ref={this.vTree}
                                     treeFilter={this.treeFilter.bind(this)}
                                     isFilterValueExist={!!selectedActionType}
                                     initCollapsedLevels={[0]}
                                     treeListItem={[ActionsUpdateListItem, {
                                         actionsTypesWithCount,
                                         rolesInfo,
                                         getRoles: this.getRoles.bind(this),
                                         className: style.item_wrapper,
                                         onChange: this.onCheckboxChange.bind(this),
                                         selectedActions,
                                     }]}/>
                </>}
            {isModalOpen && <EditActionsModal onClose={this.showModal.bind(this, false)}
                                              action_type={selectedActionType}
                                              selectedActions={Object.keys(selectedActions)
                                                  .filter((key) => {
                                                      return selectedActions[key] && key !== selectedActionType;
                                                  })}/>}

            <Switch>
                <Route path="/settings/actions-update/:action_id/:proposition_index?"
                       render={() => <AddActionModal updateAllActions={this.getActions.bind(this, true)}
                                                     isForUpdate={true}/>}/>
            </Switch>
        </div>;
    }
}
