import * as React from 'react';

import { Dict } from '../../../../../types';
import { EMPTY_DATA } from '../../../../constants';
import { Window } from '../../../../ui/FullModal';
import { Link } from '../../../../ui/Link';
import { Request2 } from '../../../../utils/request';
import { SimpleError } from '../../../SimpleError';
import Spin from '../../../Spin';
import BotModal from '../BotModal';
import * as style from '../index.css';
import { BOTS_REQUESTS, REQUESTS } from '../request';

interface IBotGroupAlert {
    bp_enabled: boolean;
    sharding_policy: {
        max_shard: number;
        min_shard: number;
    };
    action: {
        actions?: Dict<any> [];
        action_type: string;
        save_fetched?: boolean;
        state_tag?: string;
        additional_tags?: {
            comment: string;
            tag: string;
        }[];
        node_type?: string;
    };
    bp_name: string;
    group_name: string;
    checkers: Dict<any> [];
    bp_type: string;
    fetchers: Dict<any> [];
}

interface IBotGroup {
    alerts: IBotGroupAlert[];
    group_name: string;
}

interface IBotGroupModalProps {
    onClose: () => void;
    groupName: string;
}

interface IBotGroupModalState {
    isLoading: boolean;
    loadingError: Error | null;
    groups: IBotGroup[];
    currentGroup: IBotGroup | null;
    oddAlerts: IBotGroupAlert[];
    upsertError: Error | null;
    isWorking: boolean;
    currentEditBotName: string | null;
}

export default class BotGroupModal extends React.Component<IBotGroupModalProps, IBotGroupModalState> {
    state: IBotGroupModalState = {
        isLoading: false,
        loadingError: null,
        groups: [],
        currentGroup: null,
        oddAlerts: [],
        upsertError: null,
        isWorking: false,
        currentEditBotName: null,
    };
    request = new Request2({ requestConfigs: BOTS_REQUESTS });

    componentDidMount(): void {
        this.getGroups();
    }

    componentDidUpdate(prevProps: Readonly<IBotGroupModalProps>): void {
        if (this.props.groupName !== prevProps.groupName) {
            this.getGroups();
        }
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    getGroups() {
        const { groupName } = this.props;
        this.setState({ isLoading: true, loadingError: null }, () => {
            this.request.exec(REQUESTS.GET_BOT_GROUPS)
                .then(response => {
                    const groups: IBotGroup[] = response?.groups ?? [];
                    const currentGroup = groups.filter(group => group.group_name === groupName)[0] ?? null;

                    let { alerts } = currentGroup;
                    const sortedAlerts: IBotGroupAlert[] = [];
                    let oddAlerts: IBotGroupAlert[] = [];

                    while (alerts.length > 0) {
                        if (sortedAlerts.length === 0) {
                            const controlIndex = alerts.findIndex((alert) => {
                                let action: any = alert.action;
                                if (Array.isArray(action)) {
                                    action = action.filter(actionItem => actionItem.action_type = 'chain_node')[0];
                                }

                                return action.node_type === 'control';
                            });

                            if (controlIndex > -1) {
                                sortedAlerts.push(alerts[controlIndex]);
                                alerts = [...alerts.slice(0, controlIndex), ...alerts.slice(controlIndex + 1)];
                            } else {
                                oddAlerts = alerts;
                                alerts = [];
                            }

                        } else {
                            const lastSortedAlert = sortedAlerts[sortedAlerts.length - 1];
                            const { bp_name: lastSortedAlertName } = lastSortedAlert;

                            const nextAlertIndex = alerts.findIndex((alert) => {
                                let action: any = alert.action;
                                if (action.actions && Array.isArray(action.actions)) {
                                    action = action.actions
                                        .filter(actionItem => actionItem.action_type = 'chain_node')[0];
                                }

                                return action?.acceptable_states?.includes(lastSortedAlertName)
                                    || action?.acceptable_states === lastSortedAlertName;
                            });

                            if (nextAlertIndex > -1) {
                                sortedAlerts.push(alerts[nextAlertIndex]);
                                alerts = [...alerts.slice(0, nextAlertIndex), ...alerts.slice(nextAlertIndex + 1)];
                            } else {
                                oddAlerts = alerts;
                                alerts = [];
                            }
                        }
                    }

                    currentGroup.alerts = sortedAlerts;

                    this.setState({ groups, isLoading: false, currentGroup, oddAlerts });

                })
                .catch(loadingError => {
                    this.setState({ loadingError, isLoading: false });
                });
        });
    }

    upsertBot(state: Dict<any> ) {
        this.setState({ upsertError: null, isWorking: true }, () => {
            this.request.exec(REQUESTS.UPSERT_BOTS,
                {
                    body: {
                        backgrounds:
                            [Object.assign({}, state.formData, { bp_type: state.schema.toString() })],
                    },
                })
                .then(() => {
                    this.getGroups();
                    this.closeBotModal();
                })
                .catch(upsertError => {
                    this.setState({
                        upsertError, isWorking: false,
                    });
                });
        });
    }

    openBotModal(currentEditBotName: string) {
        this.setState({ currentEditBotName });
    }

    closeBotModal() {
        this.setState({ currentEditBotName: null });
    }

    render() {
        const { onClose } = this.props;
        const { isLoading, loadingError, currentGroup, oddAlerts, currentEditBotName, upsertError } = this.state;

        const alerts = currentGroup?.alerts || [];

        return <Window title={`Группа роботов ${currentGroup?.group_name ?? ''}`} onClose={onClose.bind(this)}>
            {isLoading
                ? <Spin/>
                : <div className={style.bot_group_modal}>
                    {loadingError
                        ? <SimpleError error={loadingError} data={{ label: 'Ошибка при загрузке групп роботов' }}/>
                        : null}
                    <div className={`${style.group_item} ${style.header}`}>
                        <div className={style.group_item_title}>Робот</div>
                        <div className={style.is_enabled}>Состояние</div>
                        <div className={style.state_tag}>Тег</div>
                        <div className={style.node_type}>Тип ноды</div>
                    </div>
                    {alerts.length
                        ? alerts.map((alert, index, arr) => {
                            let action: any = alert.action;
                            if (Array.isArray(action)) {
                                action = action.filter(actionItem => actionItem.action_type = 'chain_node')[0];
                            }

                            const isControl = action.node_type === 'control';

                            return <div key={alert.bp_name}>
                                <div className={style.group_item}>
                                    <Link onClick={this.openBotModal.bind(this, alert.bp_name)}
                                          className={style.group_item_title}
                                          title={alert.bp_name}>{alert.bp_name}</Link>
                                    <div className={`${style.is_enabled} `
                                    + ` ${alert.bp_enabled ? style.enabled : style.disabled}`}>
                                        {alert.bp_enabled ? 'Включён' : 'Выключен'}
                                    </div>
                                    <div className={style.state_tag}>{alert?.action?.additional_tags
                                        ?.map(additional_tag => additional_tag.tag).join(', ')
                                    ?? EMPTY_DATA}</div>
                                    <div className={style.node_type}>{alert?.action?.node_type ?? EMPTY_DATA}</div>
                                </div>
                                {isControl && action?.acceptable_states?.length
                                    ? <SimpleError error={new Error('Элемент типа control содержит непустой массив acceptable_states')}/>
                                    : null}
                                {index !== arr.length - 1
                                    ? <div className={style.arrow_container}>
                                        <div className={style.arrow}/>
                                    </div>
                                    : null}
                            </div>;
                        })
                        : <>
                            <h4>Не найдено элементов группы</h4>
                            {oddAlerts.length
                                ? <SimpleError error={new Error('Не найдено элемента с типом control')}/>
                                : null}
                        </>}
                    {oddAlerts.length
                        ? <>
                            <h3>Упс, кажется, цепочка оборвалась. Вот оставшиееся элементы:</h3>
                            {oddAlerts.map((alert) => {
                                return <div key={alert.bp_name}
                                            className={`${style.group_item} ${style.odd_group_item}`}>
                                    <Link onClick={this.openBotModal.bind(this, alert.bp_name)}
                                          className={style.group_item_title}
                                          title={alert.bp_name}>{alert.bp_name}</Link>
                                    <div className={`${style.is_enabled} `
                                    + ` ${alert.bp_enabled ? style.enabled : style.disabled}`}>
                                        {alert.bp_enabled ? 'Включён' : 'Выключен'}
                                    </div>
                                    <div className={style.state_tag}>{alert?.action?.additional_tags
                                        ?.map(additional_tag => additional_tag.tag).join(', ')
                                    ?? EMPTY_DATA}</div>
                                    <div className={style.node_type}>{alert?.action?.node_type ?? EMPTY_DATA}</div>
                                </div>;
                            })}
                        </> : null}
                </div>}
            {currentEditBotName !== null
                ? <BotModal onClose={this.closeBotModal.bind(this)}
                            updateBot={this.upsertBot.bind(this)}
                            botName={currentEditBotName}
                            error={upsertError}/>
                : null}
        </Window>;
    }
}
