import * as React from 'react';

import { Dict } from '../../../../../../types';
import { Button, ButtonTypes, CancelButton, SaveButton } from '../../../../../ui/Button';
import { Window } from '../../../../../ui/FullModal';
import * as coreStyle from '../../../../../ui/index.css';
import { Request2 } from '../../../../../utils/request';
import { deepCopy } from '../../../../../utils/utils';
import { FormConstructor } from '../../../../FormConstructor';
import { controlType, ISchemaItem } from '../../../../FormConstructor/types';
import Spin from '../../../../Spin';
import { IBot } from '../../component';
import * as style from '../../index.css';
import { BOTS_REQUESTS, REQUESTS } from '../../request';

interface IBotConfigModalProps {
    bot: IBot | Dict<any>;
    onSaveClick: (data: { filtersFormData: Dict<any>; itemsFormData: Dict<any>; actionsFormData: Dict<any> }) => void;
    onClose: () => void;
}

interface IBotConfigModalState {
    filters: Dict<any>;
    filtersInitialData: Dict<any>[];
    filtersFormData: Dict<any>[];
    filtersTypeSchema: Dict<ISchemaItem>;

    items: Dict<any>;
    itemsInitialData: Dict<any>[];
    itemsFormData: Dict<any>[];
    itemsTypeSchema: Dict<ISchemaItem>;

    actions: Dict<any>;
    actionsInitialData: Dict<any>;
    actionsFormData: Dict<any>;
    actionsTypeSchema: Dict<ISchemaItem>;

    isLoading: boolean;
    loadingError: Error | null;
}

export default class BotConfigModal extends React.Component<IBotConfigModalProps, IBotConfigModalState> {
    state: IBotConfigModalState = {
        filters: {},
        filtersInitialData: [],
        filtersFormData: [],
        filtersTypeSchema: {},

        items: {},
        itemsInitialData: [],
        itemsFormData: [],
        itemsTypeSchema: {},

        actions: {},
        actionsInitialData: [],
        actionsFormData: [],
        actionsTypeSchema: {},

        isLoading: false,
        loadingError: null,
    };
    request = new Request2({ requestConfigs: BOTS_REQUESTS });

    componentDidMount(): void {
        this.setState({ isLoading: true, loadingError: null }, () => {
            this.request.exec(REQUESTS.GET_BOT_CONFIGURATIONS)
                .then(response => {
                    const { filters = {}, items = {}, actions = {} } = response?.alerts_meta;

                    const { filtersInitialData, itemsInitialData, actionsInitialData } = this.getInitialData();

                    //Every filter has scheme for all filters types, so we get it
                    const filtersValue: Dict<any> = Object.values(filters)?.[0] as Dict<any> ?? {};
                    const filtersTypeSchema: Dict<ISchemaItem> = { type: filtersValue?.scheme?.type ?? {} };

                    const actionsTypeSchema: Dict<ISchemaItem> = {
                        action_type: {
                            read_only: false,
                            variants: Object.keys(actions),
                            type: controlType.variants,
                            multi_select: false,
                            display_name: 'Тип экшена',
                        },
                    };

                    this.setState({
                        filters,
                        items,
                        actions,
                        filtersInitialData,
                        itemsInitialData,
                        actionsInitialData,
                        filtersTypeSchema,
                        actionsTypeSchema,
                        isLoading: false,
                    });
                })
                .catch(loadingError => {
                    this.setState({ loadingError, isLoading: false });
                });
        });
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    //getting initial data
    getInitialData(): { filtersInitialData: any; itemsInitialData: any; actionsInitialData: any } {
        const { bot } = this.props;
        const botSettings = bot.bp_settings && deepCopy(bot.bp_settings) || null;

        return {
            filtersInitialData: botSettings?.fetchers || [],
            itemsInitialData: botSettings?.checkers || [],
            actionsInitialData: botSettings?.action && botSettings?.action.actions || [],
        };
    }

    onFilterFormChange(index: number, data: Dict<any>) {
        const { filtersFormData } = this.state;

        if (!filtersFormData[index]) {
            filtersFormData[index] = {};
        }

        if (!filtersFormData[index].config) {
            filtersFormData[index].config = {};
        }

        filtersFormData[index].config = deepCopy(data);
        this.setState({ filtersFormData }, () => {
            //after changing type, recollect items
            this.getAcceptableItemsIds();
        });

        if (data.type) {
            //if change type, update initial data to get new schema
            const { filtersInitialData } = this.state;

            if (!filtersInitialData[index]) {
                filtersInitialData[index] = { config: {} };
            }

            if (!filtersInitialData[index].config) {
                filtersInitialData[index].config = {};
            }

            filtersInitialData[index].config.type = data.type;
            this.setState({ filtersInitialData });
        }
    }

    onItemFormChange(index: number, data: Dict<any>) {
        const { itemsFormData } = this.state;

        if (!itemsFormData[index]) {
            itemsFormData[index] = {};
        }

        itemsFormData[index] = deepCopy(data);
        this.setState({ itemsFormData });

        if (data.fetcher_type || data.iterator_type) {
            //if change fetcher type, update initial data to get new schema
            const itemsInitialData: any = this.state.itemsInitialData;

            if (!itemsInitialData[index]) {
                itemsInitialData[index] = {};
            }

            itemsInitialData[index].fetcher_type = data.fetcher_type;
            itemsInitialData[index].iterator_type = data.iterator_type;
            this.setState({ itemsInitialData });
        }
    }

    onActionFormChange(index: number, data: Dict<any>) {
        const { actionsFormData } = this.state;

        if (!actionsFormData[index]) {
            actionsFormData[index] = {};
        }

        actionsFormData[index] = deepCopy(data);
        this.setState({ actionsFormData });

        if (data.action_type) {
            //if change action type, update initial data to get new schema
            const actionsInitialData: any = this.state.actionsInitialData;

            if (!actionsInitialData[index]) {
                actionsInitialData[index] = {};
            }

            actionsInitialData[index] = data;
            this.setState({ actionsInitialData });
        }
    }

    removeFilterItem(index: number) {
        let { filtersFormData } = this.state;
        let filtersInitialData = deepCopy(filtersFormData);

        filtersInitialData = [...filtersInitialData.slice(0, index), ...filtersInitialData.slice(index + 1)];
        filtersFormData = [...filtersFormData.slice(0, index), ...filtersFormData.slice(index + 1)];

        this.setState({ filtersInitialData, filtersFormData });
    }

    removeItem(index: number) {
        let { itemsFormData } = this.state;
        let itemsInitialData = deepCopy(itemsFormData);

        itemsInitialData = [...itemsInitialData.slice(0, index), ...itemsInitialData.slice(index + 1)];
        itemsFormData = [...itemsFormData.slice(0, index), ...itemsFormData.slice(index + 1)];

        this.setState({ itemsInitialData, itemsFormData });
    }

    removeActionItem(index: number) {
        let { actionsFormData } = this.state;
        let actionsInitialData = deepCopy(actionsFormData);

        actionsInitialData = [...actionsInitialData.slice(0, index), ...actionsInitialData.slice(index + 1)];
        actionsFormData = [...actionsFormData.slice(0, index), ...actionsFormData.slice(index + 1)];

        this.setState({ actionsInitialData, actionsFormData });
    }

    addFilterItem() {
        const { filtersInitialData } = this.state;
        filtersInitialData.push({});
        this.setState({ filtersInitialData });
    }

    addItem() {
        const { itemsInitialData } = this.state;
        itemsInitialData.push({});
        this.setState({ itemsInitialData });
    }

    addActionItem(index: number | null) {
        const { actionsInitialData } = this.state;
        if (index === null) {
            actionsInitialData.unshift({});
        } else {
            actionsInitialData.splice(index + 1, 0, {});
        }

        this.setState({ actionsInitialData });
    }

    getAcceptableItemsIds() {
        const { filtersFormData, filters } = this.state;

        let filtersTypes = filtersFormData
            .map(filtersFormItem => {
                return filtersFormItem.config.type;
            })
            .filter((type: string) => type) ?? [];
        filtersTypes = Array.from(new Set(filtersTypes));

        let acceptableItems = filtersTypes.reduce((result: any[], type: string) => {
            const items = filters?.[type]?.acceptable_items ?? [];
            result.push(...items);

            return result;
        }, []);

        acceptableItems = Array.from(new Set(acceptableItems));

        const fetcherTypeVariants = Object.keys(filters);

        const itemsTypeSchema: Dict<ISchemaItem> = {
            fetcher_type: {
                variants: fetcherTypeVariants,
                type: controlType.variants,
                display_name: 'Фетчер, к которому относится итератор',
            },
            iterator_type: {
                variants: acceptableItems,
                type: controlType.variants,
                display_name: 'Тип итератора',
            },
        };

        this.setState({ itemsTypeSchema });
    }

    onSaveClick() {
        let { filtersFormData, itemsFormData, actionsFormData } = this.state;
        actionsFormData = actionsFormData.filter(obj => Object.keys(obj).length !== 0);

        this.props.onSaveClick?.({ filtersFormData, itemsFormData, actionsFormData });
    }

    render() {
        const { bot, onClose } = this.props;
        const { isLoading, loadingError } = this.state;
        const {
            filters, filtersInitialData, filtersTypeSchema,
            items, itemsInitialData, itemsTypeSchema,
            actions, actionsInitialData, actionsTypeSchema,
        } = this.state;

        return <Window title={`Конфигурация бота ${bot?.bp_name ?? ''}`}
                       error={loadingError}
                       onClose={onClose?.bind(this)}>
            {isLoading
                ? <Spin/>
                : <div className={style.bot_config_modal}>
                    <div className={style.bot_config_filter}>
                        <h3>Фильтры</h3>
                        {filtersInitialData?.map((filtersArrayItem, index) => {
                            const { config = {} } = filtersArrayItem;
                            const { type = '' } = config;

                            const filterSchema = filters?.[type]?.scheme || null;

                            return <div key={index} className={style.array_item}>
                                <FormConstructor schema={filterSchema ? filterSchema : filtersTypeSchema}
                                                 initialData={config}
                                                 onChange={this.onFilterFormChange.bind(this, index)}/>
                                <div className={style.delete_button_container}>
                                    <Button basic
                                            colorType={ButtonTypes.negative}
                                            onClick={this.removeFilterItem.bind(this, index)}>Удалить</Button>
                                </div>

                            </div>;
                        })}

                        <Button basic
                                colorType={ButtonTypes.positive}
                                onClick={this.addFilterItem.bind(this)}>
                            Добавить
                        </Button>
                    </div>
                    <div className={style.bot_config_items}>
                        <h3>Items</h3>
                        {itemsInitialData?.map((itemsArrayItem, index) => {
                            const { iterator_type } = itemsArrayItem;

                            const itemSchema = items?.[iterator_type]?.scheme
                                && Object.assign({}, itemsTypeSchema, items?.[iterator_type]?.scheme)
                                || null;

                            return <div key={index} className={style.array_item}>
                                <FormConstructor schema={itemSchema ? itemSchema : itemsTypeSchema}
                                                 initialData={itemsArrayItem}
                                                 onChange={this.onItemFormChange.bind(this, index)}/>
                                <div className={style.delete_button_container}>
                                    <Button basic
                                            colorType={ButtonTypes.negative}
                                            onClick={this.removeItem.bind(this, index)}>Удалить</Button>
                                </div>
                            </div>;
                        })}
                        <Button basic
                                colorType={ButtonTypes.positive}
                                onClick={this.addItem.bind(this)}>Добавить</Button>
                    </div>
                    <div className={style.bot_config_actions}>
                        <h3>Actions</h3>
                        <Button basic
                                colorType={ButtonTypes.positive}
                                onClick={this.addActionItem.bind(this, null)}>Добавить</Button>
                        {actionsInitialData?.map((actionArrayItem, index) => {
                            const { action_type = '' } = actionArrayItem;

                            const actionSchema = actions?.[action_type]?.scheme
                                && Object.assign({}, actionsTypeSchema, actions?.[action_type]?.scheme)
                                || null;

                            return <div key={index}>
                                <div className={style.array_item}>
                                    <FormConstructor schema={actionSchema ? actionSchema : actionsTypeSchema}
                                                     initialData={actionArrayItem}
                                                     onChange={this.onActionFormChange.bind(this, index)}/>
                                    <div className={style.delete_button_container}>
                                        <Button basic
                                                colorType={ButtonTypes.negative}
                                                onClick={this.removeActionItem.bind(this, index)}>Удалить</Button>
                                    </div>
                                </div>
                                <Button basic
                                        colorType={ButtonTypes.positive}
                                        onClick={this.addActionItem.bind(this, index)}>Добавить</Button>
                            </div>;
                        })}
                    </div>
                    <div className={`${style.controls} ${coreStyle.button_container}`}>
                        <CancelButton onClick={this.props.onClose?.bind(this)}/>
                        <SaveButton onClick={this.onSaveClick.bind(this)}/>
                    </div>
                </div>
            }
        </Window>;
    }
}
