import React from 'react';

import { Button, ButtonTypes, SaveButton } from '../../../../ui/Button';
import { Window } from '../../../../ui/FullModal';
import ProposeExistWarningModal from '../../../../ui/ProposeExistWarningModal/ProposeExistWarningModal';
import TextArea from '../../../../ui/TextArea';
import { Request2 } from '../../../../utils/request';
import { Copy } from '../../../Copy';
import { BOTS_REQUESTS, REQUESTS } from '../request';

interface IBotProposeButtonProps {
    schemaExist: boolean;
    bp_type: string;
    formData: Record<string, any>;
    onClose: (success: boolean) => void;
    checkIfPropositionExist: () => boolean;
    isNew: boolean;
}

interface IBotProposeButtonState {
    isLoading: boolean;
    error: null;
    isCommentModalOpen: boolean;
    isWarningModalOpen: boolean;
    comment: string;
    success: boolean;
}

export default class BotProposeButton extends React.Component<IBotProposeButtonProps, IBotProposeButtonState> {
    state: IBotProposeButtonState = {
        isLoading: false,
        error: null,
        isCommentModalOpen: false,
        isWarningModalOpen: false,
        comment: '',
        success: false,
    };

    request = new Request2({ requestConfigs: BOTS_REQUESTS });

    addComment() {
        if (!this.props.checkIfPropositionExist()) {
            this.setState({ isCommentModalOpen: true });
        } else {
            this.setState({ isWarningModalOpen: true });
        }
    }

    propose() {
        const { bp_type, formData, isNew } = this.props;
        const { comment } = this.state;

        if (isNew) {
            delete formData.bp_id;
            delete formData.bp_revision;
        }

        this.setState({ error: null, isLoading: true, isCommentModalOpen: false }, () => {
            this.request.exec(REQUESTS.PROPOSE_BOT,
                {
                    body: Object.assign({}, formData, {
                        bp_type: bp_type.toString(),
                    }),
                    queryParams: {
                        comment: encodeURIComponent(comment),
                    },
                })
                .then(() => {
                    this.setState({
                        isLoading: false,
                        success: true,
                    });
                })
                .catch(error => {
                    this.setState({ error, isLoading: false });
                });
        });
    }

    closeModals(error?: Error) {
        this.setState({ isCommentModalOpen: false }, () => {
            error
                ? this.setState({ error: null })
                : this.props.onClose(true);
        });
    }

    render() {
        const { schemaExist, bp_type, formData } = this.props;
        const { isCommentModalOpen, isLoading, success, error, comment, isWarningModalOpen } = this.state;
        const href = `#/settings/bots/proposition/${bp_type}/${formData.bp_name}`;

        return <>
            <Button disabled={!schemaExist}
                    colorType={ButtonTypes.positive}
                    basic={true}
                    isLoading={isLoading}
                    onClick={this.addComment.bind(this)}>
                Предложить
            </Button>

            {isCommentModalOpen
                ? <Window onClose={() => this.setState({ isCommentModalOpen: false })}
                          title={'Оставьте комментарий к предложению'}>
                    <TextArea value={''}
                              placeholder={'Комментарий - обязательное поле'}
                              onChange={(comment) => this.setState({ comment })}/>
                    <SaveButton onClick={this.propose.bind(this)} disabled={!comment}/>
                </Window>
                : null
            }

            {isWarningModalOpen
                ? <ProposeExistWarningModal onClose={() => this.setState({ isWarningModalOpen: false })}/>
                : null}

            {success || error
                ? <Window onClose={this.closeModals.bind(this, error)}
                          title={'Предложение бота'}
                          error={error}>
                    {success
                        ? <div>
                            <h3>Предложение успешно создано</h3>
                            <Copy text={`${location.origin}${location.pathname}${href}`}>
                                Копировать ссылку на предложение
                            </Copy>
                        </div>
                        : null
                    }
                </Window>
                : null
            }
        </>;
    }
}
